<?php
if (defined('CRASHIFY_FUNCTIONS_INCLUDED')) return;
define('CRASHIFY_FUNCTIONS_INCLUDED', true);

require_once __DIR__ . '/../config/config.php';

// includes/functions.php - Common Functions
// توابع مشترک و کاربردی برای سایت Crashify

require_once __DIR__ . '/../config/config.php';
require_once '../config/database.php';
require_once '../config/languages.php';

// Security Functions
function sanitizeInput($input) {
    if (is_array($input)) {
        return array_map('sanitizeInput', $input);
    }
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

function validatePassword($password) {
    // At least 8 characters, 1 uppercase, 1 lowercase, 1 number
    return preg_match('/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)[a-zA-Z\d@$!%*?&]{8,}$/', $password);
}

function generateToken($length = 32) {
    return bin2hex(random_bytes($length));
}

function hashPassword($password) {
    return password_hash($password, PASSWORD_ARGON2ID, [
        'memory_cost' => 65536,
        'time_cost' => 4,
        'threads' => 3
    ]);
}

function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

function generateJWT($payload) {
    $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
    $payload = json_encode($payload);
    
    $base64Header = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
    $base64Payload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));
    
    $signature = hash_hmac('sha256', $base64Header . "." . $base64Payload, JWT_SECRET, true);
    $base64Signature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
    
    return $base64Header . "." . $base64Payload . "." . $base64Signature;
}

function verifyJWT($jwt) {
    $parts = explode('.', $jwt);
    if (count($parts) !== 3) {
        return false;
    }
    
    [$header, $payload, $signature] = $parts;
    
    $validSignature = hash_hmac('sha256', $header . "." . $payload, JWT_SECRET, true);
    $validSignature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($validSignature));
    
    if (!hash_equals($signature, $validSignature)) {
        return false;
    }
    
    $payload = json_decode(base64_decode(str_replace(['-', '_'], ['+', '/'], $payload)), true);
    
    // Check expiration
    if (isset($payload['exp']) && $payload['exp'] < time()) {
        return false;
    }
    
    return $payload;
}

// User Functions
function isLoggedIn() {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

function getCurrentUser() {
    if (!isLoggedIn()) {
        return null;
    }
    
    global $db;
    return $db->fetch("SELECT * FROM users WHERE id = ? AND status = 'active'", [$_SESSION['user_id']]);
}

function getUserBalance($userId, $currency = 'USD') {
    global $db;
    $balance = $db->fetch("SELECT balance FROM user_balances WHERE user_id = ? AND currency = ?", [$userId, $currency]);
    return $balance ? floatval($balance['balance']) : 0.0;
}

function updateUserBalance($userId, $currency, $amount, $type = 'add') {
    global $db;
    
    $db->beginTransaction();
    try {
        $currentBalance = getUserBalance($userId, $currency);
        
        if ($type === 'subtract' && $currentBalance < $amount) {
            throw new Exception('Insufficient balance');
        }
        
        $newBalance = $type === 'add' ? $currentBalance + $amount : $currentBalance - $amount;
        
        $db->query("INSERT INTO user_balances (user_id, currency, balance) VALUES (?, ?, ?) 
                   ON DUPLICATE KEY UPDATE balance = ?", 
                   [$userId, $currency, $newBalance, $newBalance]);
        
        // Log transaction
        $db->insert('transactions', [
            'user_id' => $userId,
            'type' => $type,
            'amount' => $amount,
            'currency' => $currency,
            'balance_before' => $currentBalance,
            'balance_after' => $newBalance,
            'created_at' => date('Y-m-d H:i:s')
        ]);
        
        $db->commit();
        return true;
    } catch (Exception $e) {
        $db->rollback();
        return false;
    }
}

// Crypto Functions
function generateCryptoAddress($currency, $userId) {
    // This would integrate with actual crypto wallet APIs
    $addresses = [
        'BTC' => '1' . substr(hash('sha256', $userId . $currency . time()), 0, 33),
        'ETH' => '0x' . substr(hash('sha256', $userId . $currency . time()), 0, 40),
        'USDT' => '0x' . substr(hash('sha256', $userId . $currency . time()), 0, 40),
        'LTC' => 'L' . substr(hash('sha256', $userId . $currency . time()), 0, 33)
    ];
    
    return $addresses[$currency] ?? null;
}

function validateCryptoAddress($address, $currency) {
    $patterns = [
        'BTC' => '/^[13][a-km-zA-HJ-NP-Z1-9]{25,34}$/',
        'ETH' => '/^0x[a-fA-F0-9]{40}$/',
        'USDT' => '/^0x[a-fA-F0-9]{40}$/',
        'LTC' => '/^[LM3][a-km-zA-HJ-NP-Z1-9]{26,33}$/'
    ];
    
    return isset($patterns[$currency]) && preg_match($patterns[$currency], $address);
}

function getCryptoPrice($currency) {
    // This would integrate with actual crypto price APIs
    $prices = [
        'BTC' => 45000 + rand(-1000, 1000),
        'ETH' => 3000 + rand(-100, 100),
        'USDT' => 1.00,
        'LTC' => 150 + rand(-10, 10)
    ];
    
    return $prices[$currency] ?? 0;
}

// Betting Functions
function calculateOdds($probability) {
    if ($probability <= 0 || $probability >= 1) {
        return 1.01;
    }
    
    $fairOdds = 1 / $probability;
    $oddsWithMargin = $fairOdds * (1 + HOUSE_EDGE);
    
    return round($oddsWithMargin, 2);
}

function calculatePayout($stake, $odds) {
    return round($stake * $odds, 2);
}

function validateBet($userId, $amount, $odds) {
    if ($amount < MIN_BET_AMOUNT || $amount > MAX_BET_AMOUNT) {
        return ['valid' => false, 'message' => 'Invalid bet amount'];
    }
    
    if ($odds < 1.01 || $odds > MAX_ODDS) {
        return ['valid' => false, 'message' => 'Invalid odds'];
    }
    
    $payout = calculatePayout($amount, $odds);
    if ($payout > MAX_PAYOUT) {
        return ['valid' => false, 'message' => 'Payout exceeds maximum limit'];
    }
    
    $balance = getUserBalance($userId);
    if ($balance < $amount) {
        return ['valid' => false, 'message' => 'Insufficient balance'];
    }
    
    return ['valid' => true];
}

// Utility Functions
function formatCurrency($amount, $currency = 'USD', $decimals = 2) {
    global $lang;
    return $lang->formatCurrency($amount, $currency);
}

function formatDate($timestamp, $format = 'default') {
    global $lang;
    return $lang->formatDate($timestamp, $format);
}

function generateSlug($text) {
    $text = strtolower($text);
    $text = preg_replace('/[^a-z0-9\s-]/', '', $text);
    $text = preg_replace('/[\s-]+/', '-', $text);
    return trim($text, '-');
}

function uploadFile($file, $allowedTypes = null, $maxSize = null) {
    $allowedTypes = $allowedTypes ?: ALLOWED_IMAGE_TYPES;
    $maxSize = $maxSize ?: MAX_FILE_SIZE;
    
    if ($file['error'] !== UPLOAD_ERR_OK) {
        return ['success' => false, 'message' => 'Upload error'];
    }
    
    if ($file['size'] > $maxSize) {
        return ['success' => false, 'message' => 'File too large'];
    }
    
    $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($extension, $allowedTypes)) {
        return ['success' => false, 'message' => 'Invalid file type'];
    }
    
    $filename = uniqid() . '.' . $extension;
    $destination = UPLOAD_PATH . $filename;
    
    if (move_uploaded_file($file['tmp_name'], $destination)) {
        return ['success' => true, 'filename' => $filename];
    }
    
    return ['success' => false, 'message' => 'Upload failed'];
}

function sendEmail($to, $subject, $body, $isHTML = true) {
    // This would integrate with actual email service
    // For now, just log the email
    error_log("Email to {$to}: {$subject}");
    return true;
}

function logActivity($userId, $action, $details = []) {
    global $db;
    
    $db->insert('activity_logs', [
        'user_id' => $userId,
        'action' => $action,
        'details' => json_encode($details),
        'ip_address' => $_SERVER['REMOTE_ADDR'] ?? '',
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
        'created_at' => date('Y-m-d H:i:s')
    ]);
}

function getRealTimeData($type, $params = []) {
    // This would integrate with real-time data providers
    switch ($type) {
        case 'sports_odds':
            return getSportsOdds($params);
        case 'crypto_prices':
            return getCryptoPrices($params);
        case 'live_scores':
            return getLiveScores($params);
        default:
            return [];
    }
}

function getSportsOdds($params = []) {
    // Mock data for demonstration
    return [
        [
            'event_id' => 1,
            'home_team' => 'Manchester United',
            'away_team' => 'Liverpool',
            'home_odds' => 2.10,
            'draw_odds' => 3.40,
            'away_odds' => 3.20,
            'updated_at' => time()
        ]
    ];
}

function getCryptoPrices($currencies = ['BTC', 'ETH', 'USDT', 'LTC']) {
    $prices = [];
    foreach ($currencies as $currency) {
        $prices[$currency] = getCryptoPrice($currency);
    }
    return $prices;
}

function getLiveScores($params = []) {
    // Mock data for demonstration
    return [
        [
            'event_id' => 1,
            'home_team' => 'Manchester United',
            'away_team' => 'Liverpool',
            'home_score' => 1,
            'away_score' => 2,
            'status' => 'live',
            'minute' => 67
        ]
    ];
}

// Cache Functions
function cacheGet($key) {
    if (!CACHE_ENABLED) return null;
    
    $filename = '../cache/' . md5($key) . '.cache';
    if (!file_exists($filename)) return null;
    
    $data = unserialize(file_get_contents($filename));
    if ($data['expires'] < time()) {
        unlink($filename);
        return null;
    }
    
    return $data['value'];
}

function cacheSet($key, $value, $lifetime = null) {
    if (!CACHE_ENABLED) return false;
    
    $lifetime = $lifetime ?: CACHE_LIFETIME;
    $data = [
        'value' => $value,
        'expires' => time() + $lifetime
    ];
    
    $filename = '../cache/' . md5($key) . '.cache';
    return file_put_contents($filename, serialize($data)) !== false;
}

function cacheDelete($key) {
    $filename = '../cache/' . md5($key) . '.cache';
    if (file_exists($filename)) {
        return unlink($filename);
    }
    return true;
}

// Response Functions
function jsonResponse($data, $status = 200) {
    http_response_code($status);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

function redirectTo($url, $permanent = false) {
    $code = $permanent ? 301 : 302;
    http_response_code($code);
    header("Location: $url");
    exit;
}

function showError($message, $code = 400) {
    http_response_code($code);
    include '../pages/error.php';
    exit;
}
// includes/functions.php - Core Functions
// توابع اصلی سایت Crashify

// Security Functions
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function verifyCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

function sanitizeInput($input) {
    if (is_array($input)) {
        return array_map('sanitizeInput', $input);
    }
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

function validatePassword($password) {
    // At least 8 characters, 1 uppercase, 1 lowercase, 1 number
    return preg_match('/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)[a-zA-Z\d@$!%*?&]{8,}$/', $password);
}

function hashPassword($password) {
    return password_hash($password, PASSWORD_ARGON2ID, [
        'memory_cost' => 65536,
        'time_cost' => 4,
        'threads' => 3
    ]);
}

function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

function generateSecureToken($length = 32) {
    return bin2hex(random_bytes($length));
}

// User Functions
function getCurrentUser() {
    if (!isset($_SESSION['user_id'])) {
        return null;
    }
    
    $user = fetchOne(
        "SELECT u.*, 
                COUNT(b.id) as total_bets,
                COALESCE(SUM(CASE WHEN b.status = 'won' THEN 1 ELSE 0 END), 0) as total_wins,
                COALESCE(SUM(CASE WHEN b.status = 'lost' THEN 1 ELSE 0 END), 0) as total_losses,
                COALESCE(SUM(CASE WHEN b.status = 'won' THEN b.actual_payout ELSE 0 END), 0) as total_winnings,
                COALESCE(SUM(b.stake), 0) as total_wagered
         FROM users u 
         LEFT JOIN bets b ON u.id = b.user_id 
         WHERE u.id = ? AND u.status = 'active'
         GROUP BY u.id",
        [$_SESSION['user_id']]
    );
    
    if ($user) {
        // Calculate win rate
        $user['win_rate'] = $user['total_bets'] > 0 ? 
            ($user['total_wins'] / $user['total_bets']) * 100 : 0;
        
        // Update last activity
        updateRecord('users', 
            ['last_login' => date('Y-m-d H:i:s')], 
            'id = ?', 
            [$user['id']]
        );
    }
    
    return $user;
}

function loginUser($email, $password, $rememberMe = false) {
    $user = fetchOne(
        "SELECT * FROM users WHERE (email = ? OR username = ?) AND status = 'active'",
        [$email, $email]
    );
    
    if (!$user || !verifyPassword($password, $user['password_hash'])) {
        return ['success' => false, 'message' => 'Invalid credentials'];
    }
    
    // Check if account is locked
    if (isAccountLocked($user['id'])) {
        return ['success' => false, 'message' => 'Account temporarily locked'];
    }
    
    // Start session
    session_regenerate_id(true);
    $_SESSION['user_id'] = $user['id'];
    $_SESSION['username'] = $user['username'];
    $_SESSION['login_time'] = time();
    
    // Update user login info
    updateRecord('users', [
        'last_login' => date('Y-m-d H:i:s'),
        'last_ip' => $_SERVER['REMOTE_ADDR'] ?? ''
    ], 'id = ?', [$user['id']]);
    
    // Create session record
    $sessionId = session_id();
    insertRecord('user_sessions', [
        'id' => $sessionId,
        'user_id' => $user['id'],
        'ip_address' => $_SERVER['REMOTE_ADDR'] ?? '',
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? ''
    ]);
    
    // Set remember me cookie
    if ($rememberMe) {
        $token = generateSecureToken();
        setcookie('remember_token', $token, time() + (30 * 24 * 60 * 60), '/', '', true, true);
        // Store token in database (implement remember_tokens table)
    }
    
    // Clear failed login attempts
    clearFailedLoginAttempts($user['id']);
    
    return ['success' => true, 'user' => $user];
}

function registerUser($userData) {
    // Validate input
    $errors = [];
    
    if (empty($userData['username']) || strlen($userData['username']) < 3) {
        $errors[] = 'Username must be at least 3 characters';
    }
    
    if (!validateEmail($userData['email'])) {
        $errors[] = 'Invalid email address';
    }
    
    if (!validatePassword($userData['password'])) {
        $errors[] = 'Password must be at least 8 characters with uppercase, lowercase, and number';
    }
    
    if ($userData['password'] !== $userData['confirm_password']) {
        $errors[] = 'Passwords do not match';
    }
    
    if (!empty($errors)) {
        return ['success' => false, 'errors' => $errors];
    }
    
    // Check if user exists
    $existingUser = fetchOne(
        "SELECT id FROM users WHERE email = ? OR username = ?",
        [$userData['email'], $userData['username']]
    );
    
    if ($existingUser) {
        return ['success' => false, 'message' => 'User already exists'];
    }
    
    try {
        beginTransaction();
        
        // Create user
        $userId = insertRecord('users', [
            'username' => $userData['username'],
            'email' => $userData['email'],
            'password_hash' => hashPassword($userData['password']),
            'first_name' => $userData['first_name'] ?? '',
            'last_name' => $userData['last_name'] ?? '',
            'country' => $userData['country'] ?? '',
            'date_of_birth' => $userData['date_of_birth'] ?? null
        ]);
        
        // Send verification email
        $verificationToken = generateSecureToken();
        // Store verification token (implement email_verifications table)
        
        // Create welcome bonus
        createWelcomeBonus($userId);
        
        commit();
        
        return ['success' => true, 'user_id' => $userId];
        
    } catch (Exception $e) {
        rollback();
        error_log("Registration failed: " . $e->getMessage());
        return ['success' => false, 'message' => 'Registration failed'];
    }
}

function logoutUser() {
    if (isset($_SESSION['user_id'])) {
        // Remove session from database
        deleteRecord('user_sessions', 'id = ?', [session_id()]);
        
        // Clear remember me cookie
        if (isset($_COOKIE['remember_token'])) {
            setcookie('remember_token', '', time() - 3600, '/', '', true, true);
            // Remove token from database
        }
    }
    
    // Destroy session
    session_destroy();
    session_start();
    session_regenerate_id(true);
}

function isAccountLocked($userId) {
    // Check failed login attempts in last hour
    $attempts = fetchColumn(
        "SELECT COUNT(*) FROM login_attempts 
         WHERE user_id = ? AND created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)",
        [$userId]
    );
    
    return $attempts >= 5;
}

function recordFailedLoginAttempt($userId) {
    insertRecord('login_attempts', [
        'user_id' => $userId,
        'ip_address' => $_SERVER['REMOTE_ADDR'] ?? '',
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? ''
    ]);
}

function clearFailedLoginAttempts($userId) {
    deleteRecord('login_attempts', 'user_id = ?', [$userId]);
}

// Sports & Betting Functions
function getSportsData() {
    return fetchAll(
        "SELECT s.*, 
                COUNT(CASE WHEN e.status = 'live' THEN 1 END) as live_count,
                COUNT(e.id) as total_events
         FROM sports s 
         LEFT JOIN events e ON s.id = e.sport_id 
         WHERE s.active = 1 
         GROUP BY s.id 
         ORDER BY s.sort_order, s.name"
    );
}

function getFeaturedEvents($limit = 6) {
    return fetchAll(
        "SELECT e.*, 
                s.name as sport_name, s.icon as sport_icon,
                l.name as league_name, l.logo as league_logo,
                ht.name as home_team, ht.logo as home_team_logo,
                at.name as away_team, at.logo as away_team_logo,
                o1.odds as home_odds,
                o2.odds as draw_odds,
                o3.odds as away_odds
         FROM events e
         JOIN sports s ON e.sport_id = s.id
         JOIN leagues l ON e.league_id = l.id
         JOIN teams ht ON e.home_team_id = ht.id
         JOIN teams at ON e.away_team_id = at.id
         LEFT JOIN markets m ON e.id = m.event_id AND m.type = 'match_result'
         LEFT JOIN odds o1 ON m.id = o1.market_id AND o1.selection = 'home'
         LEFT JOIN odds o2 ON m.id = o2.market_id AND o2.selection = 'draw'
         LEFT JOIN odds o3 ON m.id = o3.market_id AND o3.selection = 'away'
         WHERE e.featured = 1 AND e.status IN ('scheduled', 'live')
         ORDER BY e.start_time ASC
         LIMIT ?",
        [$limit]
    );
}

function getLiveEvents($limit = 10) {
    return fetchAll(
        "SELECT e.*, 
                s.name as sport_name, s.icon as sport_icon,
                l.name as league_name, l.logo as league_logo,
                ht.name as home_team, ht.logo as home_team_logo,
                at.name as away_team, at.logo as away_team_logo,
                o1.odds as home_odds,
                o2.odds as draw_odds,
                o3.odds as away_odds
         FROM events e
         JOIN sports s ON e.sport_id = s.id
         JOIN leagues l ON e.league_id = l.id
         JOIN teams ht ON e.home_team_id = ht.id
         JOIN teams at ON e.away_team_id = at.id
         LEFT JOIN markets m ON e.id = m.event_id AND m.type = 'match_result'
         LEFT JOIN odds o1 ON m.id = o1.market_id AND o1.selection = 'home'
         LEFT JOIN odds o2 ON m.id = o2.market_id AND o2.selection = 'draw'
         LEFT JOIN odds o3 ON m.id = o3.market_id AND o3.selection = 'away'
         WHERE e.status = 'live'
         ORDER BY e.start_time ASC
         LIMIT ?",
        [$limit]
    );
}

function getUpcomingEvents($sport = 'all', $league = 'all', $timeFilter = 'today', $search = '', $limit = 50) {
    $conditions = ["e.status = 'scheduled'"];
    $params = [];
    
    if ($sport !== 'all') {
        $conditions[] = "s.slug = ?";
        $params[] = $sport;
    }
    
    if ($league !== 'all') {
        $conditions[] = "l.slug = ?";
        $params[] = $league;
    }
    
    switch ($timeFilter) {
        case 'today':
            $conditions[] = "DATE(e.start_time) = CURDATE()";
            break;
        case 'tomorrow':
            $conditions[] = "DATE(e.start_time) = DATE_ADD(CURDATE(), INTERVAL 1 DAY)";
            break;
        case 'week':
            $conditions[] = "e.start_time BETWEEN NOW() AND DATE_ADD(NOW(), INTERVAL 7 DAY)";
            break;
    }
    
    if (!empty($search)) {
        $conditions[] = "(ht.name LIKE ? OR at.name LIKE ? OR l.name LIKE ?)";
        $searchTerm = "%{$search}%";
        $params[] = $searchTerm;
        $params[] = $searchTerm;
        $params[] = $searchTerm;
    }
    
    $whereClause = implode(' AND ', $conditions);
    $params[] = $limit;
    
    return fetchAll(
        "SELECT e.*, 
                s.name as sport_name, s.icon as sport_icon,
                l.name as league_name, l.logo as league_logo,
                ht.name as home_team, ht.logo as home_team_logo,
                at.name as away_team, at.logo as away_team_logo,
                o1.odds as home_odds,
                o2.odds as draw_odds,
                o3.odds as away_odds
         FROM events e
         JOIN sports s ON e.sport_id = s.id
         JOIN leagues l ON e.league_id = l.id
         JOIN teams ht ON e.home_team_id = ht.id
         JOIN teams at ON e.away_team_id = at.id
         LEFT JOIN markets m ON e.id = m.event_id AND m.type = 'match_result'
         LEFT JOIN odds o1 ON m.id = o1.market_id AND o1.selection = 'home'
         LEFT JOIN odds o2 ON m.id = o2.market_id AND o2.selection = 'draw'
         LEFT JOIN odds o3 ON m.id = o3.market_id AND o3.selection = 'away'
         WHERE {$whereClause}
         ORDER BY e.start_time ASC
         LIMIT ?",
        $params
    );
}

function placeBet($userId, $betData) {
    try {
        beginTransaction();
        
        // Validate bet data
        $event = fetchOne("SELECT * FROM events WHERE id = ? AND status IN ('scheduled', 'live')", [$betData['event_id']]);
        if (!$event) {
            throw new Exception("Event not found or not available for betting");
        }
        
        $odds = fetchOne("SELECT * FROM odds WHERE id = ? AND active = 1", [$betData['odds_id']]);
        if (!$odds) {
            throw new Exception("Odds not found or not active");
        }
        
        // Validate stake
        $stake = floatval($betData['stake']);
        $minBet = getSetting('min_bet_amount', 0.01);
        $maxBet = getSetting('max_bet_amount', 10000);
        
        if ($stake < $minBet || $stake > $maxBet) {
            throw new Exception("Stake must be between {$minBet} and {$maxBet}");
        }
        
        // Check user balance
        $user = fetchOne("SELECT balance FROM users WHERE id = ?", [$userId]);
        if ($user['balance'] < $stake) {
            throw new Exception("Insufficient balance");
        }
        
        // Calculate potential payout
        $potentialPayout = $stake * $odds['odds'];
        $maxPayout = getSetting('max_payout', 100000);
        
        if ($potentialPayout > $maxPayout) {
            throw new Exception("Potential payout exceeds maximum limit");
        }
        
        // Deduct stake from user balance
        updateRecord('users', 
            ['balance' => $user['balance'] - $stake], 
            'id = ?', 
            [$userId]
        );
        
        // Create bet record
        $betId = insertRecord('bets', [
            'user_id' => $userId,
            'event_id' => $betData['event_id'],
            'market_id' => $odds['market_id'],
            'odds_id' => $betData['odds_id'],
            'selection' => $betData['selection'],
            'odds' => $odds['odds'],
            'stake' => $stake,
            'potential_payout' => $potentialPayout
        ]);
        
        // Create transaction record
        insertRecord('transactions', [
            'user_id' => $userId,
            'type' => 'bet',
            'amount' => -$stake,
            'status' => 'completed',
            'reference_id' => $betId,
            'reference_type' => 'bet'
        ]);
        
        // Create notification
        createNotification($userId, 'bet_placed', 'Bet Placed', 
            "Your bet of {$stake} has been placed successfully");
        
        commit();
        
        return ['success' => true, 'bet_id' => $betId];
        
    } catch (Exception $e) {
        rollback();
        error_log("Bet placement failed: " . $e->getMessage());
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Wallet Functions
function getUserBalance($userId) {
    return fetchOne(
        "SELECT balance, bonus_balance FROM users WHERE id = ?",
        [$userId]
    );
}

function createDeposit($userId, $amount, $currency, $walletAddress) {
    try {
        $transactionId = insertRecord('transactions', [
            'user_id' => $userId,
            'type' => 'deposit',
            'amount' => $amount,
            'currency' => $currency,
            'wallet_address' => $walletAddress,
            'status' => 'pending'
        ]);
        
        return ['success' => true, 'transaction_id' => $transactionId];
        
    } catch (Exception $e) {
        error_log("Deposit creation failed: " . $e->getMessage());
        return ['success' => false, 'message' => 'Deposit creation failed'];
    }
}

function processWithdrawal($userId, $amount, $walletAddress) {
    try {
        beginTransaction();
        
        // Check balance
        $user = fetchOne("SELECT balance FROM users WHERE id = ?", [$userId]);
        if ($user['balance'] < $amount) {
            throw new Exception("Insufficient balance");
        }
        
        // Deduct from balance
        updateRecord('users', 
            ['balance' => $user['balance'] - $amount], 
            'id = ?', 
            [$userId]
        );
        
        // Create withdrawal record
        $transactionId = insertRecord('transactions', [
            'user_id' => $userId,
            'type' => 'withdrawal',
            'amount' => -$amount,
            'wallet_address' => $walletAddress,
            'status' => 'pending'
        ]);
        
        commit();
        
        return ['success' => true, 'transaction_id' => $transactionId];
        
    } catch (Exception $e) {
        rollback();
        error_log("Withdrawal failed: " . $e->getMessage());
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Bonus Functions
function createWelcomeBonus($userId) {
    $welcomeBonus = fetchOne(
        "SELECT * FROM bonuses WHERE type = 'welcome' AND active = 1 LIMIT 1"
    );
    
    if ($welcomeBonus) {
        $amount = $welcomeBonus['amount'];
        $wageringRequirement = $amount * $welcomeBonus['wagering_requirement'];
        $expiresAt = date('Y-m-d H:i:s', strtotime("+{$welcomeBonus['valid_days']} days"));
        
        insertRecord('user_bonuses', [
            'user_id' => $userId,
            'bonus_id' => $welcomeBonus['id'],
            'amount' => $amount,
            'wagering_requirement' => $wageringRequirement,
            'expires_at' => $expiresAt
        ]);
        
        // Add to user bonus balance
        updateRecord('users', 
            ['bonus_balance' => $amount], 
            'id = ?', 
            [$userId]
        );
    }
}

// Notification Functions
function createNotification($userId, $type, $title, $message, $data = null) {
    return insertRecord('notifications', [
        'user_id' => $userId,
        'type' => $type,
        'title' => $title,
        'message' => $message,
        'data' => $data ? json_encode($data) : null
    ]);
}

function getUserNotifications($userId, $limit = 20) {
    return fetchAll(
        "SELECT * FROM notifications 
         WHERE user_id = ? 
         ORDER BY created_at DESC 
         LIMIT ?",
        [$userId, $limit]
    );
}

// Utility Functions
function formatCurrency($amount, $currency = 'USD', $decimals = 2) {
    return number_format($amount, $decimals) . ' ' . $currency;
}

function formatEventDate($datetime) {
    return date('M j, Y H:i', strtotime($datetime));
}

function getSetting($key, $default = null) {
    $setting = fetchOne("SELECT value, type FROM settings WHERE key_name = ?", [$key]);
    
    if (!$setting) {
        return $default;
    }
    
    switch ($setting['type']) {
        case 'boolean':
            return (bool) $setting['value'];
        case 'integer':
            return (int) $setting['value'];
        case 'float':
            return (float) $setting['value'];
        case 'json':
            return json_decode($setting['value'], true);
        default:
            return $setting['value'];
    }
}

function setSetting($key, $value, $type = 'string') {
    if ($type === 'json') {
        $value = json_encode($value);
    }
    
    return updateRecord('settings', 
        ['value' => $value, 'type' => $type], 
        'key_name = ?', 
        [$key]
    );
}

function isCurrentPage($path) {
    return $_SERVER['REQUEST_URI'] === $path || 
           (strpos($_SERVER['REQUEST_URI'], '?') !== false && 
            strpos($_SERVER['REQUEST_URI'], $path) === 0);
}

function redirect($url, $statusCode = 302) {
    header("Location: {$url}", true, $statusCode);
    exit;
}

function getClientIP() {
    $ipKeys = ['HTTP_CF_CONNECTING_IP', 'HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR'];
    
    foreach ($ipKeys as $key) {
        if (array_key_exists($key, $_SERVER) === true) {
            foreach (explode(',', $_SERVER[$key]) as $ip) {
                $ip = trim($ip);
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                    return $ip;
                }
            }
        }
    }
    
    return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
}

function logActivity($userId, $action, $details = null) {
    insertRecord('activity_logs', [
        'user_id' => $userId,
        'action' => $action,
        'details' => $details ? json_encode($details) : null,
        'ip_address' => getClientIP(),
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? ''
    ]);
}

function rateLimitCheck($key, $maxAttempts = 10, $timeWindow = 3600) {
    $cacheKey = "rate_limit:{$key}";
    $attempts = apcu_fetch($cacheKey) ?: 0;
    
    if ($attempts >= $maxAttempts) {
        return false;
    }
    
    apcu_store($cacheKey, $attempts + 1, $timeWindow);
    return true;
 }
?>