<?php
// config/database.php - Database Configuration and Connection
// تنظیمات و اتصال پایگاه داده

class Database {
    private static $instance = null;
    private $connection;
    private $host;
    private $username;
    private $password;
    private $database;
    private $charset;
    private $options;

    private function __construct() {
        $this->host = defined('DB_HOST') ? DB_HOST : (getenv('DB_HOST') ?: '127.0.0.1');
        $this->username = defined('DB_USER') ? DB_USER : (getenv('DB_USER') ?: 'root');
        $this->password = defined('DB_PASS') ? DB_PASS : (getenv('DB_PASS') ?: '');
        $this->database = defined('DB_NAME') ? DB_NAME : (getenv('DB_NAME') ?: 'crashify');
        $this->charset = defined('DB_CHARSET') ? DB_CHARSET : (getenv('DB_CHARSET') ?: 'utf8mb4');
        
        // PDO options
        $this->options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
            PDO::ATTR_PERSISTENT => true,
            PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES {$this->charset} COLLATE utf8mb4_unicode_ci"
        ];

        $this->connect();
    }

    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function connect() {
        try {
            $dsn = "mysql:host={$this->host};dbname={$this->database};charset={$this->charset}";
            $this->connection = new PDO($dsn, $this->username, $this->password, $this->options);
            
            // Set timezone
            $this->connection->exec("SET time_zone = '+00:00'");
            
        } catch (PDOException $e) {
            error_log("Database connection failed: " . $e->getMessage());
            throw new Exception("Database connection failed");
        }
    }

    public function getConnection() {
        // Check if connection is still alive
        if ($this->connection === null) {
            $this->connect();
        }
        
        try {
            $this->connection->query('SELECT 1');
        } catch (PDOException $e) {
            $this->connect();
        }
        
        return $this->connection;
    }

    // Prevent cloning
    private function __clone() {}

    // Prevent unserialization
    public function __wakeup() {
        throw new Exception("Cannot unserialize singleton");
    }
}

// Database helper functions
function getDB() {
    return Database::getInstance()->getConnection();
}

function executeQuery($sql, $params = []) {
    try {
        $db = getDB();
        $stmt = $db->prepare($sql);
        $stmt->execute($params);
        return $stmt;
    } catch (PDOException $e) {
        error_log("Query execution failed: " . $e->getMessage());
        error_log("SQL: " . $sql);
        error_log("Params: " . json_encode($params));
        throw new Exception("Database query failed");
    }
}

function fetchOne($sql, $params = []) {
    $stmt = executeQuery($sql, $params);
    return $stmt->fetch();
}

function fetchAll($sql, $params = []) {
    $stmt = executeQuery($sql, $params);
    return $stmt->fetchAll();
}

function fetchColumn($sql, $params = []) {
    $stmt = executeQuery($sql, $params);
    return $stmt->fetchColumn();
}

function insertRecord($table, $data) {
    $columns = array_keys($data);
    $placeholders = ':' . implode(', :', $columns);
    $columnList = implode(', ', $columns);
    
    $sql = "INSERT INTO {$table} ({$columnList}) VALUES ({$placeholders})";
    
    executeQuery($sql, $data);
    return getDB()->lastInsertId();
}

function updateRecord($table, $data, $where, $whereParams = []) {
    $setParts = [];
    foreach (array_keys($data) as $column) {
        $setParts[] = "{$column} = :{$column}";
    }
    $setClause = implode(', ', $setParts);
    
    $sql = "UPDATE {$table} SET {$setClause} WHERE {$where}";
    
    $params = array_merge($data, $whereParams);
    $stmt = executeQuery($sql, $params);
    
    return $stmt->rowCount();
}

function deleteRecord($table, $where, $params = []) {
    $sql = "DELETE FROM {$table} WHERE {$where}";
    $stmt = executeQuery($sql, $params);
    return $stmt->rowCount();
}

// Transaction helpers
function beginTransaction() {
    return getDB()->beginTransaction();
}

function commit() {
    return getDB()->commit();
}

function rollback() {
    return getDB()->rollBack();
}

function inTransaction() {
    return getDB()->inTransaction();
}

// Database schema creation and migration functions
function createTables() {
    $db = getDB();
    
    // Users table
    $db->exec("
        CREATE TABLE IF NOT EXISTS users (
            id INT PRIMARY KEY AUTO_INCREMENT,
            username VARCHAR(50) UNIQUE NOT NULL,
            email VARCHAR(255) UNIQUE NOT NULL,
            password_hash VARCHAR(255) NOT NULL,
            first_name VARCHAR(100),
            last_name VARCHAR(100),
            phone VARCHAR(20),
            country VARCHAR(2),
            date_of_birth DATE,
            avatar VARCHAR(255),
            balance DECIMAL(20,8) DEFAULT 0.00000000,
            bonus_balance DECIMAL(20,8) DEFAULT 0.00000000,
            level INT DEFAULT 1,
            vip_status ENUM('none', 'bronze', 'silver', 'gold', 'platinum', 'diamond') DEFAULT 'none',
            total_bets INT DEFAULT 0,
            total_wins INT DEFAULT 0,
            total_losses INT DEFAULT 0,
            win_rate DECIMAL(5,2) DEFAULT 0.00,
            total_wagered DECIMAL(20,8) DEFAULT 0.00000000,
            total_winnings DECIMAL(20,8) DEFAULT 0.00000000,
            loyalty_points INT DEFAULT 0,
            email_verified BOOLEAN DEFAULT FALSE,
            phone_verified BOOLEAN DEFAULT FALSE,
            identity_verified BOOLEAN DEFAULT FALSE,
            two_factor_enabled BOOLEAN DEFAULT FALSE,
            two_factor_secret VARCHAR(32),
            status ENUM('active', 'suspended', 'banned') DEFAULT 'active',
            last_login TIMESTAMP NULL,
            last_ip VARCHAR(45),
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            
            INDEX idx_username (username),
            INDEX idx_email (email),
            INDEX idx_status (status),
            INDEX idx_created_at (created_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");

    // User sessions table
    $db->exec("
        CREATE TABLE IF NOT EXISTS user_sessions (
            id VARCHAR(128) PRIMARY KEY,
            user_id INT NOT NULL,
            ip_address VARCHAR(45),
            user_agent TEXT,
            last_activity TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            INDEX idx_user_id (user_id),
            INDEX idx_last_activity (last_activity)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");

    // Sports table
    $db->exec("
        CREATE TABLE IF NOT EXISTS sports (
            id INT PRIMARY KEY AUTO_INCREMENT,
            name VARCHAR(100) NOT NULL,
            slug VARCHAR(50) UNIQUE NOT NULL,
            icon VARCHAR(50),
            category ENUM('traditional', 'esports', 'virtual') DEFAULT 'traditional',
            active BOOLEAN DEFAULT TRUE,
            sort_order INT DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            
            INDEX idx_slug (slug),
            INDEX idx_category (category),
            INDEX idx_active (active)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");

    // Leagues table
    $db->exec("
        CREATE TABLE IF NOT EXISTS leagues (
            id INT PRIMARY KEY AUTO_INCREMENT,
            sport_id INT NOT NULL,
            name VARCHAR(200) NOT NULL,
            slug VARCHAR(100) UNIQUE NOT NULL,
            country VARCHAR(2),
            logo VARCHAR(255),
            active BOOLEAN DEFAULT TRUE,
            sort_order INT DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            
            FOREIGN KEY (sport_id) REFERENCES sports(id) ON DELETE CASCADE,
            INDEX idx_sport_id (sport_id),
            INDEX idx_slug (slug),
            INDEX idx_active (active)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");

    // Teams table
    $db->exec("
        CREATE TABLE IF NOT EXISTS teams (
            id INT PRIMARY KEY AUTO_INCREMENT,
            name VARCHAR(200) NOT NULL,
            slug VARCHAR(100) UNIQUE NOT NULL,
            short_name VARCHAR(10),
            logo VARCHAR(255),
            country VARCHAR(2),
            sport_id INT NOT NULL,
            active BOOLEAN DEFAULT TRUE,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            
            FOREIGN KEY (sport_id) REFERENCES sports(id) ON DELETE CASCADE,
            INDEX idx_sport_id (sport_id),
            INDEX idx_slug (slug),
            INDEX idx_active (active)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");

    // Events table
    $db->exec("
        CREATE TABLE IF NOT EXISTS events (
            id INT PRIMARY KEY AUTO_INCREMENT,
            sport_id INT NOT NULL,
            league_id INT NOT NULL,
            home_team_id INT NOT NULL,
            away_team_id INT NOT NULL,
            name VARCHAR(500) NOT NULL,
            start_time TIMESTAMP NOT NULL,
            status ENUM('scheduled', 'live', 'finished', 'cancelled', 'postponed') DEFAULT 'scheduled',
            home_score INT DEFAULT 0,
            away_score INT DEFAULT 0,
            match_time VARCHAR(10),
            featured BOOLEAN DEFAULT FALSE,
            live_streaming BOOLEAN DEFAULT FALSE,
            total_markets INT DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            
            FOREIGN KEY (sport_id) REFERENCES sports(id) ON DELETE CASCADE,
            FOREIGN KEY (league_id) REFERENCES leagues(id) ON DELETE CASCADE,
            FOREIGN KEY (home_team_id) REFERENCES teams(id) ON DELETE CASCADE,
            FOREIGN KEY (away_team_id) REFERENCES teams(id) ON DELETE CASCADE,
            INDEX idx_sport_id (sport_id),
            INDEX idx_league_id (league_id),
            INDEX idx_start_time (start_time),
            INDEX idx_status (status),
            INDEX idx_featured (featured)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");

    // Markets table
    $db->exec("
        CREATE TABLE IF NOT EXISTS markets (
            id INT PRIMARY KEY AUTO_INCREMENT,
            event_id INT NOT NULL,
            name VARCHAR(200) NOT NULL,
            type VARCHAR(50) NOT NULL,
            active BOOLEAN DEFAULT TRUE,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            
            FOREIGN KEY (event_id) REFERENCES events(id) ON DELETE CASCADE,
            INDEX idx_event_id (event_id),
            INDEX idx_type (type),
            INDEX idx_active (active)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");

    // Odds table
    $db->exec("
        CREATE TABLE IF NOT EXISTS odds (
            id INT PRIMARY KEY AUTO_INCREMENT,
            market_id INT NOT NULL,
            selection VARCHAR(100) NOT NULL,
            odds DECIMAL(10,2) NOT NULL,
            active BOOLEAN DEFAULT TRUE,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            
            FOREIGN KEY (market_id) REFERENCES markets(id) ON DELETE CASCADE,
            INDEX idx_market_id (market_id),
            INDEX idx_active (active),
            INDEX idx_updated_at (updated_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");

    // Bets table
    $db->exec("
        CREATE TABLE IF NOT EXISTS bets (
            id INT PRIMARY KEY AUTO_INCREMENT,
            user_id INT NOT NULL,
            event_id INT NOT NULL,
            market_id INT NOT NULL,
            odds_id INT NOT NULL,
            bet_type ENUM('single', 'accumulator', 'system') DEFAULT 'single',
            selection VARCHAR(100) NOT NULL,
            odds DECIMAL(10,2) NOT NULL,
            stake DECIMAL(20,8) NOT NULL,
            potential_payout DECIMAL(20,8) NOT NULL,
            actual_payout DECIMAL(20,8) DEFAULT 0.00000000,
            status ENUM('pending', 'won', 'lost', 'void', 'cancelled') DEFAULT 'pending',
            placed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            settled_at TIMESTAMP NULL,
            
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            FOREIGN KEY (event_id) REFERENCES events(id) ON DELETE CASCADE,
            FOREIGN KEY (market_id) REFERENCES markets(id) ON DELETE CASCADE,
            FOREIGN KEY (odds_id) REFERENCES odds(id) ON DELETE CASCADE,
            INDEX idx_user_id (user_id),
            INDEX idx_event_id (event_id),
            INDEX idx_status (status),
            INDEX idx_placed_at (placed_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");

    // Transactions table
    $db->exec("
        CREATE TABLE IF NOT EXISTS transactions (
            id INT PRIMARY KEY AUTO_INCREMENT,
            user_id INT NOT NULL,
            type ENUM('deposit', 'withdrawal', 'bet', 'win', 'bonus', 'refund') NOT NULL,
            amount DECIMAL(20,8) NOT NULL,
            currency VARCHAR(10) DEFAULT 'USD',
            status ENUM('pending', 'completed', 'failed', 'cancelled') DEFAULT 'pending',
            transaction_hash VARCHAR(255),
            wallet_address VARCHAR(255),
            network VARCHAR(50),
            confirmations INT DEFAULT 0,
            required_confirmations INT DEFAULT 1,
            fee DECIMAL(20,8) DEFAULT 0.00000000,
            reference_id INT,
            reference_type VARCHAR(50),
            notes TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            INDEX idx_user_id (user_id),
            INDEX idx_type (type),
            INDEX idx_status (status),
            INDEX idx_transaction_hash (transaction_hash),
            INDEX idx_created_at (created_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");

    // Bonuses table
    $db->exec("
        CREATE TABLE IF NOT EXISTS bonuses (
            id INT PRIMARY KEY AUTO_INCREMENT,
            name VARCHAR(200) NOT NULL,
            type ENUM('welcome', 'deposit', 'reload', 'no_deposit', 'free_bet', 'cashback', 'loyalty') NOT NULL,
            amount DECIMAL(20,8),
            percentage DECIMAL(5,2),
            max_amount DECIMAL(20,8),
            min_deposit DECIMAL(20,8),
            wagering_requirement INT DEFAULT 1,
            valid_days INT DEFAULT 30,
            code VARCHAR(50),
            active BOOLEAN DEFAULT TRUE,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            expires_at TIMESTAMP NULL,
            
            INDEX idx_type (type),
            INDEX idx_code (code),
            INDEX idx_active (active)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");

    // User bonuses table
    $db->exec("
        CREATE TABLE IF NOT EXISTS user_bonuses (
            id INT PRIMARY KEY AUTO_INCREMENT,
            user_id INT NOT NULL,
            bonus_id INT NOT NULL,
            amount DECIMAL(20,8) NOT NULL,
            wagering_requirement DECIMAL(20,8) NOT NULL,
            wagered_amount DECIMAL(20,8) DEFAULT 0.00000000,
            status ENUM('active', 'completed', 'expired', 'cancelled') DEFAULT 'active',
            claimed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            expires_at TIMESTAMP NOT NULL,
            completed_at TIMESTAMP NULL,
            
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            FOREIGN KEY (bonus_id) REFERENCES bonuses(id) ON DELETE CASCADE,
            INDEX idx_user_id (user_id),
            INDEX idx_status (status),
            INDEX idx_expires_at (expires_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");

    // Notifications table
    $db->exec("
        CREATE TABLE IF NOT EXISTS notifications (
            id INT PRIMARY KEY AUTO_INCREMENT,
            user_id INT NOT NULL,
            type VARCHAR(50) NOT NULL,
            title VARCHAR(200) NOT NULL,
            message TEXT NOT NULL,
            data JSON,
            read_at TIMESTAMP NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            INDEX idx_user_id (user_id),
            INDEX idx_type (type),
            INDEX idx_read_at (read_at),
            INDEX idx_created_at (created_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");

    // Settings table
    $db->exec("
        CREATE TABLE IF NOT EXISTS settings (
            id INT PRIMARY KEY AUTO_INCREMENT,
            key_name VARCHAR(100) UNIQUE NOT NULL,
            value TEXT,
            type ENUM('string', 'integer', 'float', 'boolean', 'json') DEFAULT 'string',
            description TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            
            INDEX idx_key_name (key_name)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");

    // Insert default settings
    $defaultSettings = [
        ['site_name', 'Crashify', 'string', 'Site name'],
        ['site_description', 'Premium Crypto Betting Platform', 'string', 'Site description'],
        ['maintenance_mode', '0', 'boolean', 'Maintenance mode'],
        ['registration_enabled', '1', 'boolean', 'User registration enabled'],
        ['min_bet_amount', '0.01', 'float', 'Minimum bet amount'],
        ['max_bet_amount', '10000', 'float', 'Maximum bet amount'],
        ['max_payout', '100000', 'float', 'Maximum payout'],
        ['house_edge', '2.5', 'float', 'House edge percentage'],
        ['supported_currencies', '["BTC", "ETH", "USDT", "BNB"]', 'json', 'Supported cryptocurrencies']
    ];

    foreach ($defaultSettings as $setting) {
        $db->prepare("INSERT IGNORE INTO settings (key_name, value, type, description) VALUES (?, ?, ?, ?)")
           ->execute($setting);
    }

    return true;
}

// Initialize database
try {
    createTables();
} catch (Exception $e) {
    error_log("Database initialization failed: " . $e->getMessage());
}
?>
