// assets/js/wallet.js - Wallet Management JavaScript (Continued)
// جاوااسکریپت مدیریت کیف پول (ادامه)

            `;
        }
        
        container.innerHTML = html;
        container.style.display = html ? 'block' : 'none';
    }
    
    // Show payment modal
    showPaymentModal(paymentData) {
        const modal = document.createElement('div');
        modal.className = 'payment-modal';
        modal.innerHTML = `
            <div class="modal-content">
                <div class="modal-header">
                    <h5>تکمیل پرداخت</h5>
                    <button class="close-modal">&times;</button>
                </div>
                <div class="modal-body">
                    <div class="payment-info">
                        <div class="payment-amount">
                            <h3>$${paymentData.amount}</h3>
                            <p>مبلغ قابل پرداخت</p>
                        </div>
                        <div class="payment-method">
                            <p><strong>روش پرداخت:</strong> ${paymentData.payment_method}</p>
                        </div>
                        <div class="payment-instructions">
                            ${paymentData.instructions || ''}
                        </div>
                    </div>
                    <div class="payment-actions">
                        <a href="${paymentData.payment_url}" target="_blank" class="btn btn-primary">
                            ادامه پرداخت
                        </a>
                        <button class="btn btn-secondary close-modal">انصراف</button>
                    </div>
                </div>
            </div>
        `;
        
        document.body.appendChild(modal);
        
        // Add close functionality
        modal.querySelectorAll('.close-modal').forEach(btn => {
            btn.addEventListener('click', () => {
                document.body.removeChild(modal);
            });
        });
        
        // Show modal
        setTimeout(() => modal.classList.add('show'), 10);
    }
    
    // Show withdrawal success
    showWithdrawalSuccess(data) {
        const modal = document.createElement('div');
        modal.className = 'success-modal';
        modal.innerHTML = `
            <div class="modal-content">
                <div class="modal-header">
                    <h5><i class="fas fa-check-circle text-success"></i> درخواست برداشت ثبت شد</h5>
                </div>
                <div class="modal-body">
                    <div class="success-info">
                        <p><strong>مبلغ:</strong> $${data.amount}</p>
                        <p><strong>شماره تراکنش:</strong> ${data.transaction_id}</p>
                        <p><strong>زمان پردازش:</strong> ${data.processing_time}</p>
                        <div class="alert alert-info mt-3">
                            <i class="fas fa-info-circle"></i>
                            درخواست شما در حال بررسی است و پس از تأیید پردازش خواهد شد.
                        </div>
                    </div>
                    <button class="btn btn-primary close-modal">متوجه شدم</button>
                </div>
            </div>
        `;
        
        document.body.appendChild(modal);
        
        // Add close functionality
        modal.querySelector('.close-modal').addEventListener('click', () => {
            document.body.removeChild(modal);
        });
        
        // Show modal
        setTimeout(() => modal.classList.add('show'), 10);
    }
    
    // Render payment methods
    renderPaymentMethods() {
        // Render deposit methods
        const depositSelect = document.querySelector('.deposit-payment-methods');
        if (depositSelect && this.paymentMethods.deposit) {
            let html = '<option value="">انتخاب روش پرداخت</option>';
            this.paymentMethods.deposit.forEach(method => {
                html += `<option value="${method.id}">${method.name}</option>`;
            });
            depositSelect.innerHTML = html;
        }
        
        // Render withdrawal methods
        const withdrawalSelect = document.querySelector('.withdrawal-payment-methods');
        if (withdrawalSelect && this.paymentMethods.withdrawal) {
            let html = '<option value="">انتخاب روش برداشت</option>';
            this.paymentMethods.withdrawal.forEach(method => {
                html += `<option value="${method.id}">${method.name}</option>`;
            });
            withdrawalSelect.innerHTML = html;
        }
    }
    
    // Update limits display
    updateLimitsDisplay() {
        // Update deposit limits
        if (this.limits.deposit) {
            const depositLimits = document.querySelector('.deposit-limits');
            if (depositLimits) {
                depositLimits.innerHTML = `
                    <div class="limit-item">
                        <span>حداقل واریز:</span>
                        <span>$${this.limits.deposit.min_amount}</span>
                    </div>
                    <div class="limit-item">
                        <span>حداکثر واریز:</span>
                        <span>$${this.limits.deposit.max_amount}</span>
                    </div>
                    <div class="limit-item">
                        <span>حد روزانه باقی‌مانده:</span>
                        <span>$${this.limits.deposit.daily_remaining}</span>
                    </div>
                `;
            }
        }
        
        // Update withdrawal limits
        if (this.limits.withdrawal) {
            const withdrawalLimits = document.querySelector('.withdrawal-limits');
            if (withdrawalLimits) {
                withdrawalLimits.innerHTML = `
                    <div class="limit-item">
                        <span>حداقل برداشت:</span>
                        <span>$${this.limits.withdrawal.min_amount}</span>
                    </div>
                    <div class="limit-item">
                        <span>حداکثر برداشت:</span>
                        <span>$${this.limits.withdrawal.max_amount}</span>
                    </div>
                    <div class="limit-item">
                        <span>حد روزانه باقی‌مانده:</span>
                        <span>$${this.limits.withdrawal.daily_remaining}</span>
                    </div>
                `;
            }
        }
    }
    
    // Display recent transactions
    displayRecentTransactions(transactions) {
        const container = document.querySelector('.recent-transactions');
        if (!container) return;
        
        if (transactions.length === 0) {
            container.innerHTML = '<p class="text-center text-muted">تراکنش اخیری یافت نشد</p>';
            return;
        }
        
        let html = '';
        transactions.forEach(transaction => {
            html += this.generateTransactionHTML(transaction);
        });
        
        container.innerHTML = html;
    }
    
    // Update transactions pagination
    updateTransactionsPagination(paginationData) {
        const container = document.querySelector('.transactions-pagination');
        if (!container) return;
        
        let html = '<nav><ul class="pagination justify-content-center">';
        
        // Previous button
        if (paginationData.current_page > 1) {
            html += `<li class="page-item">
                <a class="page-link" href="#" data-page="${paginationData.current_page - 1}">قبلی</a>
            </li>`;
        }
        
        // Page numbers
        const startPage = Math.max(1, paginationData.current_page - 2);
        const endPage = Math.min(paginationData.total_pages, paginationData.current_page + 2);
        
        for (let i = startPage; i <= endPage; i++) {
            const isActive = i === paginationData.current_page;
            html += `<li class="page-item ${isActive ? 'active' : ''}">
                <a class="page-link" href="#" data-page="${i}">${i}</a>
            </li>`;
        }
        
        // Next button
        if (paginationData.current_page < paginationData.total_pages) {
            html += `<li class="page-item">
                <a class="page-link" href="#" data-page="${paginationData.current_page + 1}">بعدی</a>
            </li>`;
        }
        
        html += '</ul></nav>';
        container.innerHTML = html;
        
        // Add click handlers
        container.querySelectorAll('.page-link').forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                const page = parseInt(e.target.dataset.page);
                this.loadTransactions(page);
            });
        });
    }
    
    // Start periodic updates
    startPeriodicUpdates() {
        // Update balance every 30 seconds
        setInterval(() => {
            this.loadWalletData();
        }, 30000);
        
        // Check for pending transactions every 60 seconds
        setInterval(() => {
            this.checkPendingTransactions();
        }, 60000);
    }
    
    // Check pending transactions
    async checkPendingTransactions() {
        try {
            const response = await fetch('/api/wallet.php?endpoint=pending-status');
            const data = await response.json();
            
            if (data.success && data.data.updates.length > 0) {
                // Show notifications for updated transactions
                data.data.updates.forEach(update => {
                    if (update.status === 'completed') {
                        this.showNotification(`تراکنش ${update.type} با موفقیت تکمیل شد`, 'success');
                    } else if (update.status === 'failed') {
                        this.showNotification(`تراکنش ${update.type} ناموفق بود`, 'error');
                    }
                });
                
                // Reload wallet data
                this.loadWalletData();
            }
        } catch (error) {
            console.error('Error checking pending transactions:', error);
        }
    }
    
    // Exchange currency
    async exchangeCurrency(fromCurrency, toCurrency, amount) {
        try {
            this.showLoading('در حال تبدیل ارز...');
            
            const response = await fetch('/api/wallet.php?endpoint=exchange', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    from_currency: fromCurrency,
                    to_currency: toCurrency,
                    amount: amount
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showNotification(`تبدیل ارز با موفقیت انجام شد`, 'success');
                this.loadWalletData();
                return result.data;
            } else {
                throw new Error(result.error || 'خطا در تبدیل ارز');
            }
        } catch (error) {
            console.error('Error exchanging currency:', error);
            this.showNotification(error.message, 'error');
            throw error;
        } finally {
            this.hideLoading();
        }
    }
    
    // Get exchange rate
    async getExchangeRate(fromCurrency, toCurrency) {
        try {
            const response = await fetch(`/api/wallet.php?endpoint=exchange-rate&from=${fromCurrency}&to=${toCurrency}`);
            const data = await response.json();
            
            if (data.success) {
                return data.data.rate;
            } else {
                throw new Error(data.error || 'خطا در دریافت نرخ ارز');
            }
        } catch (error) {
            console.error('Error getting exchange rate:', error);
            throw error;
        }
    }
    
    // Generate transaction export
    async exportTransactions(format = 'csv', filters = {}) {
        try {
            this.showLoading('در حال تولید فایل...');
            
            const params = new URLSearchParams({
                format: format,
                ...filters
            });
            
            const response = await fetch(`/api/wallet.php?endpoint=export&${params}`);
            
            if (response.ok) {
                const blob = await response.blob();
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `transactions_${Date.now()}.${format}`;
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
                window.URL.revokeObjectURL(url);
                
                this.showNotification('فایل با موفقیت دانلود شد', 'success');
            } else {
                throw new Error('خطا در تولید فایل');
            }
        } catch (error) {
            console.error('Error exporting transactions:', error);
            this.showNotification(error.message, 'error');
        } finally {
            this.hideLoading();
        }
    }
    
    // Utility functions
    formatCurrency(amount, currency = 'USD') {
        return new Intl.NumberFormat('fa-IR', {
            style: 'currency',
            currency: currency,
            minimumFractionDigits: 2
        }).format(amount);
    }
    
    formatDate(dateString) {
        return new Intl.DateTimeFormat('fa-IR', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        }).format(new Date(dateString));
    }
    
    showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `alert alert-${type} notification fade-in`;
        notification.innerHTML = `
            <div class="notification-content">
                <i class="fas fa-${this.getNotificationIcon(type)}"></i>
                <span>${message}</span>
            </div>
            <button class="notification-close">&times;</button>
        `;
        
        // Add to container
        let container = document.querySelector('.notifications-container');
        if (!container) {
            container = document.createElement('div');
            container.className = 'notifications-container';
            document.body.appendChild(container);
        }
        
        container.appendChild(notification);
        
        // Add close functionality
        notification.querySelector('.notification-close').addEventListener('click', () => {
            this.removeNotification(notification);
        });
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            this.removeNotification(notification);
        }, 5000);
    }
    
    removeNotification(notification) {
        if (notification.parentNode) {
            notification.classList.add('fade-out');
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 300);
        }
    }
    
    getNotificationIcon(type) {
        const icons = {
            'success': 'check-circle',
            'error': 'exclamation-circle',
            'warning': 'exclamation-triangle',
            'info': 'info-circle'
        };
        
        return icons[type] || 'info-circle';
    }
    
    showLoading(message = 'در حال بارگذاری...') {
        let loader = document.querySelector('.wallet-loading-overlay');
        if (!loader) {
            loader = document.createElement('div');
            loader.className = 'wallet-loading-overlay';
            loader.innerHTML = `
                <div class="loading-content">
                    <div class="loading-spinner"></div>
                    <div class="loading-message">${message}</div>
                </div>
            `;
            document.body.appendChild(loader);
        } else {
            loader.querySelector('.loading-message').textContent = message;
        }
        
        loader.style.display = 'flex';
        this.isLoading = true;
    }
    
    hideLoading() {
        const loader = document.querySelector('.wallet-loading-overlay');
        if (loader) {
            loader.style.display = 'none';
        }
        this.isLoading = false;
    }
    
    // Event system for wallet updates
    addEventListener(event, callback) {
        if (!this.eventListeners) {
            this.eventListeners = new Map();
        }
        
        if (!this.eventListeners.has(event)) {
            this.eventListeners.set(event, []);
        }
        
        this.eventListeners.get(event).push(callback);
    }
    
    removeEventListener(event, callback) {
        if (this.eventListeners && this.eventListeners.has(event)) {
            const listeners = this.eventListeners.get(event);
            const index = listeners.indexOf(callback);
            if (index > -1) {
                listeners.splice(index, 1);
            }
        }
    }
    
    triggerEvent(event, data) {
        if (this.eventListeners && this.eventListeners.has(event)) {
            this.eventListeners.get(event).forEach(callback => {
                try {
                    callback(data);
                } catch (error) {
                    console.error('Error in wallet event listener:', error);
                }
            });
        }
    }
    
    // Cleanup
    destroy() {
        // Clear intervals
        if (this.updateInterval) {
            clearInterval(this.updateInterval);
        }
        
        if (this.pendingCheckInterval) {
            clearInterval(this.pendingCheckInterval);
        }
        
        // Clear event listeners
        if (this.eventListeners) {
            this.eventListeners.clear();
        }
        
        // Remove loading overlay
        const loader = document.querySelector('.wallet-loading-overlay');
        if (loader) {
            document.body.removeChild(loader);
        }
        
        // Remove notifications container
        const notifications = document.querySelector('.notifications-container');
        if (notifications) {
            document.body.removeChild(notifications);
        }
    }
}

// Initialize wallet manager when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.walletManager = new WalletManager();
});

// Cleanup on page unload
window.addEventListener('beforeunload', () => {
    if (window.walletManager) {
        window.walletManager.destroy();
    }
});

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = WalletManager;
}

