<?php
// admin/settings.php - Site Settings
// تنظیمات سایت

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/auth.php';

// Require admin authentication
requireAuth();
requireAdmin();

// Page settings
$pageTitle = 'Site Settings - Admin';
$bodyClass = 'admin-settings';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'update_general':
            $settings = [
                'site_name' => $_POST['site_name'] ?? '',
                'site_description' => $_POST['site_description'] ?? '',
                'site_keywords' => $_POST['site_keywords'] ?? '',
                'contact_email' => $_POST['contact_email'] ?? '',
                'support_email' => $_POST['support_email'] ?? '',
                'default_language' => $_POST['default_language'] ?? 'en',
                'default_currency' => $_POST['default_currency'] ?? 'USD',
                'timezone' => $_POST['timezone'] ?? 'UTC',
                'maintenance_mode' => isset($_POST['maintenance_mode']) ? '1' : '0'
            ];
            
            foreach ($settings as $key => $value) {
                execute(
                    "INSERT INTO site_settings (setting_key, setting_value, updated_at, updated_by) 
                     VALUES (?, ?, NOW(), ?)
                     ON DUPLICATE KEY UPDATE 
                     setting_value = VALUES(setting_value), 
                     updated_at = NOW(), 
                     updated_by = VALUES(updated_by)",
                    [$key, $value, $currentUser['id']]
                );
            }
            $successMessage = "General settings updated successfully";
            break;
            
        case 'update_betting':
            $settings = [
                'min_bet_amount' => $_POST['min_bet_amount'] ?? '1.00',
                'max_bet_amount' => $_POST['max_bet_amount'] ?? '10000.00',
                'max_win_amount' => $_POST['max_win_amount'] ?? '100000.00',
                'house_edge' => $_POST['house_edge'] ?? '2.00',
                'auto_cashout_enabled' => isset($_POST['auto_cashout_enabled']) ? '1' : '0',
                'max_auto_cashout' => $_POST['max_auto_cashout'] ?? '1000.00',
                'betting_enabled' => isset($_POST['betting_enabled']) ? '1' : '0'
            ];
            
            foreach ($settings as $key => $value) {
                execute(
                    "INSERT INTO site_settings (setting_key, setting_value, updated_at, updated_by) 
                     VALUES (?, ?, NOW(), ?)
                     ON DUPLICATE KEY UPDATE 
                     setting_value = VALUES(setting_value), 
                     updated_at = NOW(), 
                     updated_by = VALUES(updated_by)",
                    [$key, $value, $currentUser['id']]
                );
            }
            $successMessage = "Betting settings updated successfully";
            break;
            
        case 'update_payments':
            $settings = [
                'min_deposit' => $_POST['min_deposit'] ?? '10.00',
                'max_deposit' => $_POST['max_deposit'] ?? '10000.00',
                'min_withdrawal' => $_POST['min_withdrawal'] ?? '20.00',
                'max_withdrawal' => $_POST['max_withdrawal'] ?? '5000.00',
                'withdrawal_fee' => $_POST['withdrawal_fee'] ?? '0.00',
                'withdrawal_processing_time' => $_POST['withdrawal_processing_time'] ?? '24',
                'auto_withdrawal_limit' => $_POST['auto_withdrawal_limit'] ?? '1000.00',
                'kyc_required' => isset($_POST['kyc_required']) ? '1' : '0',
                'deposits_enabled' => isset($_POST['deposits_enabled']) ? '1' : '0',
                'withdrawals_enabled' => isset($_POST['withdrawals_enabled']) ? '1' : '0'
            ];
            
            foreach ($settings as $key => $value) {
                execute(
                    "INSERT INTO site_settings (setting_key, setting_value, updated_at, updated_by) 
                     VALUES (?, ?, NOW(), ?)
                     ON DUPLICATE KEY UPDATE 
                     setting_value = VALUES(setting_value), 
                     updated_at = NOW(), 
                     updated_by = VALUES(updated_by)",
                    [$key, $value, $currentUser['id']]
                );
            }
            $successMessage = "Payment settings updated successfully";
            break;
            
        case 'update_security':
            $settings = [
                'registration_enabled' => isset($_POST['registration_enabled']) ? '1' : '0',
                'email_verification' => isset($_POST['email_verification']) ? '1' : '0',
                'two_factor_auth' => isset($_POST['two_factor_auth']) ? '1' : '0',
                'session_timeout' => $_POST['session_timeout'] ?? '3600',
                'max_login_attempts' => $_POST['max_login_attempts'] ?? '5',
                'lockout_duration' => $_POST['lockout_duration'] ?? '900',
                'password_min_length' => $_POST['password_min_length'] ?? '8',
                'require_strong_password' => isset($_POST['require_strong_password']) ? '1' : '0'
            ];
            
            foreach ($settings as $key => $value) {
                execute(
                    "INSERT INTO site_settings (setting_key, setting_value, updated_at, updated_by) 
                     VALUES (?, ?, NOW(), ?)
                     ON DUPLICATE KEY UPDATE 
                     setting_value = VALUES(setting_value), 
                     updated_at = NOW(), 
                     updated_by = VALUES(updated_by)",
                    [$key, $value, $currentUser['id']]
                );
            }
            $successMessage = "Security settings updated successfully";
            break;
    }
}

// Get current settings
$currentSettings = [];
$settingsResult = fetchAll("SELECT setting_key, setting_value FROM site_settings");
foreach ($settingsResult as $setting) {
    $currentSettings[$setting['setting_key']] = $setting['setting_value'];
}

// Default values
$defaults = [
    'site_name' => 'Crashify',
    'site_description' => 'The Ultimate Crash Betting Experience',
    'site_keywords' => 'crash game, betting, casino, gambling',
    'contact_email' => 'contact@crashify.com',
    'support_email' => 'support@crashify.com',
    'default_language' => 'en',
    'default_currency' => 'USD',
    'timezone' => 'UTC',
    'maintenance_mode' => '0',
    'min_bet_amount' => '1.00',
    'max_bet_amount' => '10000.00',
    'max_win_amount' => '100000.00',
    'house_edge' => '2.00',
    'auto_cashout_enabled' => '1',
    'max_auto_cashout' => '1000.00',
    'betting_enabled' => '1',
    'min_deposit' => '10.00',
    'max_deposit' => '10000.00',
    'min_withdrawal' => '20.00',
    'max_withdrawal' => '5000.00',
    'withdrawal_fee' => '0.00',
    'withdrawal_processing_time' => '24',
    'auto_withdrawal_limit' => '1000.00',
    'kyc_required' => '0',
    'deposits_enabled' => '1',
    'withdrawals_enabled' => '1',
    'registration_enabled' => '1',
    'email_verification' => '1',
    'two_factor_auth' => '0',
    'session_timeout' => '3600',
    'max_login_attempts' => '5',
    'lockout_duration' => '900',
    'password_min_length' => '8',
    'require_strong_password' => '1'
];

// Merge with defaults
foreach ($defaults as $key => $value) {
    if (!isset($currentSettings[$key])) {
        $currentSettings[$key] = $value;
    }
}

// Get current tab
$tab = $_GET['tab'] ?? 'general';

include '../includes/header.php';
?>

<!-- Admin Settings -->
<div class="admin-container">
    <?php include 'sidebar.php'; ?>
    
    <main class="admin-main">
        <!-- Header -->
        <header class="admin-header">
            <div class="header-left">
                <h1 class="page-title">Site Settings</h1>
                <p class="page-subtitle">Configure website settings and preferences</p>
            </div>
            
            <div class="header-right">
                <button class="btn btn-outline" onclick="resetToDefaults()">
                    <i class="fas fa-undo"></i>
                    Reset to Defaults
                </button>
                <button class="btn btn-success" onclick="saveAllSettings()">
                    <i class="fas fa-save"></i>
                    Save All
                </button>
            </div>
        </header>
        
        <?php if (isset($successMessage)): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i>
            <?php echo $successMessage; ?>
        </div>
        <?php endif; ?>
        
        <!-- Settings Tabs -->
        <section class="tabs-section">
            <nav class="settings-tabs">
                <a href="?tab=general" class="tab-link <?php echo $tab === 'general' ? 'active' : ''; ?>">
                    <i class="fas fa-cog"></i>
                    General
                </a>
                <a href="?tab=betting" class="tab-link <?php echo $tab === 'betting' ? 'active' : ''; ?>">
                    <i class="fas fa-dice"></i>
                    Betting
                </a>
                <a href="?tab=payments" class="tab-link <?php echo $tab === 'payments' ? 'active' : ''; ?>">
                    <i class="fas fa-credit-card"></i>
                    Payments
                </a>
                <a href="?tab=security" class="tab-link <?php echo $tab === 'security' ? 'active' : ''; ?>">
                    <i class="fas fa-shield-alt"></i>
                    Security
                </a>
                <a href="?tab=notifications" class="tab-link <?php echo $tab === 'notifications' ? 'active' : ''; ?>">
                    <i class="fas fa-bell"></i>
                    Notifications
                </a>
            </nav>
        </section>
        
        <!-- General Settings -->
        <?php if ($tab === 'general'): ?>
        <section class="settings-section">
            <form method="POST" class="settings-form">
                <input type="hidden" name="action" value="update_general">
                
                <div class="settings-card">
                    <div class="card-header">
                        <h3>Site Information</h3>
                        <p>Basic website information and branding</p>
                    </div>
                    
                    <div class="card-body">
                        <div class="form-row">
                            <div class="form-group">
                                <label for="siteName">Site Name</label>
                                <input type="text" name="site_name" id="siteName" 
                                       value="<?php echo htmlspecialchars($currentSettings['site_name']); ?>" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="defaultLanguage">Default Language</label>
                                <select name="default_language" id="defaultLanguage">
                                    <option value="en" <?php echo $currentSettings['default_language'] === 'en' ? 'selected' : ''; ?>>English</option>
                                    <option value="fa" <?php echo $currentSettings['default_language'] === 'fa' ? 'selected' : ''; ?>>Persian</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="siteDescription">Site Description</label>
                            <textarea name="site_description" id="siteDescription" rows="3"><?php echo htmlspecialchars($currentSettings['site_description']); ?></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label for="siteKeywords">SEO Keywords</label>
                            <input type="text" name="site_keywords" id="siteKeywords" 
                                   value="<?php echo htmlspecialchars($currentSettings['site_keywords']); ?>"
                                   placeholder="Separate keywords with commas">
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label for="contactEmail">Contact Email</label>
                                <input type="email" name="contact_email" id="contactEmail" 
                                       value="<?php echo htmlspecialchars($currentSettings['contact_email']); ?>" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="supportEmail">Support Email</label>
                                <input type="email" name="support_email" id="supportEmail" 
                                       value="<?php echo htmlspecialchars($currentSettings['support_email']); ?>" required>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label for="defaultCurrency">Default Currency</label>
                                <select name="default_currency" id="defaultCurrency">
                                    <option value="USD" <?php echo $currentSettings['default_currency'] === 'USD' ? 'selected' : ''; ?>>USD - US Dollar</option>
                                    <option value="EUR" <?php echo $currentSettings['default_currency'] === 'EUR' ? 'selected' : ''; ?>>EUR - Euro</option>
                                    <option value="GBP" <?php echo $currentSettings['default_currency'] === 'GBP' ? 'selected' : ''; ?>>GBP - British Pound</option>
                                    <option value="BTC" <?php echo $currentSettings['default_currency'] === 'BTC' ? 'selected' : ''; ?>>BTC - Bitcoin</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="timezone">Timezone</label>
                                <select name="timezone" id="timezone">
                                    <option value="UTC" <?php echo $currentSettings['timezone'] === 'UTC' ? 'selected' : ''; ?>>UTC</option>
                                    <option value="America/New_York" <?php echo $currentSettings['timezone'] === 'America/New_York' ? 'selected' : ''; ?>>Eastern Time</option>
                                    <option value="Europe/London" <?php echo $currentSettings['timezone'] === 'Europe/London' ? 'selected' : ''; ?>>London</option>
                                    <option value="Asia/Tehran" <?php echo $currentSettings['timezone'] === 'Asia/Tehran' ? 'selected' : ''; ?>>Tehran</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <div class="checkbox-group">
                                <label class="checkbox-label">
                                    <input type="checkbox" name="maintenance_mode" 
                                           <?php echo $currentSettings['maintenance_mode'] === '1' ? 'checked' : ''; ?>>
                                    <span class="checkbox-custom"></span>
                                    Enable Maintenance Mode
                                </label>
                                <p class="help-text">When enabled, only administrators can access the site</p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i>
                        Save General Settings
                    </button>
                </div>
            </form>
        </section>
        <?php endif; ?>
        
        <!-- Betting Settings -->
        <?php if ($tab === 'betting'): ?>
        <section class="settings-section">
            <form method="POST" class="settings-form">
                <input type="hidden" name="action" value="update_betting">
                
                <div class="settings-card">
                    <div class="card-header">
                        <h3>Betting Configuration</h3>
                        <p>Configure betting limits and game settings</p>
                    </div>
                    
                    <div class="card-body">
                        <div class="form-row">
                            <div class="form-group">
                                <label for="minBetAmount">Minimum Bet Amount</label>
                                <input type="number" name="min_bet_amount" id="minBetAmount" 
                                       value="<?php echo $currentSettings['min_bet_amount']; ?>" 
                                       step="0.01" min="0.01" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="maxBetAmount">Maximum Bet Amount</label>
                                <input type="number" name="max_bet_amount" id="maxBetAmount" 
                                       value="<?php echo $currentSettings['max_bet_amount']; ?>" 
                                       step="0.01" min="1" required>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label for="maxWinAmount">Maximum Win Amount</label>
                                <input type="number" name="max_win_amount" id="maxWinAmount" 
                                       value="<?php echo $currentSettings['max_win_amount']; ?>" 
                                       step="0.01" min="1" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="houseEdge">House Edge (%)</label>
                                <input type="number" name="house_edge" id="houseEdge" 
                                       value="<?php echo $currentSettings['house_edge']; ?>" 
                                       step="0.01" min="0" max="10" required>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="maxAutoCashout">Maximum Auto Cashout</label>
                            <input type="number" name="max_auto_cashout" id="maxAutoCashout" 
                                   value="<?php echo $currentSettings['max_auto_cashout']; ?>" 
                                   step="0.01" min="1" required>
                        </div>
                        
                        <div class="checkbox-group">
                            <label class="checkbox-label">
                                <input type="checkbox" name="auto_cashout_enabled" 
                                       <?php echo $currentSettings['auto_cashout_enabled'] === '1' ? 'checked' : ''; ?>>
                                <span class="checkbox-custom"></span>
                                Enable Auto Cashout Feature
                            </label>
                        </div>
                        
                        <div class="checkbox-group">
                            <label class="checkbox-label">
                                <input type="checkbox" name="betting_enabled" 
                                       <?php echo $currentSettings['betting_enabled'] === '1' ? 'checked' : ''; ?>>
                                <span class="checkbox-custom"></span>
                                Enable Betting (Master Switch)
                            </label>
                            <p class="help-text">When disabled, all betting functionality is turned off</p>
                        </div>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i>
                        Save Betting Settings
                    </button>
                </div>
            </form>
        </section>
        <?php endif; ?>
        
        <!-- Payment Settings -->
        <?php if ($tab === 'payments'): ?>
        <section class="settings-section">
            <form method="POST" class="settings-form">
                <input type="hidden" name="action" value="update_payments">
                
                <div class="settings-card">
                    <div class="card-header">
                        <h3>Payment Configuration</h3>
                        <p>Configure deposit and withdrawal settings</p>
                    </div>
                    
                    <div class="card-body">
                        <div class="form-section">
                            <h4>Deposit Settings</h4>
                            
                            <div class="form-row">
                                <div class="form-group">
                                    <label for="minDeposit">Minimum Deposit</label>
                                    <input type="number" name="min_deposit" id="minDeposit" 
                                           value="<?php echo $currentSettings['min_deposit']; ?>" 
                                           step="0.01" min="1" required>
                                </div>
                                
                                <div class="form-group">
                                    <label for="maxDeposit">Maximum Deposit</label>
                                    <input type="number" name="max_deposit" id="maxDeposit" 
                                           value="<?php echo $currentSettings['max_deposit']; ?>" 
                                           step="0.01" min="1" required>
                                </div>
                            </div>
                            
                            <div class="checkbox-group">
                                <label class="checkbox-label">
                                    <input type="checkbox" name="deposits_enabled" 
                                           <?php echo $currentSettings['deposits_enabled'] === '1' ? 'checked' : ''; ?>>
                                    <span class="checkbox-custom"></span>
                                    Enable Deposits
                                </label>
                            </div>
                        </div>
                        
                        <div class="form-section">
                            <h4>Withdrawal Settings</h4>
                            
                            <div class="form-row">
                                <div class="form-group">
                                    <label for="minWithdrawal">Minimum Withdrawal</label>
                                    <input type="number" name="min_withdrawal" id="minWithdrawal" 
                                           value="<?php echo $currentSettings['min_withdrawal']; ?>" 
                                           step="0.01" min="1" required>
                                </div>
                                
                                <div class="form-group">
                                    <label for="maxWithdrawal">Maximum Withdrawal</label>
                                    <input type="number" name="max_withdrawal" id="maxWithdrawal" 
                                           value="<?php echo $currentSettings['max_withdrawal']; ?>" 
                                           step="0.01" min="1" required>
                                </div>
                            </div>
                            
                            <div class="form-row">
                                <div class="form-group">
                                    <label for="withdrawalFee">Withdrawal Fee</label>
                                    <input type="number" name="withdrawal_fee" id="withdrawalFee" 
                                           value="<?php echo $currentSettings['withdrawal_fee']; ?>" 
                                           step="0.01" min="0">
                                </div>
                                
                                <div class="form-group">
                                    <label for="withdrawalProcessingTime">Processing Time (hours)</label>
                                    <input type="number" name="withdrawal_processing_time" id="withdrawalProcessingTime" 
                                           value="<?php echo $currentSettings['withdrawal_processing_time']; ?>" 
                                           min="1" required>
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label for="autoWithdrawalLimit">Auto Withdrawal Limit</label>
                                <input type="number" name="auto_withdrawal_limit" id="autoWithdrawalLimit" 
                                       value="<?php echo $currentSettings['auto_withdrawal_limit']; ?>" 
                                       step="0.01" min="0">
                                <p class="help-text">Withdrawals above this amount require manual approval</p>
                            </div>
                            
                            <div class="checkbox-group">
                                <label class="checkbox-label">
                                    <input type="checkbox" name="withdrawals_enabled" 
                                           <?php echo $currentSettings['withdrawals_enabled'] === '1' ? 'checked' : ''; ?>>
                                    <span class="checkbox-custom"></span>
                                    Enable Withdrawals
                                </label>
                            </div>
                            
                            <div class="checkbox-group">
                                <label class="checkbox-label">
                                    <input type="checkbox" name="kyc_required" 
                                           <?php echo $currentSettings['kyc_required'] === '1' ? 'checked' : ''; ?>>
                                    <span class="checkbox-custom"></span>
                                    Require KYC for Withdrawals
                                </label>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i>
                        Save Payment Settings
                    </button>
                </div>
            </form>
        </section>
        <?php endif; ?>
        
        <!-- Security Settings -->
        <?php if ($tab === 'security'): ?>
        <section class="settings-section">
            <form method="POST" class="settings-form">
                <input type="hidden" name="action" value="update_security">
                
                <div class="settings-card">
                    <div class="card-header">
                        <h3>Security Configuration</h3>
                        <p>Configure authentication and security settings</p>
                    </div>
                    
                    <div class="card-body">
                        <div class="form-section">
                            <h4>Registration & Authentication</h4>
                            
                            <div class="checkbox-group">
                                <label class="checkbox-label">
                                    <input type="checkbox" name="registration_enabled" 
                                           <?php echo $currentSettings['registration_enabled'] === '1' ? 'checked' : ''; ?>>
                                    <span class="checkbox-custom"></span>
                                    Enable User Registration
                                </label>
                            </div>
                            
                            <div class="checkbox-group">
                                <label class="checkbox-label">
                                    <input type="checkbox" name="email_verification" 
                                           <?php echo $currentSettings['email_verification'] === '1' ? 'checked' : ''; ?>>
                                    <span class="checkbox-custom"></span>
                                    Require Email Verification
                                </label>
                            </div>
                            
                            <div class="checkbox-group">
                                <label class="checkbox-label">
                                    <input type="checkbox" name="two_factor_auth" 
                                           <?php echo $currentSettings['two_factor_auth'] === '1' ? 'checked' : ''; ?>>
                                    <span class="checkbox-custom"></span>
                                    Enable Two-Factor Authentication
                                </label>
                            </div>
                        </div>
                        
                        <div class="form-section">
                            <h4>Session & Login Security</h4>
                            
                            <div class="form-row">
                                <div class="form-group">
                                    <label for="sessionTimeout">Session Timeout (seconds)</label>
                                    <input type="number" name="session_timeout" id="sessionTimeout" 
                                           value="<?php echo $currentSettings['session_timeout']; ?>" 
                                           min="300" required>
                                </div>
                                
                                <div class="form-group">
                                    <label for="maxLoginAttempts">Max Login Attempts</label>
                                    <input type="number" name="max_login_attempts" id="maxLoginAttempts" 
                                           value="<?php echo $currentSettings['max_login_attempts']; ?>" 
                                           min="3" max="10" required>
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label for="lockoutDuration">Lockout Duration (seconds)</label>
                                <input type="number" name="lockout_duration" id="lockoutDuration" 
                                       value="<?php echo $currentSettings['lockout_duration']; ?>" 
                                       min="300" required>
                            </div>
                        </div>
                        
                        <div class="form-section">
                            <h4>Password Requirements</h4>
                            
                            <div class="form-group">
                                <label for="passwordMinLength">Minimum Password Length</label>
                                <input type="number" name="password_min_length" id="passwordMinLength" 
                                       value="<?php echo $currentSettings['password_min_length']; ?>" 
                                       min="6" max="20" required>
                            </div>
                            
                            <div class="checkbox-group">
                                <label class="checkbox-label">
                                    <input type="checkbox" name="require_strong_password" 
                                           <?php echo $currentSettings['require_strong_password'] === '1' ? 'checked' : ''; ?>>
                                    <span class="checkbox-custom"></span>
                                    Require Strong Passwords
                                </label>
                                <p class="help-text">Requires uppercase, lowercase, numbers, and special characters</p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i>
                        Save Security Settings
                    </button>
                </div>
            </form>
        </section>
        <?php endif; ?>
        
        <!-- Notifications Settings -->
        <?php if ($tab === 'notifications'): ?>
        <section class="settings-section">
            <div class="settings-card">
                <div class="card-header">
                    <h3>Notification Settings</h3>
                    <p>Configure email and system notifications</p>
                </div>
                
                <div class="card-body">
                    <div class="notification-placeholder">
                        <i class="fas fa-bell"></i>
                        <h4>Notification Settings</h4>
                        <p>Email templates and notification preferences will be configured here.</p>
                        <button class="btn btn-primary">Configure Notifications</button>
                    </div>
                </div>
            </div>
        </section>
        <?php endif; ?>
    </main>
</div>

<script>
function resetToDefaults() {
    if (confirm('Are you sure you want to reset all settings to default values? This action cannot be undone.')) {
        // Reset to defaults
        window.location.href = '/admin/settings.php?action=reset_defaults';
    }
}

function saveAllSettings() {
    // Save all forms
    const forms = document.querySelectorAll('.settings-form');
    let savedCount = 0;
    
    forms.forEach(form => {
        const formData = new FormData(form);
        
        fetch('/admin/settings.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.text())
        .then(() => {
            savedCount++;
            if (savedCount === forms.length) {
                alert('All settings saved successfully!');
                window.location.reload();
            }
        })
        .catch(error => {
            console.error('Error saving settings:', error);
        });
    });
}

// Auto-save functionality
document.addEventListener('DOMContentLoaded', function() {
    const inputs = document.querySelectorAll('input, select, textarea');
    
    inputs.forEach(input => {
        input.addEventListener('change', function() {
            // Add visual indicator that settings have changed
            const saveButton = document.querySelector('.btn-success');
            if (saveButton) {
                saveButton.classList.add('pulse');
            }
        });
    });
});
</script>

<?php include '../includes/footer.php'; ?>
