<?php
// admin/reports.php - Reports and Analytics
// گزارشات و آنالیتیک

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/auth.php';

// Require admin authentication
requireAuth();
requireAdmin();

// Page settings
$pageTitle = 'Reports & Analytics - Admin';
$bodyClass = 'admin-reports';

// Get date range parameters
$dateFrom = $_GET['date_from'] ?? date('Y-m-d', strtotime('-30 days'));
$dateTo = $_GET['date_to'] ?? date('Y-m-d');
$period = $_GET['period'] ?? '30days';

// Get current tab
$tab = $_GET['tab'] ?? 'overview';

// Financial Reports
$financialStats = fetchRow(
    "SELECT 
        SUM(CASE WHEN type = 'deposit' AND status = 'completed' THEN amount ELSE 0 END) as total_deposits,
        SUM(CASE WHEN type = 'withdrawal' AND status = 'completed' THEN amount ELSE 0 END) as total_withdrawals,
        COUNT(CASE WHEN type = 'deposit' AND status = 'completed' THEN 1 END) as deposit_count,
        COUNT(CASE WHEN type = 'withdrawal' AND status = 'completed' THEN 1 END) as withdrawal_count,
        SUM(CASE WHEN type = 'deposit' AND status = 'completed' AND DATE(created_at) BETWEEN ? AND ? THEN amount ELSE 0 END) as period_deposits,
        SUM(CASE WHEN type = 'withdrawal' AND status = 'completed' AND DATE(created_at) BETWEEN ? AND ? THEN amount ELSE 0 END) as period_withdrawals
     FROM transactions",
    [$dateFrom, $dateTo, $dateFrom, $dateTo]
);

// Betting Reports
$bettingStats = fetchRow(
    "SELECT 
        COUNT(*) as total_bets,
        SUM(bet_amount) as total_wagered,
        SUM(CASE WHEN status = 'won' THEN payout_amount ELSE 0 END) as total_payouts,
        SUM(bet_amount) - SUM(CASE WHEN status = 'won' THEN payout_amount ELSE 0 END) as house_profit,
        AVG(bet_amount) as avg_bet_amount,
        COUNT(CASE WHEN DATE(created_at) BETWEEN ? AND ? THEN 1 END) as period_bets,
        SUM(CASE WHEN DATE(created_at) BETWEEN ? AND ? THEN bet_amount ELSE 0 END) as period_wagered
     FROM user_bets",
    [$dateFrom, $dateTo, $dateFrom, $dateTo]
);

// User Reports
$userStats = fetchRow(
    "SELECT 
        COUNT(*) as total_users,
        COUNT(CASE WHEN status = 'active' THEN 1 END) as active_users,
        COUNT(CASE WHEN DATE(created_at) BETWEEN ? AND ? THEN 1 END) as new_users_period,
        COUNT(CASE WHEN last_activity > DATE_SUB(NOW(), INTERVAL 24 HOUR) THEN 1 END) as active_24h,
        COUNT(CASE WHEN last_activity > DATE_SUB(NOW(), INTERVAL 7 DAY) THEN 1 END) as active_7d,
        COUNT(CASE WHEN last_activity > DATE_SUB(NOW(), INTERVAL 30 DAY) THEN 1 END) as active_30d
     FROM users",
    [$dateFrom, $dateTo]
);

// Daily revenue data for charts
$dailyRevenue = fetchAll(
    "SELECT 
        DATE(created_at) as date,
        SUM(CASE WHEN type = 'deposit' AND status = 'completed' THEN amount ELSE 0 END) as deposits,
        SUM(CASE WHEN type = 'withdrawal' AND status = 'completed' THEN amount ELSE 0 END) as withdrawals
     FROM transactions 
     WHERE DATE(created_at) BETWEEN ? AND ?
     GROUP BY DATE(created_at)
     ORDER BY date ASC",
    [$dateFrom, $dateTo]
);

// Daily betting data
$dailyBetting = fetchAll(
    "SELECT 
        DATE(created_at) as date,
        COUNT(*) as bet_count,
        SUM(bet_amount) as total_wagered,
        SUM(CASE WHEN status = 'won' THEN payout_amount ELSE 0 END) as total_payouts
     FROM user_bets 
     WHERE DATE(created_at) BETWEEN ? AND ?
     GROUP BY DATE(created_at)
     ORDER BY date ASC",
    [$dateFrom, $dateTo]
);

// Top players by wagered amount
$topPlayers = fetchAll(
    "SELECT 
        u.username,
        u.email,
        COUNT(ub.id) as total_bets,
        SUM(ub.bet_amount) as total_wagered,
        SUM(CASE WHEN ub.status = 'won' THEN ub.payout_amount ELSE 0 END) as total_winnings,
        (SUM(ub.bet_amount) - SUM(CASE WHEN ub.status = 'won' THEN ub.payout_amount ELSE 0 END)) as net_loss
     FROM users u
     JOIN user_bets ub ON u.id = ub.user_id
     WHERE DATE(ub.created_at) BETWEEN ? AND ?
     GROUP BY u.id
     ORDER BY total_wagered DESC
     LIMIT 10",
    [$dateFrom, $dateTo]
);

// Game performance
$gameStats = fetchAll(
    "SELECT 
        game_type,
        COUNT(*) as bet_count,
        SUM(bet_amount) as total_wagered,
        SUM(CASE WHEN status = 'won' THEN payout_amount ELSE 0 END) as total_payouts,
        (SUM(bet_amount) - SUM(CASE WHEN status = 'won' THEN payout_amount ELSE 0 END)) as profit
     FROM user_bets 
     WHERE DATE(created_at) BETWEEN ? AND ?
     GROUP BY game_type
     ORDER BY total_wagered DESC",
    [$dateFrom, $dateTo]
);

include '../includes/header.php';
?>

<!-- Admin Reports -->
<div class="admin-container">
    <?php include 'sidebar.php'; ?>
    
    <main class="admin-main">
        <!-- Header -->
        <header class="admin-header">
            <div class="header-left">
                <h1 class="page-title">Reports & Analytics</h1>
                <p class="page-subtitle">Comprehensive business intelligence and reporting</p>
            </div>
            
            <div class="header-right">
                <div class="date-range-picker">
                    <input type="date" id="dateFrom" value="<?php echo $dateFrom; ?>">
                    <span>to</span>
                    <input type="date" id="dateTo" value="<?php echo $dateTo; ?>">
                    <button class="btn btn-primary" onclick="updateDateRange()">
                        <i class="fas fa-filter"></i>
                        Apply
                    </button>
                </div>
                
                <button class="btn btn-outline" onclick="exportReport()">
                    <i class="fas fa-download"></i>
                    Export
                </button>
            </div>
        </header>
        
        <!-- Quick Period Filters -->
        <section class="period-filters">
            <div class="filter-buttons">
                <button class="filter-btn <?php echo $period === 'today' ? 'active' : ''; ?>" onclick="setPeriod('today')">Today</button>
                <button class="filter-btn <?php echo $period === '7days' ? 'active' : ''; ?>" onclick="setPeriod('7days')">7 Days</button>
                <button class="filter-btn <?php echo $period === '30days' ? 'active' : ''; ?>" onclick="setPeriod('30days')">30 Days</button>
                <button class="filter-btn <?php echo $period === '90days' ? 'active' : ''; ?>" onclick="setPeriod('90days')">90 Days</button>
                <button class="filter-btn <?php echo $period === 'custom' ? 'active' : ''; ?>" onclick="setPeriod('custom')">Custom</button>
            </div>
        </section>
        
        <!-- Report Tabs -->
        <section class="tabs-section">
            <nav class="report-tabs">
                <a href="?tab=overview&period=<?php echo $period; ?>&date_from=<?php echo $dateFrom; ?>&date_to=<?php echo $dateTo; ?>" 
                   class="tab-link <?php echo $tab === 'overview' ? 'active' : ''; ?>">
                    <i class="fas fa-chart-line"></i>
                    Overview
                </a>
                <a href="?tab=financial&period=<?php echo $period; ?>&date_from=<?php echo $dateFrom; ?>&date_to=<?php echo $dateTo; ?>" 
                   class="tab-link <?php echo $tab === 'financial' ? 'active' : ''; ?>">
                    <i class="fas fa-dollar-sign"></i>
                    Financial
                </a>
                <a href="?tab=betting&period=<?php echo $period; ?>&date_from=<?php echo $dateFrom; ?>&date_to=<?php echo $dateTo; ?>" 
                   class="tab-link <?php echo $tab === 'betting' ? 'active' : ''; ?>">
                    <i class="fas fa-dice"></i>
                    Betting
                </a>
                <a href="?tab=users&period=<?php echo $period; ?>&date_from=<?php echo $dateFrom; ?>&date_to=<?php echo $dateTo; ?>" 
                   class="tab-link <?php echo $tab === 'users' ? 'active' : ''; ?>">
                    <i class="fas fa-users"></i>
                    Users
                </a>
                <a href="?tab=games&period=<?php echo $period; ?>&date_from=<?php echo $dateFrom; ?>&date_to=<?php echo $dateTo; ?>" 
                   class="tab-link <?php echo $tab === 'games' ? 'active' : ''; ?>">
                    <i class="fas fa-gamepad"></i>
                    Games
                </a>
            </nav>
        </section>
        
        <!-- Overview Tab -->
        <?php if ($tab === 'overview'): ?>
        <section class="report-section">
            <!-- Key Metrics -->
            <div class="metrics-grid">
                <div class="metric-card revenue">
                    <div class="metric-icon">
                        <i class="fas fa-chart-line"></i>
                    </div>
                    <div class="metric-content">
                        <div class="metric-value">$<?php echo number_format(($financialStats['period_deposits'] ?? 0) - ($financialStats['period_withdrawals'] ?? 0), 2); ?></div>
                        <div class="metric-label">Net Revenue</div>
                        <div class="metric-change positive">
                            <i class="fas fa-arrow-up"></i>
                            +12.5%
                        </div>
                    </div>
                </div>
                
                <div class="metric-card deposits">
                    <div class="metric-icon">
                        <i class="fas fa-arrow-down"></i>
                    </div>
                    <div class="metric-content">
                        <div class="metric-value">$<?php echo number_format($financialStats['period_deposits'] ?? 0, 2); ?></div>
                        <div class="metric-label">Total Deposits</div>
                        <div class="metric-change positive">
                            <i class="fas fa-arrow-up"></i>
                            +8.3%
                        </div>
                    </div>
                </div>
                
                <div class="metric-card profit">
                    <div class="metric-icon">
                        <i class="fas fa-trophy"></i>
                    </div>
                    <div class="metric-content">
                        <div class="metric-value">$<?php echo number_format($bettingStats['house_profit'] ?? 0, 2); ?></div>
                        <div class="metric-label">House Profit</div>
                        <div class="metric-change positive">
                            <i class="fas fa-arrow-up"></i>
                            +15.7%
                        </div>
                    </div>
                </div>
                
                <div class="metric-card users">
                    <div class="metric-icon">
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="metric-content">
                        <div class="metric-value"><?php echo number_format($userStats['new_users_period'] ?? 0); ?></div>
                        <div class="metric-label">New Users</div>
                        <div class="metric-change positive">
                            <i class="fas fa-arrow-up"></i>
                            +22.1%
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Charts -->
            <div class="charts-grid">
                <div class="chart-card">
                    <div class="chart-header">
                        <h3>Revenue Overview</h3>
                        <div class="chart-controls">
                            <select id="revenueChartType">
                                <option value="line">Line Chart</option>
                                <option value="bar">Bar Chart</option>
                            </select>
                        </div>
                    </div>
                    <div class="chart-container">
                        <canvas id="revenueChart" width="800" height="300"></canvas>
                    </div>
                </div>
                
                <div class="chart-card">
                    <div class="chart-header">
                        <h3>Betting Activity</h3>
                    </div>
                    <div class="chart-container">
                        <canvas id="bettingChart" width="800" height="300"></canvas>
                    </div>
                </div>
            </div>
        </section>
        <?php endif; ?>
        
        <!-- Financial Tab -->
        <?php if ($tab === 'financial'): ?>
        <section class="report-section">
            <div class="financial-summary">
                <div class="summary-cards">
                    <div class="summary-card">
                        <h4>Total Deposits</h4>
                        <div class="amount positive">$<?php echo number_format($financialStats['total_deposits'] ?? 0, 2); ?></div>
                        <div class="count"><?php echo number_format($financialStats['deposit_count'] ?? 0); ?> transactions</div>
                    </div>
                    
                    <div class="summary-card">
                        <h4>Total Withdrawals</h4>
                        <div class="amount negative">$<?php echo number_format($financialStats['total_withdrawals'] ?? 0, 2); ?></div>
                        <div class="count"><?php echo number_format($financialStats['withdrawal_count'] ?? 0); ?> transactions</div>
                    </div>
                    
                    <div class="summary-card">
                        <h4>Net Balance</h4>
                        <div class="amount <?php echo ($financialStats['total_deposits'] - $financialStats['total_withdrawals']) >= 0 ? 'positive' : 'negative'; ?>">
                            $<?php echo number_format(($financialStats['total_deposits'] ?? 0) - ($financialStats['total_withdrawals'] ?? 0), 2); ?>
                        </div>
                        <div class="count">All time</div>
                    </div>
                </div>
            </div>
            
            <div class="financial-chart">
                <div class="chart-header">
                    <h3>Daily Financial Activity</h3>
                </div>
                <div class="chart-container">
                    <canvas id="financialChart" width="1000" height="400"></canvas>
                </div>
            </div>
        </section>
        <?php endif; ?>
        
        <!-- Betting Tab -->
        <?php if ($tab === 'betting'): ?>
        <section class="report-section">
            <div class="betting-summary">
                <div class="summary-cards">
                    <div class="summary-card">
                        <h4>Total Bets</h4>
                        <div class="amount"><?php echo number_format($bettingStats['total_bets'] ?? 0); ?></div>
                        <div class="count">All time</div>
                    </div>
                    
                    <div class="summary-card">
                        <h4>Total Wagered</h4>
                        <div class="amount">$<?php echo number_format($bettingStats['total_wagered'] ?? 0, 2); ?></div>
                        <div class="count">Avg: $<?php echo number_format($bettingStats['avg_bet_amount'] ?? 0, 2); ?></div>
                    </div>
                    
                    <div class="summary-card">
                        <h4>Total Payouts</h4>
                        <div class="amount negative">$<?php echo number_format($bettingStats['total_payouts'] ?? 0, 2); ?></div>
                        <div class="count">To players</div>
                    </div>
                    
                    <div class="summary-card">
                        <h4>House Profit</h4>
                        <div class="amount positive">$<?php echo number_format($bettingStats['house_profit'] ?? 0, 2); ?></div>
                        <div class="count">
                            <?php 
                            $profitMargin = $bettingStats['total_wagered'] > 0 ? 
                                ($bettingStats['house_profit'] / $bettingStats['total_wagered']) * 100 : 0;
                            echo number_format($profitMargin, 1); ?>% margin
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="betting-chart">
                <div class="chart-header">
                    <h3>Daily Betting Activity</h3>
                </div>
                <div class="chart-container">
                    <canvas id="bettingActivityChart" width="1000" height="400"></canvas>
                </div>
            </div>
        </section>
        <?php endif; ?>
        
        <!-- Users Tab -->
        <?php if ($tab === 'users'): ?>
        <section class="report-section">
            <div class="user-summary">
                <div class="summary-cards">
                    <div class="summary-card">
                        <h4>Total Users</h4>
                        <div class="amount"><?php echo number_format($userStats['total_users'] ?? 0); ?></div>
                        <div class="count">Registered</div>
                    </div>
                    
                    <div class="summary-card">
                        <h4>Active Users</h4>
                        <div class="amount"><?php echo number_format($userStats['active_users'] ?? 0); ?></div>
                        <div class="count">Verified accounts</div>
                    </div>
                    
                    <div class="summary-card">
                        <h4>Daily Active</h4>
                        <div class="amount"><?php echo number_format($userStats['active_24h'] ?? 0); ?></div>
                        <div class="count">Last 24 hours</div>
                    </div>
                    
                    <div class="summary-card">
                        <h4>Monthly Active</h4>
                        <div class="amount"><?php echo number_format($userStats['active_30d'] ?? 0); ?></div>
                        <div class="count">Last 30 days</div>
                    </div>
                </div>
            </div>
            
            <!-- Top Players Table -->
            <div class="top-players">
                <div class="table-header">
                    <h3>Top Players (by Wagered Amount)</h3>
                </div>
                <div class="table-container">
                    <table class="report-table">
                        <thead>
                            <tr>
                                <th>Rank</th>
                                <th>Player</th>
                                <th>Total Bets</th>
                                <th>Total Wagered</th>
                                <th>Total Winnings</th>
                                <th>Net P&L</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($topPlayers as $index => $player): ?>
                            <tr>
                                <td><?php echo $index + 1; ?></td>
                                <td>
                                    <div class="player-info">
                                        <div class="username"><?php echo htmlspecialchars($player['username']); ?></div>
                                        <div class="email"><?php echo htmlspecialchars($player['email']); ?></div>
                                    </div>
                                </td>
                                <td><?php echo number_format($player['total_bets']); ?></td>
                                <td>$<?php echo number_format($player['total_wagered'], 2); ?></td>
                                <td>$<?php echo number_format($player['total_winnings'], 2); ?></td>
                                <td class="<?php echo $player['net_loss'] >= 0 ? 'positive' : 'negative'; ?>">
                                    $<?php echo number_format($player['net_loss'], 2); ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </section>
        <?php endif; ?>
        
        <!-- Games Tab -->
        <?php if ($tab === 'games'): ?>
        <section class="report-section">
            <div class="games-performance">
                <div class="table-header">
                    <h3>Game Performance</h3>
                </div>
                <div class="table-container">
                    <table class="report-table">
                        <thead>
                            <tr>
                                <th>Game Type</th>
                                <th>Total Bets</th>
                                <th>Total Wagered</th>
                                <th>Total Payouts</th>
                                <th>House Profit</th>
                                <th>Profit Margin</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($gameStats as $game): ?>
                            <tr>
                                <td>
                                    <div class="game-type">
                                        <i class="fas fa-dice"></i>
                                        <?php echo ucfirst($game['game_type']); ?>
                                    </div>
                                </td>
                                <td><?php echo number_format($game['bet_count']); ?></td>
                                <td>$<?php echo number_format($game['total_wagered'], 2); ?></td>
                                <td>$<?php echo number_format($game['total_payouts'], 2); ?></td>
                                <td class="positive">$<?php echo number_format($game['profit'], 2); ?></td>
                                <td>
                                    <?php 
                                    $margin = $game['total_wagered'] > 0 ? 
                                        ($game['profit'] / $game['total_wagered']) * 100 : 0;
                                    echo number_format($margin, 1); ?>%
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </section>
        <?php endif; ?>
    </main>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
// Initialize charts
document.addEventListener('DOMContentLoaded', function() {
    initializeCharts();
});

function initializeCharts() {
    // Revenue Chart
    if (document.getElementById('revenueChart')) {
        const revenueData = <?php echo json_encode($dailyRevenue); ?>;
        
        const ctx = document.getElementById('revenueChart').getContext('2d');
        new Chart(ctx, {
            type: 'line',
            data: {
                labels: revenueData.map(item => item.date),
                datasets: [{
                    label: 'Deposits',
                    data: revenueData.map(item => parseFloat(item.deposits)),
                    borderColor: '#00ff88',
                    backgroundColor: 'rgba(0, 255, 136, 0.1)',
                    tension: 0.4
                }, {
                    label: 'Withdrawals',
                    data: revenueData.map(item => parseFloat(item.withdrawals)),
                    borderColor: '#ff4757',
                    backgroundColor: 'rgba(255, 71, 87, 0.1)',
                    tension: 0.4
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return '$' + value.toLocaleString();
                            }
                        }
                    }
                }
            }
        });
    }
    
    // Betting Chart
    if (document.getElementById('bettingChart')) {
        const bettingData = <?php echo json_encode($dailyBetting); ?>;
        
        const ctx = document.getElementById('bettingChart').getContext('2d');
        new Chart(ctx, {
            type: 'bar',
            data: {
                labels: bettingData.map(item => item.date),
                datasets: [{
                    label: 'Bets Count',
                    data: bettingData.map(item => parseInt(item.bet_count)),
                    backgroundColor: 'rgba(54, 162, 235, 0.8)',
                    yAxisID: 'y'
                }, {
                    label: 'Total Wagered',
                    data: bettingData.map(item => parseFloat(item.total_wagered)),
                    backgroundColor: 'rgba(255, 206, 86, 0.8)',
                    yAxisID: 'y1'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        type: 'linear',
                        display: true,
                        position: 'left',
                    },
                    y1: {
                        type: 'linear',
                        display: true,
                        position: 'right',
                        grid: {
                            drawOnChartArea: false,
                        },
                        ticks: {
                            callback: function(value) {
                                return '$' + value.toLocaleString();
                            }
                        }
                    }
                }
            }
        });
    }
}

function setPeriod(period) {
    const today = new Date();
    let dateFrom, dateTo;
    
    switch(period) {
        case 'today':
            dateFrom = dateTo = today.toISOString().split('T')[0];
            break;
        case '7days':
            dateFrom = new Date(today.getTime() - 7 * 24 * 60 * 60 * 1000).toISOString().split('T')[0];
            dateTo = today.toISOString().split('T')[0];
            break;
        case '30days':
            dateFrom = new Date(today.getTime() - 30 * 24 * 60 * 60 * 1000).toISOString().split('T')[0];
            dateTo = today.toISOString().split('T')[0];
            break;
        case '90days':
            dateFrom = new Date(today.getTime() - 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0];
            dateTo = today.toISOString().split('T')[0];
            break;
        case 'custom':
            return; // Don't change dates for custom
    }
    
    if (period !== 'custom') {
        document.getElementById('dateFrom').value = dateFrom;
        document.getElementById('dateTo').value = dateTo;
        updateDateRange();
    }
}

function updateDateRange() {
    const dateFrom = document.getElementById('dateFrom').value;
    const dateTo = document.getElementById('dateTo').value;
    const currentTab = '<?php echo $tab; ?>';
    
    window.location.href = `?tab=${currentTab}&period=custom&date_from=${dateFrom}&date_to=${dateTo}`;
}

function exportReport() {
    const dateFrom = document.getElementById('dateFrom').value;
    const dateTo = document.getElementById('dateTo').value;
    const currentTab = '<?php echo $tab; ?>';
    
    window.open(`/api/admin/reports/export?tab=${currentTab}&date_from=${dateFrom}&date_to=${dateTo}`, '_blank');
}
</script>

<?php include '../includes/footer.php'; ?>