    <?php
// pages/wallet.php - Wallet & Payment Management
// کیف پول و مدیریت پرداخت‌ها

require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

// Require authentication
requireAuth();

// Page settings
$pageTitle = 'Wallet - Deposits & Withdrawals';
$bodyClass = 'wallet-page';
$breadcrumbs = [
    ['title' => 'Wallet']
];

// Get current action
$currentAction = $_GET['action'] ?? 'overview';

// Get user's transaction history
$transactions = fetchAll(
    "SELECT * FROM transactions 
     WHERE user_id = ? 
     ORDER BY created_at DESC 
     LIMIT 50",
    [$currentUser['id']]
);

// Get payment methods
$paymentMethods = [
    'crypto' => [
        'bitcoin' => [
            'name' => 'Bitcoin',
            'symbol' => 'BTC',
            'icon' => 'fab fa-bitcoin',
            'min_deposit' => 0.001,
            'max_deposit' => 10,
            'min_withdraw' => 0.002,
            'max_withdraw' => 5,
            'fee' => 0.0005,
            'processing_time' => '10-30 minutes',
            'address' => 'bc1qxy2kgdygjrsqtzq2n0yrf2493p83kkfjhx0wlh'
        ],
        'ethereum' => [
            'name' => 'Ethereum',
            'symbol' => 'ETH',
            'icon' => 'fab fa-ethereum',
            'min_deposit' => 0.01,
            'max_deposit' => 100,
            'min_withdraw' => 0.02,
            'max_withdraw' => 50,
            'fee' => 0.005,
            'processing_time' => '5-15 minutes',
            'address' => '0x742d35Cc6634C0532925a3b8D4C9db96590b5c8e'
        ],
        'usdt' => [
            'name' => 'Tether USDT',
            'symbol' => 'USDT',
            'icon' => 'fas fa-dollar-sign',
            'min_deposit' => 10,
            'max_deposit' => 10000,
            'min_withdraw' => 20,
            'max_withdraw' => 5000,
            'fee' => 1,
            'processing_time' => '5-10 minutes',
            'address' => 'TQrZ8tKfjkdZs4ftzHBTHSUjrxLqd6OVEC'
        ]
    ],
    'traditional' => [
        'visa' => [
            'name' => 'Visa Card',
            'icon' => 'fab fa-cc-visa',
            'min_deposit' => 20,
            'max_deposit' => 5000,
            'fee_percentage' => 3.5,
            'processing_time' => 'Instant'
        ],
        'mastercard' => [
            'name' => 'Mastercard',
            'icon' => 'fab fa-cc-mastercard',
            'min_deposit' => 20,
            'max_deposit' => 5000,
            'fee_percentage' => 3.5,
            'processing_time' => 'Instant'
        ]
    ]
];

// Get user's pending transactions
$pendingTransactions = fetchAll(
    "SELECT * FROM transactions 
     WHERE user_id = ? AND status = 'pending' 
     ORDER BY created_at DESC",
    [$currentUser['id']]
);

$additionalScripts = [
    '/assets/js/wallet.js',
    '/assets/js/qrcode.js'
];

include '../includes/header.php';
?>

<!-- Wallet Header -->
<section class="wallet-header">
    <div class="container">
        <div class="header-content">
            <div class="wallet-info">
                <h1 class="page-title">
                    <i class="fas fa-wallet"></i>
                    Wallet
                </h1>
                <p class="page-description">
                    Manage your deposits, withdrawals, and transaction history securely.
                </p>
            </div>
            
            <div class="wallet-balance">
                <div class="balance-card main-balance">
                    <div class="balance-header">
                        <h3>Main Balance</h3>
                        <button class="refresh-btn" onclick="refreshBalance()">
                            <i class="fas fa-sync-alt"></i>
                        </button>
                    </div>
                    <div class="balance-amount">
                        <span class="currency">$</span>
                        <span class="amount" id="mainBalance"><?php echo number_format($currentUser['balance'], 2); ?></span>
                    </div>
                    <div class="balance-actions">
                        <button class="btn btn-primary" onclick="showDepositModal()">
                            <i class="fas fa-plus"></i>
                            Deposit
                        </button>
                        <button class="btn btn-outline" onclick="showWithdrawModal()">
                            <i class="fas fa-minus"></i>
                            Withdraw
                        </button>
                    </div>
                </div>
                
                <?php if ($currentUser['bonus_balance'] > 0): ?>
                <div class="balance-card bonus-balance">
                    <div class="balance-header">
                        <h3>Bonus Balance</h3>
                        <i class="fas fa-info-circle" title="Bonus funds with wagering requirements"></i>
                    </div>
                    <div class="balance-amount">
                        <span class="currency">$</span>
                        <span class="amount"><?php echo number_format($currentUser['bonus_balance'], 2); ?></span>
                    </div>
                    <div class="bonus-info">
                        <span class="wagering-info">Wagering required</span>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</section>

<!-- Wallet Navigation -->
<section class="wallet-nav">
    <div class="container">
        <div class="nav-tabs">
            <button class="tab-btn <?php echo $currentAction === 'overview' ? 'active' : ''; ?>" 
                    onclick="switchTab('overview')">
                <i class="fas fa-chart-pie"></i>
                <span>Overview</span>
            </button>
            <button class="tab-btn <?php echo $currentAction === 'deposit' ? 'active' : ''; ?>" 
                    onclick="switchTab('deposit')">
                <i class="fas fa-plus"></i>
                <span>Deposit</span>
            </button>
            <button class="tab-btn <?php echo $currentAction === 'withdraw' ? 'active' : ''; ?>" 
                    onclick="switchTab('withdraw')">
                <i class="fas fa-minus"></i>
                <span>Withdraw</span>
            </button>
            <button class="tab-btn <?php echo $currentAction === 'history' ? 'active' : ''; ?>" 
                    onclick="switchTab('history')">
                <i class="fas fa-history"></i>
                <span>History</span>
            </button>
        </div>
    </div>
</section>

<!-- Pending Transactions Alert -->
<?php if (!empty($pendingTransactions)): ?>
<section class="pending-transactions">
    <div class="container">
        <div class="alert alert-info">
            <div class="alert-icon">
                <i class="fas fa-clock"></i>
            </div>
            <div class="alert-content">
                <h4>Pending Transactions</h4>
                <p>You have <?php echo count($pendingTransactions); ?> pending transaction(s). Processing times may vary.</p>
            </div>
            <button class="alert-close" onclick="this.parentElement.parentElement.style.display='none'">
                <i class="fas fa-times"></i>
            </button>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- Wallet Content -->
<section class="wallet-content">
    <div class="container">
        <!-- Overview Tab -->
        <div class="tab-content <?php echo $currentAction === 'overview' ? 'active' : ''; ?>" id="overviewTab">
            <div class="overview-grid">
                <!-- Quick Stats -->
                <div class="stats-widget">
                    <h3 class="widget-title">
                        <i class="fas fa-chart-bar"></i>
                        Quick Stats
                    </h3>
                    <div class="stats-grid">
                        <div class="stat-item">
                            <div class="stat-value">
                                $<?php echo number_format(array_sum(array_column(array_filter($transactions, fn($t) => $t['type'] === 'deposit' && $t['status'] === 'completed'), 'amount')), 2); ?>
                            </div>
                            <div class="stat-label">Total Deposited</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-value">
                                $<?php echo number_format(array_sum(array_column(array_filter($transactions, fn($t) => $t['type'] === 'withdrawal' && $t['status'] === 'completed'), 'amount')), 2); ?>
                            </div>
                            <div class="stat-label">Total Withdrawn</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-value">
                                <?php echo count(array_filter($transactions, fn($t) => $t['type'] === 'deposit')); ?>
                            </div>
                            <div class="stat-label">Deposits</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-value">
                                <?php echo count(array_filter($transactions, fn($t) => $t['type'] === 'withdrawal')); ?>
                            </div>
                            <div class="stat-label">Withdrawals</div>
                        </div>
                    </div>
                </div>
                
                <!-- Recent Transactions -->
                <div class="recent-transactions-widget">
                    <div class="widget-header">
                        <h3 class="widget-title">
                            <i class="fas fa-history"></i>
                            Recent Transactions
                        </h3>
                        <a href="?action=history" class="view-all-link">View All</a>
                    </div>
                    <div class="transactions-list">
                        <?php if (empty($transactions)): ?>
                        <div class="empty-state">
                            <i class="fas fa-receipt"></i>
                            <p>No transactions yet</p>
                            <button class="btn btn-primary btn-sm" onclick="showDepositModal()">
                                Make Your First Deposit
                            </button>
                        </div>
                        <?php else: ?>
                        <?php foreach (array_slice($transactions, 0, 5) as $transaction): ?>
                        <div class="transaction-item">
                            <div class="transaction-icon">
                                <i class="fas fa-<?php echo $transaction['type'] === 'deposit' ? 'plus' : 'minus'; ?>"></i>
                            </div>
                            <div class="transaction-info">
                                <div class="transaction-type"><?php echo ucfirst($transaction['type']); ?></div>
                                <div class="transaction-method"><?php echo $transaction['payment_method']; ?></div>
                                <div class="transaction-time"><?php echo timeAgo($transaction['created_at']); ?></div>
                            </div>
                            <div class="transaction-amount">
                                <div class="amount <?php echo $transaction['type']; ?>">
                                    <?php echo $transaction['type'] === 'deposit' ? '+' : '-'; ?>$<?php echo number_format($transaction['amount'], 2); ?>
                                </div>
                                <div class="status status-<?php echo $transaction['status']; ?>">
                                    <?php echo ucfirst($transaction['status']); ?>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Deposit Tab -->
        <div class="tab-content <?php echo $currentAction === 'deposit' ? 'active' : ''; ?>" id="depositTab">
            <div class="deposit-content">
                <div class="payment-methods">
                    <h3 class="section-title">
                        <i class="fas fa-credit-card"></i>
                        Choose Payment Method
                    </h3>
                    
                    <!-- Crypto Methods -->
                    <div class="method-category">
                        <h4 class="category-title">
                            <i class="fab fa-bitcoin"></i>
                            Cryptocurrency (Recommended)
                        </h4>
                        <div class="methods-grid">
                            <?php foreach ($paymentMethods['crypto'] as $key => $method): ?>
                            <div class="payment-method crypto" data-method="<?php echo $key; ?>">
                                <div class="method-icon">
                                    <i class="<?php echo $method['icon']; ?>"></i>
                                </div>
                                <div class="method-info">
                                    <h5 class="method-name"><?php echo $method['name']; ?></h5>
                                    <div class="method-details">
                                        <span class="processing-time"><?php echo $method['processing_time']; ?></span>
                                        <span class="fee">Fee: <?php echo $method['fee']; ?> <?php echo $method['symbol']; ?></span>
                                    </div>
                                </div>
                                <div class="method-limits">
                                    <div class="limit">Min: <?php echo $method['min_deposit']; ?> <?php echo $method['symbol']; ?></div>
                                    <div class="limit">Max: <?php echo $method['max_deposit']; ?> <?php echo $method['symbol']; ?></div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    
                    <!-- Traditional Methods -->
                    <div class="method-category">
                        <h4 class="category-title">
                            <i class="fas fa-credit-card"></i>
                            Credit/Debit Cards
                        </h4>
                        <div class="methods-grid">
                            <?php foreach ($paymentMethods['traditional'] as $key => $method): ?>
                            <div class="payment-method traditional" data-method="<?php echo $key; ?>">
                                <div class="method-icon">
                                    <i class="<?php echo $method['icon']; ?>"></i>
                                </div>
                                <div class="method-info">
                                    <h5 class="method-name"><?php echo $method['name']; ?></h5>
                                    <div class="method-details">
                                        <span class="processing-time"><?php echo $method['processing_time']; ?></span>
                                        <span class="fee">Fee: <?php echo $method['fee_percentage']; ?>%</span>
                                    </div>
                                </div>
                                <div class="method-limits">
                                    <div class="limit">Min: $<?php echo $method['min_deposit']; ?></div>
                                    <div class="limit">Max: $<?php echo $method['max_deposit']; ?></div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Withdraw Tab -->
        <div class="tab-content <?php echo $currentAction === 'withdraw' ? 'active' : ''; ?>" id="withdrawTab">
            <div class="withdraw-content">
                <div class="withdraw-info">
                    <div class="info-card">
                        <h3>Withdrawal Information</h3>
                        <ul class="info-list">
                            <li><i class="fas fa-check"></i> Minimum withdrawal: $20</li>
                            <li><i class="fas fa-check"></i> Processing time: 1-24 hours</li>
                            <li><i class="fas fa-check"></i> No fees for crypto withdrawals</li>
                            <li><i class="fas fa-check"></i> KYC verification required</li>
                        </ul>
                    </div>
                </div>
                
                <div class="withdraw-form-container">
                    <h3 class="section-title">
                        <i class="fas fa-money-bill-wave"></i>
                        Withdraw Funds
                    </h3>
                    
                    <form class="withdraw-form" id="withdrawForm">
                        <div class="form-group">
                            <label for="withdrawMethod">Withdrawal Method</label>
                            <select id="withdrawMethod" name="method" required>
                                <option value="">Select method...</option>
                                <?php foreach ($paymentMethods['crypto'] as $key => $method): ?>
                                <option value="<?php echo $key; ?>"><?php echo $method['name']; ?> (<?php echo $method['symbol']; ?>)</option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="withdrawAmount">Amount (USD)</label>
                            <div class="amount-input">
                                <span class="currency-symbol">$</span>
                                <input type="number" 
                                       id="withdrawAmount" 
                                       name="amount" 
                                       min="20" 
                                       max="<?php echo $currentUser['balance']; ?>" 
                                       step="0.01" 
                                       placeholder="0.00" 
                                       required>
                                <button type="button" class="max-btn" onclick="setMaxAmount()">MAX</button>
                            </div>
                            <div class="amount-info">
                                <span>Available: $<?php echo number_format($currentUser['balance'], 2); ?></span>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="withdrawAddress">Wallet Address</label>
                            <input type="text" 
                                   id="withdrawAddress" 
                                   name="address" 
                                   placeholder="Enter your wallet address" 
                                   required>
                            <div class="address-help">
                                <i class="fas fa-info-circle"></i>
                                <span>Make sure the address is correct. Transactions cannot be reversed.</span>
                            </div>
                        </div>
                        
                        <div class="withdrawal-summary" id="withdrawalSummary" style="display: none;">
                            <h4>Withdrawal Summary</h4>
                            <div class="summary-row">
                                <span>Amount:</span>
                                <span id="summaryAmount">$0.00</span>
                            </div>
                            <div class="summary-row">
                                <span>Fee:</span>
                                <span id="summaryFee">$0.00</span>
                            </div>
                            <div class="summary-row total">
                                <span>You will receive:</span>
                                <span id="summaryTotal">$0.00</span>
                            </div>
                        </div>
                        
                        <button type="submit" class="btn btn-primary btn-lg">
                            <i class="fas fa-paper-plane"></i>
                            Request Withdrawal
                        </button>
                    </form>
                </div>
            </div>
        </div>
        
        <!-- History Tab -->
        <div class="tab-content <?php echo $currentAction === 'history' ? 'active' : ''; ?>" id="historyTab">
            <div class="history-content">
                <div class="history-filters">
                    <div class="filter-group">
                        <label for="typeFilter">Type</label>
                        <select id="typeFilter" onchange="filterTransactions()">
                            <option value="">All Types</option>
                            <option value="deposit">Deposits</option>
                            <option value="withdrawal">Withdrawals</option>
                        </select>
                    </div>
                    <div class="filter-group">
                        <label for="statusFilter">Status</label>
                        <select id="statusFilter" onchange="filterTransactions()">
                            <option value="">All Status</option>
                            <option value="completed">Completed</option>
                            <option value="pending">Pending</option>
                            <option value="failed">Failed</option>
                        </select>
                    </div>
                    <div class="filter-group">
                        <label for="dateFilter">Date Range</label>
                        <select id="dateFilter" onchange="filterTransactions()">
                            <option value="">All Time</option>
                            <option value="today">Today</option>
                            <option value="week">This Week</option>
                            <option value="month">This Month</option>
                            <option value="year">This Year</option>
                        </select>
                    </div>
                </div>
                
                <div class="transactions-table">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Type</th>
                                <th>Method</th>
                                <th>Amount</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody id="transactionsTableBody">
                            <?php foreach ($transactions as $transaction): ?>
                            <tr class="transaction-row" 
                                data-type="<?php echo $transaction['type']; ?>" 
                                data-status="<?php echo $transaction['status']; ?>"
                                data-date="<?php echo date('Y-m-d', strtotime($transaction['created_at'])); ?>">
                                <td>
                                    <div class="date-info">
                                        <div class="date"><?php echo date('M j, Y', strtotime($transaction['created_at'])); ?></div>
                                        <div class="time"><?php echo date('H:i', strtotime($transaction['created_at'])); ?></div>
                                    </div>
                                </td>
                                <td>
                                    <div class="type-info">
                                        <i class="fas fa-<?php echo $transaction['type'] === 'deposit' ? 'plus' : 'minus'; ?>"></i>
                                        <span><?php echo ucfirst($transaction['type']); ?></span>
                                    </div>
                                </td>
                                <td><?php echo $transaction['payment_method']; ?></td>
                                <td>
                                    <div class="amount <?php echo $transaction['type']; ?>">
                                        <?php echo $transaction['type'] === 'deposit' ? '+' : '-'; ?>$<?php echo number_format($transaction['amount'], 2); ?>
                                    </div>
                                </td>
                                <td>
                                    <span class="status-badge status-<?php echo $transaction['status']; ?>">
                                        <?php echo ucfirst($transaction['status']); ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="action-buttons">
                                        <button class="btn btn-outline btn-xs" onclick="viewTransaction(<?php echo $transaction['id']; ?>)">
                                            <i class="fas fa-eye"></i>
                                            View
                                        </button>
                                        <?php if ($transaction['status'] === 'pending' && $transaction['type'] === 'withdrawal'): ?>
                                        <button class="btn btn-danger btn-xs" onclick="cancelTransaction(<?php echo $transaction['id']; ?>)">
                                            <i class="fas fa-times"></i>
                                            Cancel
                                        </button>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- Deposit Modal -->
<div class="modal-backdrop" id="depositModal">
    <div class="modal deposit-modal">
        <div class="modal-header">
            <h3 class="modal-title">Make a Deposit</h3>
            <button class="modal-close" onclick="closeModal('depositModal')">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="modal-body" id="depositModalBody">
            <!-- Deposit form will be loaded here -->
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    initializeWallet();
});

function initializeWallet() {
    // Initialize payment method selection
    const paymentMethods = document.querySelectorAll('.payment-method');
    paymentMethods.forEach(method => {
        method.addEventListener('click', function() {
            selectPaymentMethod(this.dataset.method);
        });
    });
    
    // Initialize withdraw form
    const withdrawForm = document.getElementById('withdrawForm');
    if (withdrawForm) {
        withdrawForm.addEventListener('submit', handleWithdrawal);
        
        // Update summary on amount change
        const amountInput = document.getElementById('withdrawAmount');
        if (amountInput) {
            amountInput.addEventListener('input', updateWithdrawalSummary);
        }
    }
}

function switchTab(tabName) {
    // Update URL
    const url = new URL(window.location);
    url.searchParams.set('action', tabName);
    window.history.pushState({}, '', url);
    
    // Update tab buttons
    document.querySelectorAll('.tab-btn').forEach(btn => btn.classList.remove('active'));
    document.querySelector(`[onclick="switchTab('${tabName}')"]`).classList.add('active');
    
    // Update tab content
    document.querySelectorAll('.tab-content').forEach(content => content.classList.remove('active'));
    document.getElementById(tabName + 'Tab').classList.add('active');
}

function selectPaymentMethod(method) {
    // Remove previous selection
    document.querySelectorAll('.payment-method').forEach(m => m.classList.remove('selected'));
    
    // Select current method
    document.querySelector(`[data-method="${method}"]`).classList.add('selected');
    
    // Show deposit form for selected method
    showDepositForm(method);
}

function showDepositForm(method) {
    // This would show the appropriate deposit form
    console.log('Showing deposit form for:', method);
}

function showDepositModal() {
    openModal('depositModal');
    // Load deposit form content
}

function showWithdrawModal() {
    switchTab('withdraw');
}

function handleWithdrawal(e) {
    e.preventDefault();
    
    const formData = new FormData(e.target);
    const withdrawalData = {
        method: formData.get('method'),
        amount: parseFloat(formData.get('amount')),
        address: formData.get('address')
    };
    
    // Show loading
    Crashify.ui.showLoading();
    
    // Submit withdrawal request
    Crashify.api.request('/api/wallet/withdraw', {
        method: 'POST',
        body: JSON.stringify(withdrawalData)
    })
    .then(response => {
        if (response.success) {
            Crashify.ui.toast.success('Withdrawal request submitted successfully!');
            
            // Reset form and refresh page
            e.target.reset();
            setTimeout(() => {
                window.location.reload();
            }, 2000);
        } else {
            throw new Error(response.message);
        }
    })
    .catch(error => {
        Crashify.ui.toast.error(error.message || 'Failed to submit withdrawal request');
    })
    .finally(() => {
        Crashify.ui.hideLoading();
    });
}

function updateWithdrawalSummary() {
    const amount = parseFloat(document.getElementById('withdrawAmount').value) || 0;
    const method = document.getElementById('withdrawMethod').value;
    
    if (amount > 0 && method) {
        // Calculate fee based on method
        const fee = calculateWithdrawalFee(method, amount);
        const total = amount - fee;
        
        // Update summary
        document.getElementById('summaryAmount').textContent = `$${amount.toFixed(2)}`;
        document.getElementById('summaryFee').textContent = `$${fee.toFixed(2)}`;
        document.getElementById('summaryTotal').textContent = `$${total.toFixed(2)}`;
        
        // Show summary
        document.getElementById('withdrawalSummary').style.display = 'block';
    } else {
        document.getElementById('withdrawalSummary').style.display = 'none';
    }
}

function calculateWithdrawalFee(method, amount) {
    // This would calculate the actual fee based on the method
    return 0; // Most crypto withdrawals are free
}

function setMaxAmount() {
    const maxAmount = <?php echo $currentUser['balance']; ?>;
    document.getElementById('withdrawAmount').value = maxAmount.toFixed(2);
    updateWithdrawalSummary();
}

function refreshBalance() {
    Crashify.api.request('/api/user/balance')
        .then(response => {
            if (response.success) {
                document.getElementById('mainBalance').textContent = response.balance.toFixed(2);
                document.getElementById('headerBalance').textContent = `$${response.balance.toFixed(2)}`;
            }
        })
        .catch(error => {
            console.error('Failed to refresh balance:', error);
        });
}

function filterTransactions() {
    const typeFilter = document.getElementById('typeFilter').value;
    const statusFilter = document.getElementById('statusFilter').value;
    const dateFilter = document.getElementById('dateFilter').value;
    
    const rows = document.querySelectorAll('.transaction-row');
    
    rows.forEach(row => {
        let show = true;
        
        if (typeFilter && row.dataset.type !== typeFilter) show = false;
        if (statusFilter && row.dataset.status !== statusFilter) show = false;
        if (dateFilter && !matchesDateFilter(row.dataset.date, dateFilter)) show = false;
        
        row.style.display = show ? '' : 'none';
    });
}

function matchesDateFilter(date, filter) {
    const transactionDate = new Date(date);
    const now = new Date();
    
    switch (filter) {
        case 'today':
            return transactionDate.toDateString() === now.toDateString();
        case 'week':
            const weekAgo = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000);
            return transactionDate >= weekAgo;
        case 'month':
            return transactionDate.getMonth() === now.getMonth() && 
                   transactionDate.getFullYear() === now.getFullYear();
        case 'year':
            return transactionDate.getFullYear() === now.getFullYear();
        default:
            return true;
    }
}

function viewTransaction(transactionId) {
    // Show transaction details modal
    Crashify.api.request(`/api/transactions/${transactionId}`)
        .then(response => {
            if (response.success) {
                // Show transaction details
                showTransactionDetails(response.transaction);
            }
        });
}

function cancelTransaction(transactionId) {
    if (confirm('Are you sure you want to cancel this withdrawal?')) {
        Crashify.api.request(`/api/transactions/${transactionId}/cancel`, {
            method: 'POST'
        })
        .then(response => {
            if (response.success) {
                Crashify.ui.toast.success('Transaction cancelled successfully');
                window.location.reload();
            }
        });
    }
}
</script>

<?php include '../includes/footer.php'; ?>