<?php
// admin/transactions.php - Transaction Management
// مدیریت تراکنش‌ها

require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

// Require admin authentication
requireAuth();
requireAdmin();

// Page settings
$pageTitle = 'Transaction Management - Admin';
$bodyClass = 'admin-transactions';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $transactionId = intval($_POST['transaction_id'] ?? 0);
    
    switch ($action) {
        case 'approve':
            execute("UPDATE transactions SET status = 'completed', processed_at = NOW(), processed_by = ? WHERE id = ? AND status = 'pending'", [$currentUser['id'], $transactionId]);
            
            // Update user balance for deposits
            $transaction = fetchRow("SELECT * FROM transactions WHERE id = ?", [$transactionId]);
            if ($transaction && $transaction['type'] === 'deposit') {
                execute("UPDATE users SET balance = balance + ? WHERE id = ?", [$transaction['amount'], $transaction['user_id']]);
            }
            
            $message = "Transaction approved successfully";
            break;
            
        case 'reject':
            $reason = $_POST['reason'] ?? 'Rejected by admin';
            execute("UPDATE transactions SET status = 'failed', processed_at = NOW(), processed_by = ?, notes = ? WHERE id = ? AND status = 'pending'", [$currentUser['id'], $reason, $transactionId]);
            $message = "Transaction rejected successfully";
            break;
            
        case 'add_note':
            $note = $_POST['note'] ?? '';
            if ($note) {
                execute("UPDATE transactions SET notes = ? WHERE id = ?", [$note, $transactionId]);
                $message = "Note added successfully";
            }
            break;
    }
}

// Get filter parameters
$search = $_GET['search'] ?? '';
$type = $_GET['type'] ?? 'all';
$status = $_GET['status'] ?? 'all';
$dateFrom = $_GET['date_from'] ?? '';
$dateTo = $_GET['date_to'] ?? '';
$sortBy = $_GET['sort'] ?? 'created_at';
$sortOrder = $_GET['order'] ?? 'DESC';
$page = max(1, intval($_GET['page'] ?? 1));
$limit = 25;
$offset = ($page - 1) * $limit;

// Build WHERE clause
$whereConditions = [];
$params = [];

if ($search) {
    $whereConditions[] = "(u.username LIKE ? OR u.email LIKE ? OR t.transaction_id LIKE ?)";
    $searchTerm = "%$search%";
    $params = array_merge($params, [$searchTerm, $searchTerm, $searchTerm]);
}

if ($type !== 'all') {
    $whereConditions[] = "t.type = ?";
    $params[] = $type;
}

if ($status !== 'all') {
    $whereConditions[] = "t.status = ?";
    $params[] = $status;
}

if ($dateFrom) {
    $whereConditions[] = "DATE(t.created_at) >= ?";
    $params[] = $dateFrom;
}

if ($dateTo) {
    $whereConditions[] = "DATE(t.created_at) <= ?";
    $params[] = $dateTo;
}

$whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';

// Validate sort parameters
$allowedSorts = ['id', 'amount', 'created_at', 'processed_at', 'type', 'status'];
$sortBy = in_array($sortBy, $allowedSorts) ? "t.$sortBy" : 't.created_at';
$sortOrder = strtoupper($sortOrder) === 'ASC' ? 'ASC' : 'DESC';

// Get total count
$totalTransactions = fetchValue("SELECT COUNT(*) FROM transactions t JOIN users u ON t.user_id = u.id $whereClause", $params);

// Get transactions
$transactions = fetchAll(
    "SELECT t.*, u.username, u.email,
        (SELECT username FROM users WHERE id = t.processed_by) as processed_by_username
     FROM transactions t 
     JOIN users u ON t.user_id = u.id 
     $whereClause 
     ORDER BY $sortBy $sortOrder 
     LIMIT $limit OFFSET $offset",
    $params
);

// Calculate pagination
$totalPages = ceil($totalTransactions / $limit);

// Get transaction statistics
$transactionStats = fetchRow(
    "SELECT 
        COUNT(*) as total_transactions,
        COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_transactions,
        COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed_transactions,
        COUNT(CASE WHEN status = 'failed' THEN 1 END) as failed_transactions,
        SUM(CASE WHEN type = 'deposit' AND status = 'completed' THEN amount ELSE 0 END) as total_deposits,
        SUM(CASE WHEN type = 'withdrawal' AND status = 'completed' THEN amount ELSE 0 END) as total_withdrawals,
        COUNT(CASE WHEN DATE(created_at) = CURDATE() THEN 1 END) as today_transactions,
        SUM(CASE WHEN DATE(created_at) = CURDATE() AND status = 'completed' THEN amount ELSE 0 END) as today_volume
     FROM transactions"
);

include '../includes/header.php';
?>

<!-- Admin Transaction Management -->
<div class="admin-container">
    <?php include 'sidebar.php'; ?>
    
    <main class="admin-main">
        <!-- Header -->
        <header class="admin-header">
            <div class="header-left">
                <h1 class="page-title">Transaction Management</h1>
                <p class="page-subtitle">Monitor and manage all financial transactions</p>
            </div>
            
            <div class="header-right">
                <button class="btn btn-outline" onclick="exportTransactions()">
                    <i class="fas fa-download"></i>
                    Export CSV
                </button>
                <button class="btn btn-primary" onclick="refreshTransactions()">
                    <i class="fas fa-sync-alt"></i>
                    Refresh
                </button>
            </div>
        </header>
        
        <!-- Transaction Statistics -->
        <section class="stats-section">
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-exchange-alt"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format($transactionStats['total_transactions']); ?></div>
                        <div class="stat-label">Total Transactions</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon warning">
                        <i class="fas fa-clock"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format($transactionStats['pending_transactions']); ?></div>
                        <div class="stat-label">Pending</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon success">
                        <i class="fas fa-arrow-down"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value">$<?php echo number_format($transactionStats['total_deposits'], 2); ?></div>
                        <div class="stat-label">Total Deposits</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon danger">
                        <i class="fas fa-arrow-up"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value">$<?php echo number_format($transactionStats['total_withdrawals'], 2); ?></div>
                        <div class="stat-label">Total Withdrawals</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-calendar-day"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format($transactionStats['today_transactions']); ?></div>
                        <div class="stat-label">Today's Transactions</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-dollar-sign"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value">$<?php echo number_format($transactionStats['today_volume'], 2); ?></div>
                        <div class="stat-label">Today's Volume</div>
                    </div>
                </div>
            </div>
        </section>
        
        <!-- Filters and Search -->
        <section class="filters-section">
            <form class="filters-form" method="GET">
                <div class="filter-group">
                    <div class="search-box">
                        <i class="fas fa-search"></i>
                        <input type="text" 
                               name="search" 
                               value="<?php echo htmlspecialchars($search); ?>" 
                               placeholder="Search transactions..."
                               class="search-input">
                    </div>
                </div>
                
                <div class="filter-group">
                    <select name="type" onchange="this.form.submit()">
                        <option value="all" <?php echo $type === 'all' ? 'selected' : ''; ?>>All Types</option>
                        <option value="deposit" <?php echo $type === 'deposit' ? 'selected' : ''; ?>>Deposits</option>
                        <option value="withdrawal" <?php echo $type === 'withdrawal' ? 'selected' : ''; ?>>Withdrawals</option>
                        <option value="bet" <?php echo $type === 'bet' ? 'selected' : ''; ?>>Bets</option>
                        <option value="win" <?php echo $type === 'win' ? 'selected' : ''; ?>>Wins</option>
                        <option value="bonus" <?php echo $type === 'bonus' ? 'selected' : ''; ?>>Bonuses</option>
                    </select>
                </div>
                
                <div class="filter-group">
                    <select name="status" onchange="this.form.submit()">
                        <option value="all" <?php echo $status === 'all' ? 'selected' : ''; ?>>All Status</option>
                        <option value="pending" <?php echo $status === 'pending' ? 'selected' : ''; ?>>Pending</option>
                        <option value="completed" <?php echo $status === 'completed' ? 'selected' : ''; ?>>Completed</option>
                        <option value="failed" <?php echo $status === 'failed' ? 'selected' : ''; ?>>Failed</option>
                        <option value="cancelled" <?php echo $status === 'cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                    </select>
                </div>
                
                <div class="filter-group">
                    <input type="date" name="date_from" value="<?php echo $dateFrom; ?>" placeholder="From Date">
                </div>
                
                <div class="filter-group">
                    <input type="date" name="date_to" value="<?php echo $dateTo; ?>" placeholder="To Date">
                </div>
                
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-filter"></i>
                    Filter
                </button>
                
                <a href="/admin/transactions.php" class="btn btn-outline">
                    <i class="fas fa-times"></i>
                    Clear
                </a>
            </form>
            
            <div class="results-info">
                Showing <?php echo count($transactions); ?> of <?php echo number_format($totalTransactions); ?> transactions
            </div>
        </section>
        
        <!-- Transactions Table -->
        <section class="table-section">
            <div class="table-container">
                <table class="admin-table">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>User</th>
                            <th>Type</th>
                            <th>Amount</th>
                            <th>Status</th>
                            <th>Method</th>
                            <th>Created</th>
                            <th>Processed</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($transactions as $transaction): ?>
                        <tr class="transaction-row" data-transaction-id="<?php echo $transaction['id']; ?>">
                            <td>
                                <div class="transaction-id">
                                    <strong>#<?php echo $transaction['id']; ?></strong>
                                    <?php if ($transaction['transaction_id']): ?>
                                    <small><?php echo htmlspecialchars($transaction['transaction_id']); ?></small>
                                    <?php endif; ?>
                                </div>
                            </td>
                            
                            <td>
                                <div class="user-info">
                                    <div class="username"><?php echo htmlspecialchars($transaction['username']); ?></div>
                                    <div class="email"><?php echo htmlspecialchars($transaction['email']); ?></div>
                                </div>
                            </td>
                            
                            <td>
                                <span class="transaction-type type-<?php echo $transaction['type']; ?>">
                                    <i class="fas fa-<?php echo getTransactionIcon($transaction['type']); ?>"></i>
                                    <?php echo ucfirst($transaction['type']); ?>
                                </span>
                            </td>
                            
                            <td>
                                <div class="amount <?php echo in_array($transaction['type'], ['deposit', 'win', 'bonus']) ? 'positive' : 'negative'; ?>">
                                    <?php echo in_array($transaction['type'], ['deposit', 'win', 'bonus']) ? '+' : '-'; ?>$<?php echo number_format($transaction['amount'], 2); ?>
                                </div>
                            </td>
                            
                            <td>
                                <span class="status-badge status-<?php echo $transaction['status']; ?>">
                                    <?php echo ucfirst($transaction['status']); ?>
                                </span>
                            </td>
                            
                            <td>
                                <div class="payment-method">
                                    <?php if ($transaction['payment_method']): ?>
                                    <i class="fas fa-<?php echo getPaymentIcon($transaction['payment_method']); ?>"></i>
                                    <?php echo ucfirst($transaction['payment_method']); ?>
                                    <?php else: ?>
                                    -
                                    <?php endif; ?>
                                </div>
                            </td>
                            
                            <td>
                                <div class="date-info">
                                    <div class="date"><?php echo date('M j, Y', strtotime($transaction['created_at'])); ?></div>
                                    <div class="time"><?php echo date('H:i:s', strtotime($transaction['created_at'])); ?></div>
                                </div>
                            </td>
                            
                            <td>
                                <?php if ($transaction['processed_at']): ?>
                                <div class="date-info">
                                    <div class="date"><?php echo date('M j, Y', strtotime($transaction['processed_at'])); ?></div>
                                    <div class="time"><?php echo date('H:i:s', strtotime($transaction['processed_at'])); ?></div>
                                    <?php if ($transaction['processed_by_username']): ?>
                                    <small>by <?php echo htmlspecialchars($transaction['processed_by_username']); ?></small>
                                    <?php endif; ?>
                                </div>
                                <?php else: ?>
                                <span class="text-muted">-</span>
                                <?php endif; ?>
                            </td>
                            
                            <td>
                                <div class="action-buttons">
                                    <button class="btn btn-outline btn-xs" onclick="viewTransaction(<?php echo $transaction['id']; ?>)">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                    
                                    <?php if ($transaction['status'] === 'pending'): ?>
                                    <button class="btn btn-success btn-xs" onclick="approveTransaction(<?php echo $transaction['id']; ?>)">
                                        <i class="fas fa-check"></i>
                                    </button>
                                    <button class="btn btn-danger btn-xs" onclick="rejectTransaction(<?php echo $transaction['id']; ?>)">
                                        <i class="fas fa-times"></i>
                                    </button>
                                    <?php endif; ?>
                                    
                                    <div class="dropdown">
                                        <button class="btn btn-outline btn-xs dropdown-toggle">
                                            <i class="fas fa-ellipsis-v"></i>
                                        </button>
                                        <div class="dropdown-menu">
                                            <a href="#" onclick="addNote(<?php echo $transaction['id']; ?>)">Add Note</a>
                                            <a href="#" onclick="viewUserTransactions(<?php echo $transaction['user_id']; ?>)">User Transactions</a>
                                            <a href="#" onclick="downloadReceipt(<?php echo $transaction['id']; ?>)">Download Receipt</a>
                                        </div>
                                    </div>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            <?php if ($totalPages > 1): ?>
            <div class="pagination-container">
                <nav class="pagination">
                    <?php if ($page > 1): ?>
                    <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>" class="page-link prev">
                        <i class="fas fa-chevron-left"></i>
                        Previous
                    </a>
                    <?php endif; ?>
                    
                    <?php
                    $startPage = max(1, $page - 2);
                    $endPage = min($totalPages, $page + 2);
                    
                    for ($i = $startPage; $i <= $endPage; $i++): ?>
                    <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>" 
                       class="page-link <?php echo $i === $page ? 'active' : ''; ?>">
                        <?php echo $i; ?>
                    </a>
                    <?php endfor; ?>
                    
                    <?php if ($page < $totalPages): ?>
                    <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>" class="page-link next">
                        Next
                        <i class="fas fa-chevron-right"></i>
                    </a>
                    <?php endif; ?>
                </nav>
            </div>
            <?php endif; ?>
        </section>
    </main>
</div>

<!-- Transaction Details Modal -->
<div class="modal-backdrop" id="transactionModal">
    <div class="modal transaction-modal">
        <div class="modal-header">
            <h3 class="modal-title">Transaction Details</h3>
            <button class="modal-close" onclick="closeModal('transactionModal')">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="modal-body" id="transactionModalContent">
            <!-- Transaction details will be loaded here -->
        </div>
    </div>
</div>

<!-- Reject Transaction Modal -->
<div class="modal-backdrop" id="rejectModal">
    <div class="modal reject-modal">
        <div class="modal-header">
            <h3 class="modal-title">Reject Transaction</h3>
            <button class="modal-close" onclick="closeModal('rejectModal')">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="modal-body">
            <form id="rejectForm" method="POST">
                <input type="hidden" name="action" value="reject">
                <input type="hidden" name="transaction_id" id="rejectTransactionId">
                
                <div class="form-group">
                    <label for="rejectReason">Reason for Rejection</label>
                    <textarea name="reason" id="rejectReason" rows="4" required placeholder="Please provide a reason for rejecting this transaction..."></textarea>
                </div>
                
                <div class="modal-actions">
                    <button type="button" class="btn btn-outline" onclick="closeModal('rejectModal')">Cancel</button>
                    <button type="submit" class="btn btn-danger">Reject Transaction</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// Transaction management functions
function viewTransaction(transactionId) {
    fetch(`/api/admin/transactions/${transactionId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showTransactionDetails(data.transaction);
            }
        })
        .catch(error => {
            console.error('Error:', error);
        });
}

function approveTransaction(transactionId) {
    if (confirm('Are you sure you want to approve this transaction?')) {
        const formData = new FormData();
        formData.append('action', 'approve');
        formData.append('transaction_id', transactionId);
        
        fetch('/admin/transactions.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.text())
        .then(() => {
            window.location.reload();
        })
        .catch(error => {
            console.error('Error:', error);
        });
    }
}

function rejectTransaction(transactionId) {
    document.getElementById('rejectTransactionId').value = transactionId;
    openModal('rejectModal');
}

function addNote(transactionId) {
    const note = prompt('Add a note to this transaction:');
    if (note) {
        const formData = new FormData();
        formData.append('action', 'add_note');
        formData.append('transaction_id', transactionId);
        formData.append('note', note);
        
        fetch('/admin/transactions.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.text())
        .then(() => {
            window.location.reload();
        })
        .catch(error => {
            console.error('Error:', error);
        });
    }
}

function exportTransactions() {
    const params = new URLSearchParams(window.location.search);
    params.set('export', 'csv');
    window.open(`/api/admin/transactions/export?${params.toString()}`, '_blank');
}

function refreshTransactions() {
    window.location.reload();
}

function showTransactionDetails(transaction) {
    const content = `
        <div class="transaction-details">
            <div class="detail-section">
                <h4>Basic Information</h4>
                <div class="detail-grid">
                    <div class="detail-item">
                        <label>Transaction ID:</label>
                        <span>#${transaction.id}</span>
                    </div>
                    <div class="detail-item">
                        <label>External ID:</label>
                        <span>${transaction.transaction_id || '-'}</span>
                    </div>
                    <div class="detail-item">
                        <label>Type:</label>
                        <span class="transaction-type type-${transaction.type}">${transaction.type}</span>
                    </div>
                    <div class="detail-item">
                        <label>Amount:</label>
                        <span class="amount">$${parseFloat(transaction.amount).toFixed(2)}</span>
                    </div>
                    <div class="detail-item">
                        <label>Status:</label>
                        <span class="status-badge status-${transaction.status}">${transaction.status}</span>
                    </div>
                    <div class="detail-item">
                        <label>Payment Method:</label>
                        <span>${transaction.payment_method || '-'}</span>
                    </div>
                </div>
            </div>
            
            <div class="detail-section">
                <h4>User Information</h4>
                <div class="detail-grid">
                    <div class="detail-item">
                        <label>Username:</label>
                        <span>${transaction.username}</span>
                    </div>
                    <div class="detail-item">
                        <label>Email:</label>
                        <span>${transaction.email}</span>
                    </div>
                </div>
            </div>
            
            <div class="detail-section">
                <h4>Timestamps</h4>
                <div class="detail-grid">
                    <div class="detail-item">
                        <label>Created:</label>
                        <span>${new Date(transaction.created_at).toLocaleString()}</span>
                    </div>
                    <div class="detail-item">
                        <label>Processed:</label>
                        <span>${transaction.processed_at ? new Date(transaction.processed_at).toLocaleString() : '-'}</span>
                    </div>
                </div>
            </div>
            
            ${transaction.notes ? `
            <div class="detail-section">
                <h4>Notes</h4>
                <div class="notes-content">
                    ${transaction.notes}
                </div>
            </div>
            ` : ''}
        </div>
    `;
    
    document.getElementById('transactionModalContent').innerHTML = content;
    openModal('transactionModal');
}
</script>

<?php
// Helper functions
function getTransactionIcon($type) {
    $icons = [
        'deposit' => 'arrow-down',
        'withdrawal' => 'arrow-up',
        'bet' => 'dice',
        'win' => 'trophy',
        'bonus' => 'gift'
    ];
    return $icons[$type] ?? 'exchange-alt';
}

function getPaymentIcon($method) {
    $icons = [
        'credit_card' => 'credit-card',
        'bank_transfer' => 'university',
        'paypal' => 'paypal',
        'crypto' => 'bitcoin',
        'skrill' => 'wallet',
        'neteller' => 'wallet'
    ];
    return $icons[$method] ?? 'credit-card';
}

include '../includes/footer.php';
?>
