<?php
// pages/promotions.php - Promotions & Bonuses Page
// صفحه پروموشن‌ها و بونوس‌ها

require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

// Page settings
$pageTitle = 'Promotions & Bonuses - Exclusive Offers';
$bodyClass = 'promotions-page';
$breadcrumbs = [
    ['title' => 'Promotions']
];

// Get active promotions
$activePromotions = [
    [
        'id' => 1,
        'title' => 'Welcome Bonus',
        'subtitle' => 'Get 100% up to $500 + 200 Free Spins',
        'description' => 'Double your first deposit and enjoy 200 free spins on selected slots. Perfect way to start your Crashify journey!',
        'type' => 'welcome',
        'bonus_amount' => 500,
        'bonus_percentage' => 100,
        'free_spins' => 200,
        'min_deposit' => 20,
        'wagering_requirement' => 35,
        'valid_days' => 30,
        'image' => '/assets/images/promotions/welcome-bonus.jpg',
        'featured' => true,
        'active' => true,
        'terms' => [
            'Minimum deposit: $20',
            'Wagering requirement: 35x bonus amount',
            'Valid for 30 days from activation',
            'Free spins valid on selected slots only',
            'Maximum bet with bonus: $5 per spin'
        ]
    ],
    [
        'id' => 2,
        'title' => 'Sports Reload Bonus',
        'subtitle' => '50% up to $200 every Friday',
        'description' => 'Boost your weekend sports betting with our weekly reload bonus. Get 50% extra on your Friday deposits!',
        'type' => 'reload',
        'bonus_amount' => 200,
        'bonus_percentage' => 50,
        'min_deposit' => 30,
        'wagering_requirement' => 5,
        'valid_days' => 7,
        'image' => '/assets/images/promotions/sports-reload.jpg',
        'featured' => true,
        'active' => true,
        'recurring' => 'weekly',
        'terms' => [
            'Available every Friday',
            'Minimum deposit: $30',
            'Wagering requirement: 5x bonus amount',
            'Valid for sports betting only',
            'Minimum odds: 1.50'
        ]
    ],
    [
        'id' => 3,
        'title' => 'Cashback Monday',
        'subtitle' => 'Get 10% cashback on losses',
        'description' => 'Start your week right! Get 10% cashback on your net losses from the previous week, up to $100.',
        'type' => 'cashback',
        'cashback_percentage' => 10,
        'max_cashback' => 100,
        'image' => '/assets/images/promotions/cashback-monday.jpg',
        'featured' => false,
        'active' => true,
        'recurring' => 'weekly',
        'terms' => [
            'Calculated on net losses from previous week',
            'Maximum cashback: $100',
            'Credited every Monday',
            'No wagering requirements',
            'Minimum loss required: $50'
        ]
    ],
    [
        'id' => 4,
        'title' => 'VIP Tournament',
        'subtitle' => '$10,000 Prize Pool',
        'description' => 'Compete with other VIP players in our exclusive tournament. Climb the leaderboard and win your share of $10,000!',
        'type' => 'tournament',
        'prize_pool' => 10000,
        'start_date' => '2024-02-01',
        'end_date' => '2024-02-28',
        'image' => '/assets/images/promotions/vip-tournament.jpg',
        'featured' => true,
        'active' => true,
        'vip_only' => true,
        'terms' => [
            'VIP members only',
            'Tournament runs for entire month',
            'Points earned from all games',
            'Minimum bet: $1 per round',
            'Leaderboard updated daily'
        ]
    ]
];

// Filter promotions
$featuredPromotions = array_filter($activePromotions, fn($promo) => $promo['featured']);
$regularPromotions = array_filter($activePromotions, fn($promo) => !$promo['featured']);

// Get user's active bonuses if logged in
$userBonuses = [];
if ($currentUser) {
    $userBonuses = fetchAll(
        "SELECT ub.*, b.title, b.type, b.description 
         FROM user_bonuses ub 
         JOIN bonuses b ON ub.bonus_id = b.id 
         WHERE ub.user_id = ? AND ub.status = 'active' 
         ORDER BY ub.created_at DESC",
        [$currentUser['id']]
    );
}

$additionalScripts = [
    '/assets/js/promotions.js'
];

include '../includes/header.php';
?>

<!-- Hero Section -->
<section class="promotions-hero">
    <div class="container">
        <div class="hero-content">
            <div class="hero-text">
                <h1 class="hero-title">
                    <i class="fas fa-gift"></i>
                    Promotions & Bonuses
                </h1>
                <p class="hero-description">
                    Discover exclusive bonuses, exciting tournaments, and rewarding promotions. 
                    Boost your gaming experience with our generous offers!
                </p>
                <div class="hero-features">
                    <div class="feature-item">
                        <i class="fas fa-percentage"></i>
                        <span>Up to 100% Bonus</span>
                    </div>
                    <div class="feature-item">
                        <i class="fas fa-coins"></i>
                        <span>Free Spins</span>
                    </div>
                    <div class="feature-item">
                        <i class="fas fa-trophy"></i>
                        <span>Tournaments</span>
                    </div>
                    <div class="feature-item">
                        <i class="fas fa-undo"></i>
                        <span>Cashback</span>
                    </div>
                </div>
            </div>
            <div class="hero-visual">
                <div class="bonus-cards">
                    <div class="bonus-card floating">
                        <div class="card-header">
                            <span class="bonus-type">Welcome</span>
                            <span class="bonus-badge">NEW</span>
                        </div>
                        <div class="bonus-amount">100%</div>
                        <div class="bonus-details">up to $500</div>
                    </div>
                    <div class="bonus-card floating" style="animation-delay: 0.5s;">
                        <div class="card-header">
                            <span class="bonus-type">Reload</span>
                            <span class="bonus-badge">WEEKLY</span>
                        </div>
                        <div class="bonus-amount">50%</div>
                        <div class="bonus-details">up to $200</div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- User Active Bonuses -->
<?php if ($currentUser && !empty($userBonuses)): ?>
<section class="active-bonuses">
    <div class="container">
        <div class="section-header">
            <h2 class="section-title">
                <i class="fas fa-star"></i>
                Your Active Bonuses
            </h2>
        </div>
        
        <div class="bonuses-grid">
            <?php foreach ($userBonuses as $bonus): ?>
            <div class="active-bonus-card">
                <div class="bonus-header">
                    <h3 class="bonus-title"><?php echo $bonus['title']; ?></h3>
                    <span class="bonus-type-badge <?php echo $bonus['type']; ?>">
                        <?php echo ucfirst($bonus['type']); ?>
                    </span>
                </div>
                
                <div class="bonus-amount">
                    <span class="currency">$</span>
                    <span class="amount"><?php echo number_format($bonus['amount'], 2); ?></span>
                </div>
                
                <div class="bonus-progress">
                    <div class="progress-info">
                        <span>Wagering Progress</span>
                        <span><?php echo number_format($bonus['wagered_amount'], 2); ?> / <?php echo number_format($bonus['wagering_requirement'], 2); ?></span>
                    </div>
                    <div class="progress-bar">
                        <div class="progress-fill" style="width: <?php echo min(100, ($bonus['wagered_amount'] / $bonus['wagering_requirement']) * 100); ?>%"></div>
                    </div>
                </div>
                
                <div class="bonus-expires">
                    <i class="fas fa-clock"></i>
                    <span>Expires: <?php echo date('M j, Y', strtotime($bonus['expires_at'])); ?></span>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- Featured Promotions -->
<section class="featured-promotions">
    <div class="container">
        <div class="section-header">
            <h2 class="section-title">
                <i class="fas fa-fire"></i>
                Featured Promotions
            </h2>
        </div>
        
        <div class="featured-grid">
            <?php foreach ($featuredPromotions as $promotion): ?>
            <div class="promotion-card featured" data-promotion-id="<?php echo $promotion['id']; ?>">
                <div class="promotion-image">
                    <img src="<?php echo $promotion['image']; ?>" alt="<?php echo $promotion['title']; ?>">
                    <div class="promotion-overlay">
                        <button class="btn btn-primary" onclick="claimPromotion(<?php echo $promotion['id']; ?>)">
                            <?php echo $currentUser ? 'Claim Now' : 'Login to Claim'; ?>
                        </button>
                    </div>
                    <?php if (isset($promotion['vip_only']) && $promotion['vip_only']): ?>
                    <span class="promotion-badge vip">VIP ONLY</span>
                    <?php endif; ?>
                    <?php if (isset($promotion['recurring'])): ?>
                    <span class="promotion-badge recurring"><?php echo strtoupper($promotion['recurring']); ?></span>
                    <?php endif; ?>
                </div>
                
                <div class="promotion-content">
                    <h3 class="promotion-title"><?php echo $promotion['title']; ?></h3>
                    <p class="promotion-subtitle"><?php echo $promotion['subtitle']; ?></p>
                    <p class="promotion-description"><?php echo $promotion['description']; ?></p>
                    
                    <div class="promotion-highlights">
                        <?php if (isset($promotion['bonus_percentage'])): ?>
                        <div class="highlight">
                            <i class="fas fa-percentage"></i>
                            <span><?php echo $promotion['bonus_percentage']; ?>% Bonus</span>
                        </div>
                        <?php endif; ?>
                        
                        <?php if (isset($promotion['free_spins'])): ?>
                        <div class="highlight">
                            <i class="fas fa-coins"></i>
                            <span><?php echo $promotion['free_spins']; ?> Free Spins</span>
                        </div>
                        <?php endif; ?>
                        
                        <?php if (isset($promotion['cashback_percentage'])): ?>
                        <div class="highlight">
                            <i class="fas fa-undo"></i>
                            <span><?php echo $promotion['cashback_percentage']; ?>% Cashback</span>
                        </div>
                        <?php endif; ?>
                        
                        <?php if (isset($promotion['prize_pool'])): ?>
                        <div class="highlight">
                            <i class="fas fa-trophy"></i>
                            <span>$<?php echo number_format($promotion['prize_pool']); ?> Prize Pool</span>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="promotion-actions">
                        <button class="btn btn-primary" onclick="claimPromotion(<?php echo $promotion['id']; ?>)">
                            <?php echo $currentUser ? 'Claim Bonus' : 'Login to Claim'; ?>
                        </button>
                        <button class="btn btn-outline" onclick="showPromotionDetails(<?php echo $promotion['id']; ?>)">
                            View Terms
                        </button>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>

<!-- Regular Promotions -->
<section class="regular-promotions">
    <div class="container">
        <div class="section-header">
            <h2 class="section-title">
                <i class="fas fa-gift"></i>
                All Promotions
            </h2>
        </div>
        
        <div class="promotions-grid">
            <?php foreach ($regularPromotions as $promotion): ?>
            <div class="promotion-card" data-promotion-id="<?php echo $promotion['id']; ?>">
                <div class="promotion-image">
                    <img src="<?php echo $promotion['image']; ?>" alt="<?php echo $promotion['title']; ?>">
                    <?php if (isset($promotion['recurring'])): ?>
                    <span class="promotion-badge recurring"><?php echo strtoupper($promotion['recurring']); ?></span>
                    <?php endif; ?>
                </div>
                
                <div class="promotion-content">
                    <h3 class="promotion-title"><?php echo $promotion['title']; ?></h3>
                    <p class="promotion-subtitle"><?php echo $promotion['subtitle']; ?></p>
                    
                    <div class="promotion-key-info">
                        <?php if (isset($promotion['bonus_percentage'])): ?>
                        <div class="key-info-item">
                            <span class="label">Bonus:</span>
                            <span class="value"><?php echo $promotion['bonus_percentage']; ?>%</span>
                        </div>
                        <?php endif; ?>
                        
                        <?php if (isset($promotion['min_deposit'])): ?>
                        <div class="key-info-item">
                            <span class="label">Min Deposit:</span>
                            <span class="value">$<?php echo $promotion['min_deposit']; ?></span>
                        </div>
                        <?php endif; ?>
                        
                        <?php if (isset($promotion['wagering_requirement'])): ?>
                        <div class="key-info-item">
                            <span class="label">Wagering:</span>
                            <span class="value"><?php echo $promotion['wagering_requirement']; ?>x</span>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="promotion-actions">
                        <button class="btn btn-primary btn-sm" onclick="claimPromotion(<?php echo $promotion['id']; ?>)">
                            Claim
                        </button>
                        <button class="btn btn-outline btn-sm" onclick="showPromotionDetails(<?php echo $promotion['id']; ?>)">
                            Details
                        </button>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>

<!-- Promotion Details Modal -->
<div class="modal-backdrop" id="promotionModal">
    <div class="modal promotion-modal">
        <div class="modal-header">
            <h3 class="modal-title" id="promotionModalTitle">Promotion Details</h3>
            <button class="modal-close" onclick="closeModal('promotionModal')">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="modal-body" id="promotionModalBody">
            <!-- Promotion details will be loaded here -->
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    initializePromotions();
});

function initializePromotions() {
    // Initialize promotion cards
    const promotionCards = document.querySelectorAll('.promotion-card');
    promotionCards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.classList.add('hovered');
        });
        
        card.addEventListener('mouseleave', function() {
            this.classList.remove('hovered');
        });
    });
}

function claimPromotion(promotionId) {
    <?php if (!$currentUser): ?>
    // Redirect to login if not authenticated
    window.location.href = '/auth/login.php?redirect=' + encodeURIComponent(window.location.href);
    return;
    <?php endif; ?>
    
    // Show loading
    Crashify.ui.showLoading();
    
    // Claim promotion
    Crashify.api.request(`/api/promotions/${promotionId}/claim`, {
        method: 'POST'
    })
    .then(response => {
        if (response.success) {
            Crashify.ui.toast.success(response.message || 'Promotion claimed successfully!');
            
            // Refresh page to show updated bonuses
            setTimeout(() => {
                window.location.reload();
            }, 2000);
        } else {
            throw new Error(response.message);
        }
    })
    .catch(error => {
        Crashify.ui.toast.error(error.message || 'Failed to claim promotion');
    })
    .finally(() => {
        Crashify.ui.hideLoading();
    });
}

function showPromotionDetails(promotionId) {
    // Load promotion details
    Crashify.api.request(`/api/promotions/${promotionId}/details`)
        .then(response => {
            if (response.success) {
                document.getElementById('promotionModalTitle').textContent = response.promotion.title;
                document.getElementById('promotionModalBody').innerHTML = response.html;
                openModal('promotionModal');
            }
        })
        .catch(error => {
            Crashify.ui.toast.error('Failed to load promotion details');
        });
}
</script>

<?php include '../includes/footer.php'; ?>