<?php
// includes/notifications.php - Notification System
// سیستم اطلاع‌رسانی

class NotificationManager {
    private $pdo;
    private $user_id;
    
    public function __construct($pdo, $user_id = null) {
        $this->pdo = $pdo;
        $this->user_id = $user_id;
    }
    
    /**
     * Create a new notification
     */
    public function create($user_id, $type, $title, $message, $data = null, $priority = 'normal') {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO notifications (user_id, type, title, message, data, priority, created_at)
                VALUES (?, ?, ?, ?, ?, ?, NOW())
            ");
            
            $result = $stmt->execute([
                $user_id,
                $type,
                $title,
                $message,
                $data ? json_encode($data) : null,
                $priority
            ]);
            
            if ($result) {
                $notification_id = $this->pdo->lastInsertId();
                
                // Send real-time notification if user is online
                $this->sendRealTimeNotification($user_id, [
                    'id' => $notification_id,
                    'type' => $type,
                    'title' => $title,
                    'message' => $message,
                    'priority' => $priority,
                    'created_at' => date('Y-m-d H:i:s')
                ]);
                
                return $notification_id;
            }
            
            return false;
        } catch (Exception $e) {
            error_log("Notification creation error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get user notifications
     */
    public function getUserNotifications($user_id, $limit = 20, $offset = 0, $unread_only = false) {
        try {
            $where_clause = "WHERE user_id = ?";
            $params = [$user_id];
            
            if ($unread_only) {
                $where_clause .= " AND is_read = 0";
            }
            
            $stmt = $this->pdo->prepare("
                SELECT * FROM notifications 
                $where_clause
                ORDER BY priority DESC, created_at DESC 
                LIMIT ? OFFSET ?
            ");
            
            $params[] = $limit;
            $params[] = $offset;
            
            $stmt->execute($params);
            $notifications = $stmt->fetchAll();
            
            // Parse JSON data
            foreach ($notifications as &$notification) {
                if ($notification['data']) {
                    $notification['data'] = json_decode($notification['data'], true);
                }
            }
            
            return $notifications;
        } catch (Exception $e) {
            error_log("Get notifications error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Mark notification as read
     */
    public function markAsRead($notification_id, $user_id = null) {
        try {
            $where_clause = "WHERE id = ?";
            $params = [$notification_id];
            
            if ($user_id) {
                $where_clause .= " AND user_id = ?";
                $params[] = $user_id;
            }
            
            $stmt = $this->pdo->prepare("
                UPDATE notifications 
                SET is_read = 1, read_at = NOW() 
                $where_clause
            ");
            
            return $stmt->execute($params);
        } catch (Exception $e) {
            error_log("Mark as read error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Mark all notifications as read for user
     */
    public function markAllAsRead($user_id) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE notifications 
                SET is_read = 1, read_at = NOW() 
                WHERE user_id = ? AND is_read = 0
            ");
            
            return $stmt->execute([$user_id]);
        } catch (Exception $e) {
            error_log("Mark all as read error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Delete notification
     */
    public function delete($notification_id, $user_id = null) {
        try {
            $where_clause = "WHERE id = ?";
            $params = [$notification_id];
            
            if ($user_id) {
                $where_clause .= " AND user_id = ?";
                $params[] = $user_id;
            }
            
            $stmt = $this->pdo->prepare("DELETE FROM notifications $where_clause");
            return $stmt->execute($params);
        } catch (Exception $e) {
            error_log("Delete notification error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get unread count
     */
    public function getUnreadCount($user_id) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT COUNT(*) FROM notifications 
                WHERE user_id = ? AND is_read = 0
            ");
            $stmt->execute([$user_id]);
            return $stmt->fetchColumn();
        } catch (Exception $e) {
            error_log("Get unread count error: " . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Send real-time notification via WebSocket or Server-Sent Events
     */
    private function sendRealTimeNotification($user_id, $notification) {
        // Store in session for immediate display
        if (!isset($_SESSION['pending_notifications'])) {
            $_SESSION['pending_notifications'] = [];
        }
        
        $_SESSION['pending_notifications'][] = $notification;
        
        // Here you would integrate with WebSocket server or SSE
        // For now, we'll use session-based approach
    }
    
    /**
     * Create system notifications for different events
     */
    public function createWelcomeNotification($user_id) {
        return $this->create(
            $user_id,
            'welcome',
            'خوش آمدید!',
            'به Crashify خوش آمدید! از جوایز خوش‌آمدگویی خود استفاده کنید.',
            ['action' => 'view_bonuses', 'url' => '/pages/bonuses.php'],
            'high'
        );
    }
    
    public function createDepositNotification($user_id, $amount, $currency) {
        return $this->create(
            $user_id,
            'deposit',
            'واریز موفق',
            "مبلغ $amount $currency با موفقیت به حساب شما واریز شد.",
            ['amount' => $amount, 'currency' => $currency],
            'normal'
        );
    }
    
    public function createWithdrawalNotification($user_id, $amount, $currency, $status) {
        $messages = [
            'pending' => "درخواست برداشت $amount $currency در حال بررسی است.",
            'approved' => "درخواست برداشت $amount $currency تأیید شد.",
            'rejected' => "درخواست برداشت $amount $currency رد شد."
        ];
        
        return $this->create(
            $user_id,
            'withdrawal',
            'وضعیت برداشت',
            $messages[$status] ?? 'وضعیت برداشت تغییر کرد.',
            ['amount' => $amount, 'currency' => $currency, 'status' => $status],
            $status === 'rejected' ? 'high' : 'normal'
        );
    }
    
    public function createBonusNotification($user_id, $bonus_title, $amount) {
        return $this->create(
            $user_id,
            'bonus',
            'جایزه جدید!',
            "جایزه '$bonus_title' به مبلغ $$amount دریافت کردید.",
            ['bonus_title' => $bonus_title, 'amount' => $amount],
            'high'
        );
    }
    
    public function createBetWinNotification($user_id, $amount, $game_type) {
        return $this->create(
            $user_id,
            'win',
            'تبریک! برنده شدید!',
            "در بازی $game_type مبلغ $$amount برنده شدید.",
            ['amount' => $amount, 'game_type' => $game_type],
            'high'
        );
    }
    
    public function createSecurityNotification($user_id, $action, $ip_address) {
        $messages = [
            'login' => "ورود جدید از IP: $ip_address",
            'password_change' => 'رمز عبور تغییر کرد',
            'email_change' => 'ایمیل تغییر کرد',
            'suspicious_activity' => 'فعالیت مشکوک شناسایی شد'
        ];
        
        return $this->create(
            $user_id,
            'security',
            'اطلاعیه امنیتی',
            $messages[$action] ?? 'تغییرات امنیتی در حساب شما',
            ['action' => $action, 'ip_address' => $ip_address],
            'high'
        );
    }
    
    public function createPromotionNotification($user_id, $promotion_title, $description) {
        return $this->create(
            $user_id,
            'promotion',
            'پیشنهاد ویژه!',
            $promotion_title . ': ' . $description,
            ['promotion_title' => $promotion_title],
            'normal'
        );
    }
    
    /**
     * Clean old notifications
     */
    public function cleanOldNotifications($days = 30) {
        try {
            $stmt = $this->pdo->prepare("
                DELETE FROM notifications 
                WHERE created_at < DATE_SUB(NOW(), INTERVAL ? DAY)
            ");
            
            return $stmt->execute([$days]);
        } catch (Exception $e) {
            error_log("Clean old notifications error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get notification preferences for user
     */
    public function getUserPreferences($user_id) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT * FROM notification_preferences 
                WHERE user_id = ?
            ");
            $stmt->execute([$user_id]);
            $preferences = $stmt->fetch();
            
            if (!$preferences) {
                // Create default preferences
                $this->createDefaultPreferences($user_id);
                return $this->getUserPreferences($user_id);
            }
            
            return json_decode($preferences['preferences'], true);
        } catch (Exception $e) {
            error_log("Get user preferences error: " . $e->getMessage());
            return $this->getDefaultPreferences();
        }
    }
    
    /**
     * Update notification preferences
     */
    public function updateUserPreferences($user_id, $preferences) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO notification_preferences (user_id, preferences, updated_at)
                VALUES (?, ?, NOW())
                ON DUPLICATE KEY UPDATE 
                preferences = VALUES(preferences), 
                updated_at = VALUES(updated_at)
            ");
            
            return $stmt->execute([$user_id, json_encode($preferences)]);
        } catch (Exception $e) {
            error_log("Update user preferences error: " . $e->getMessage());
            return false;
        }
    }
    
    private function createDefaultPreferences($user_id) {
        $default_preferences = $this->getDefaultPreferences();
        return $this->updateUserPreferences($user_id, $default_preferences);
    }
    
    private function getDefaultPreferences() {
        return [
            'email' => [
                'welcome' => true,
                'deposit' => true,
                'withdrawal' => true,
                'bonus' => true,
                'win' => true,
                'security' => true,
                'promotion' => false
            ],
            'push' => [
                'welcome' => true,
                'deposit' => true,
                'withdrawal' => true,
                'bonus' => true,
                'win' => true,
                'security' => true,
                'promotion' => false
            ],
            'sms' => [
                'welcome' => false,
                'deposit' => false,
                'withdrawal' => true,
                'bonus' => false,
                'win' => false,
                'security' => true,
                'promotion' => false
            ]
        ];
    }
}

// Helper functions for easy access
function createNotification($user_id, $type, $title, $message, $data = null, $priority = 'normal') {
    global $pdo;
    $notificationManager = new NotificationManager($pdo);
    return $notificationManager->create($user_id, $type, $title, $message, $data, $priority);
}

function getUserNotifications($user_id, $limit = 20, $offset = 0, $unread_only = false) {
    global $pdo;
    $notificationManager = new NotificationManager($pdo);
    return $notificationManager->getUserNotifications($user_id, $limit, $offset, $unread_only);
}

function getUnreadNotificationCount($user_id) {
    global $pdo;
    $notificationManager = new NotificationManager($pdo);
    return $notificationManager->getUnreadCount($user_id);
}

function markNotificationAsRead($notification_id, $user_id = null) {
    global $pdo;
    $notificationManager = new NotificationManager($pdo);
    return $notificationManager->markAsRead($notification_id, $user_id);
}

function markAllNotificationsAsRead($user_id) {
    global $pdo;
    $notificationManager = new NotificationManager($pdo);
    return $notificationManager->markAllAsRead($user_id);
}

// Notification display functions
function displayNotificationIcon($type) {
    $icons = [
        'welcome' => 'fa-hand-holding-heart',
        'deposit' => 'fa-arrow-down',
        'withdrawal' => 'fa-arrow-up',
        'bonus' => 'fa-gift',
        'win' => 'fa-trophy',
        'security' => 'fa-shield-alt',
        'promotion' => 'fa-bullhorn',
        'system' => 'fa-cog',
        'default' => 'fa-bell'
    ];
    
    return $icons[$type] ?? $icons['default'];
}

function displayNotificationColor($type, $priority) {
    if ($priority === 'high') {
        return 'danger';
    }
    
    $colors = [
        'welcome' => 'success',
        'deposit' => 'success',
        'withdrawal' => 'warning',
        'bonus' => 'primary',
        'win' => 'success',
        'security' => 'danger',
        'promotion' => 'info',
        'system' => 'secondary',
        'default' => 'primary'
    ];
    
    return $colors[$type] ?? $colors['default'];
}

function formatNotificationTime($created_at) {
    $time = strtotime($created_at);
    $now = time();
    $diff = $now - $time;
    
    if ($diff < 60) {
        return 'همین الان';
    } elseif ($diff < 3600) {
        return floor($diff / 60) . ' دقیقه پیش';
    } elseif ($diff < 86400) {
        return floor($diff / 3600) . ' ساعت پیش';
    } elseif ($diff < 604800) {
        return floor($diff / 86400) . ' روز پیش';
    } else {
        return date('Y/m/d', $time);
    }
}
?>
