<?php
// api/notifications.php - Notifications API
// API اطلاع‌رسانی

require_once '../includes/auth.php';
require_once '../includes/functions.php';
require_once '../includes/notifications.php';

header('Content-Type: application/json');

// Check if user is logged in
if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit;
}

$user_id = $_SESSION['user_id'];
$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

$notificationManager = new NotificationManager($pdo, $user_id);

try {
    switch ($method) {
        case 'GET':
            handleGetRequest($notificationManager, $user_id, $action);
            break;
            
        case 'POST':
            handlePostRequest($notificationManager, $user_id, $action);
            break;
            
        case 'PUT':
            handlePutRequest($notificationManager, $user_id, $action);
            break;
            
        case 'DELETE':
            handleDeleteRequest($notificationManager, $user_id, $action);
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['success' => false, 'error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Internal server error']);
    error_log("Notifications API error: " . $e->getMessage());
}

function handleGetRequest($notificationManager, $user_id, $action) {
    switch ($action) {
        case 'list':
            $limit = intval($_GET['limit'] ?? 20);
            $offset = intval($_GET['offset'] ?? 0);
            $unread_only = isset($_GET['unread_only']) && $_GET['unread_only'] === 'true';
            
            $notifications = $notificationManager->getUserNotifications($user_id, $limit, $offset, $unread_only);
            $total_count = getTotalNotificationCount($user_id, $unread_only);
            
            echo json_encode([
                'success' => true,
                'data' => [
                    'notifications' => formatNotificationsForAPI($notifications),
                    'total_count' => $total_count,
                    'has_more' => ($offset + $limit) < $total_count
                ]
            ]);
            break;
            
        case 'unread_count':
            $count = $notificationManager->getUnreadCount($user_id);
            echo json_encode([
                'success' => true,
                'data' => ['unread_count' => $count]
            ]);
            break;
            
        case 'preferences':
            $preferences = $notificationManager->getUserPreferences($user_id);
            echo json_encode([
                'success' => true,
                'data' => ['preferences' => $preferences]
            ]);
            break;
            
        case 'recent':
            $notifications = $notificationManager->getUserNotifications($user_id, 5, 0, false);
            echo json_encode([
                'success' => true,
                'data' => ['notifications' => formatNotificationsForAPI($notifications)]
            ]);
            break;
            
        default:
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Invalid action']);
            break;
    }
}

function handlePostRequest($notificationManager, $user_id, $action) {
    $input = json_decode(file_get_contents('php://input'), true);
    
    switch ($action) {
        case 'mark_read':
            $notification_id = intval($input['notification_id'] ?? 0);
            
            if (!$notification_id) {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Notification ID required']);
                return;
            }
            
            $result = $notificationManager->markAsRead($notification_id, $user_id);
            
            if ($result) {
                echo json_encode(['success' => true, 'message' => 'Notification marked as read']);
            } else {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Failed to mark notification as read']);
            }
            break;
            
        case 'mark_all_read':
            $result = $notificationManager->markAllAsRead($user_id);
            
            if ($result) {
                echo json_encode(['success' => true, 'message' => 'All notifications marked as read']);
            } else {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Failed to mark all notifications as read']);
            }
            break;
            
        case 'test':
            // Create test notification (for development)
            if (ENVIRONMENT === 'development') {
                $result = $notificationManager->create(
                    $user_id,
                    'system',
                    'Test Notification',
                    'This is a test notification created via API',
                    ['test' => true],
                    'normal'
                );
                
                if ($result) {
                    echo json_encode(['success' => true, 'message' => 'Test notification created']);
                } else {
                    http_response_code(400);
                    echo json_encode(['success' => false, 'error' => 'Failed to create test notification']);
                }
            } else {
                http_response_code(403);
                echo json_encode(['success' => false, 'error' => 'Test notifications only available in development']);
            }
            break;
            
        default:
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Invalid action']);
            break;
    }
}

function handlePutRequest($notificationManager, $user_id, $action) {
    $input = json_decode(file_get_contents('php://input'), true);
    
    switch ($action) {
        case 'preferences':
            $preferences = $input['preferences'] ?? [];
            
            if (empty($preferences)) {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Preferences data required']);
                return;
            }
            
            $result = $notificationManager->updateUserPreferences($user_id, $preferences);
            
            if ($result) {
                echo json_encode(['success' => true, 'message' => 'Preferences updated successfully']);
            } else {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Failed to update preferences']);
            }
            break;
            
        default:
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Invalid action']);
            break;
    }
}

function handleDeleteRequest($notificationManager, $user_id, $action) {
    switch ($action) {
        case 'delete':
            $notification_id = intval($_GET['id'] ?? 0);
            
            if (!$notification_id) {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Notification ID required']);
                return;
            }
            
            $result = $notificationManager->delete($notification_id, $user_id);
            
            if ($result) {
                echo json_encode(['success' => true, 'message' => 'Notification deleted successfully']);
            } else {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Failed to delete notification']);
            }
            break;
            
        case 'clear_all':
            global $pdo;
            $stmt = $pdo->prepare("DELETE FROM notifications WHERE user_id = ?");
            $result = $stmt->execute([$user_id]);
            
            if ($result) {
                echo json_encode(['success' => true, 'message' => 'All notifications cleared']);
            } else {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Failed to clear notifications']);
            }
            break;
            
        default:
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Invalid action']);
            break;
    }
}

function formatNotificationsForAPI($notifications) {
    $formatted = [];
    
    foreach ($notifications as $notification) {
        $formatted[] = [
            'id' => $notification['id'],
            'type' => $notification['type'],
            'title' => $notification['title'],
            'message' => $notification['message'],
            'data' => $notification['data'],
            'priority' => $notification['priority'],
            'is_read' => (bool)$notification['is_read'],
            'created_at' => $notification['created_at'],
            'read_at' => $notification['read_at'],
            'formatted_time' => formatNotificationTime($notification['created_at']),
            'icon' => displayNotificationIcon($notification['type']),
            'color' => displayNotificationColor($notification['type'], $notification['priority'])
        ];
    }
    
    return $formatted;
}

function getTotalNotificationCount($user_id, $unread_only = false) {
    global $pdo;
    
    $where_clause = "WHERE user_id = ?";
    $params = [$user_id];
    
    if ($unread_only) {
        $where_clause .= " AND is_read = 0";
    }
    
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM notifications $where_clause");
    $stmt->execute($params);
    
    return $stmt->fetchColumn();
}

// WebSocket/SSE endpoint for real-time notifications
if ($action === 'stream') {
    // Set headers for Server-Sent Events
    header('Content-Type: text/event-stream');
    header('Cache-Control: no-cache');
    header('Connection: keep-alive');
    
    // Send initial connection message
    echo "data: " . json_encode(['type' => 'connected', 'message' => 'Connected to notification stream']) . "\n\n";
    
    // Check for new notifications every 5 seconds
    $last_check = time();
    
    while (true) {
        // Check if connection is still alive
        if (connection_aborted()) {
            break;
        }
        
        // Check for new notifications
        $current_time = time();
        if ($current_time - $last_check >= 5) {
            $notifications = $notificationManager->getUserNotifications($user_id, 10, 0, true);
            
            if (!empty($notifications)) {
                $formatted_notifications = formatNotificationsForAPI($notifications);
                echo "data: " . json_encode([
                    'type' => 'notifications',
                    'data' => $formatted_notifications
                ]) . "\n\n";
            }
            
            $last_check = $current_time;
        }
        
        // Send heartbeat
        echo "data: " . json_encode(['type' => 'heartbeat', 'timestamp' => time()]) . "\n\n";
        
        // Flush output
        if (ob_get_level()) {
            ob_flush();
        }
        flush();
        
        // Wait 1 second before next iteration
        sleep(1);
    }
}
?>
