<?php
// api/wallet.php - Wallet API
// API کیف پول

require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

$method = $_SERVER['REQUEST_METHOD'];
$endpoint = $_GET['endpoint'] ?? '';

try {
    // Require authentication for all wallet operations
    requireAuth();
    
    switch ($endpoint) {
        case 'balance':
            handleBalanceRequest($method);
            break;
            
        case 'deposit':
            handleDepositRequest($method);
            break;
            
        case 'withdraw':
            handleWithdrawRequest($method);
            break;
            
        case 'transactions':
            handleTransactionsRequest($method);
            break;
            
        case 'payment-methods':
            handlePaymentMethodsRequest($method);
            break;
            
        case 'limits':
            handleLimitsRequest($method);
            break;
            
        default:
            throw new Exception('Invalid endpoint');
    }
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

function handleBalanceRequest($method) {
    global $currentUser;
    
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    // Get current balance
    $user = fetchRow("SELECT balance, bonus_balance FROM users WHERE id = ?", [$currentUser['id']]);
    
    // Get pending transactions
    $pendingDeposits = fetchRow(
        "SELECT COUNT(*) as count, COALESCE(SUM(amount), 0) as total 
         FROM transactions 
         WHERE user_id = ? AND type = 'deposit' AND status = 'pending'",
        [$currentUser['id']]
    );
    
    $pendingWithdrawals = fetchRow(
        "SELECT COUNT(*) as count, COALESCE(SUM(amount), 0) as total 
         FROM transactions 
         WHERE user_id = ? AND type = 'withdrawal' AND status = 'pending'",
        [$currentUser['id']]
    );
    
    // Get recent transactions
    $recentTransactions = fetchAll(
        "SELECT * FROM transactions 
         WHERE user_id = ? 
         ORDER BY created_at DESC 
         LIMIT 5",
        [$currentUser['id']]
    );
    
    echo json_encode([
        'success' => true,
        'data' => [
            'balance' => floatval($user['balance']),
            'bonus_balance' => floatval($user['bonus_balance']),
            'total_balance' => floatval($user['balance']) + floatval($user['bonus_balance']),
            'pending' => [
                'deposits' => [
                    'count' => intval($pendingDeposits['count']),
                    'total' => floatval($pendingDeposits['total'])
                ],
                'withdrawals' => [
                    'count' => intval($pendingWithdrawals['count']),
                    'total' => floatval($pendingWithdrawals['total'])
                ]
            ],
            'recent_transactions' => $recentTransactions
        ]
    ]);
}

function handleDepositRequest($method) {
    global $currentUser;
    
    if ($method !== 'POST') {
        throw new Exception('Method not allowed');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    $amount = floatval($input['amount'] ?? 0);
    $paymentMethod = $input['payment_method'] ?? '';
    $currency = $input['currency'] ?? 'USD';
    
    if ($amount <= 0) {
        throw new Exception('Invalid amount');
    }
    
    if (!$paymentMethod) {
        throw new Exception('Payment method is required');
    }
    
    // Get site settings
    $minDeposit = getSetting('min_deposit', 10.00);
    $maxDeposit = getSetting('max_deposit', 10000.00);
    
    if ($amount < $minDeposit) {
        throw new Exception("Minimum deposit amount is $" . number_format($minDeposit, 2));
    }
    
    if ($amount > $maxDeposit) {
        throw new Exception("Maximum deposit amount is $" . number_format($maxDeposit, 2));
    }
    
    // Check if deposits are enabled
    if (!getSetting('deposits_enabled', true)) {
        throw new Exception('Deposits are currently disabled');
    }
    
    // Create deposit transaction
    $transactionId = execute(
        "INSERT INTO transactions (user_id, type, amount, currency, payment_method, status, description, created_at) 
         VALUES (?, 'deposit', ?, ?, ?, 'pending', 'Deposit via API', NOW())",
        [$currentUser['id'], $amount, $currency, $paymentMethod]
    );
    
    // Generate payment URL/reference (mock implementation)
    $paymentReference = 'DEP_' . $transactionId . '_' . time();
    
    // Update transaction with payment reference
    execute(
        "UPDATE transactions SET reference = ? WHERE id = ?",
        [$paymentReference, $transactionId]
    );
    
    // In production, integrate with actual payment processors
    $paymentUrl = generatePaymentUrl($paymentMethod, $amount, $paymentReference);
    
    echo json_encode([
        'success' => true,
        'data' => [
            'transaction_id' => $transactionId,
            'payment_reference' => $paymentReference,
            'payment_url' => $paymentUrl,
            'amount' => $amount,
            'currency' => $currency,
            'payment_method' => $paymentMethod,
            'expires_at' => date('Y-m-d H:i:s', strtotime('+30 minutes'))
        ]
    ]);
}

function handleWithdrawRequest($method) {
    global $currentUser;
    
    if ($method !== 'POST') {
        throw new Exception('Method not allowed');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    $amount = floatval($input['amount'] ?? 0);
    $paymentMethod = $input['payment_method'] ?? '';
    $paymentDetails = $input['payment_details'] ?? [];
    
    if ($amount <= 0) {
        throw new Exception('Invalid amount');
    }
    
    if (!$paymentMethod) {
        throw new Exception('Payment method is required');
    }
    
    // Get current user balance
    $user = fetchRow("SELECT balance FROM users WHERE id = ?", [$currentUser['id']]);
    
    if ($user['balance'] < $amount) {
        throw new Exception('Insufficient balance');
    }
    
    // Get site settings
    $minWithdrawal = getSetting('min_withdrawal', 20.00);
    $maxWithdrawal = getSetting('max_withdrawal', 5000.00);
    $withdrawalFee = getSetting('withdrawal_fee', 0.00);
    
    if ($amount < $minWithdrawal) {
        throw new Exception("Minimum withdrawal amount is $" . number_format($minWithdrawal, 2));
    }
    
    if ($amount > $maxWithdrawal) {
        throw new Exception("Maximum withdrawal amount is $" . number_format($maxWithdrawal, 2));
    }
    
    // Check if withdrawals are enabled
    if (!getSetting('withdrawals_enabled', true)) {
        throw new Exception('Withdrawals are currently disabled');
    }
    
    // Check KYC requirement
    if (getSetting('kyc_required', false) && $currentUser['kyc_status'] !== 'verified') {
        throw new Exception('KYC verification is required for withdrawals');
    }
    
    $totalAmount = $amount + $withdrawalFee;
    
    if ($user['balance'] < $totalAmount) {
        throw new Exception('Insufficient balance including withdrawal fee');
    }
    
    // Create withdrawal transaction
    $transactionId = execute(
        "INSERT INTO transactions (user_id, type, amount, fee, payment_method, payment_details, status, description, created_at) 
         VALUES (?, 'withdrawal', ?, ?, ?, ?, 'pending', 'Withdrawal via API', NOW())",
        [$currentUser['id'], $amount, $withdrawalFee, $paymentMethod, json_encode($paymentDetails)]
    );
    
    // Deduct amount from user balance
    execute("UPDATE users SET balance = balance - ? WHERE id = ?", [$totalAmount, $currentUser['id']]);
    
    // Generate withdrawal reference
    $withdrawalReference = 'WTH_' . $transactionId . '_' . time();
    
    execute(
        "UPDATE transactions SET reference = ? WHERE id = ?",
        [$withdrawalReference, $transactionId]
    );
    
    echo json_encode([
        'success' => true,
        'data' => [
            'transaction_id' => $transactionId,
            'withdrawal_reference' => $withdrawalReference,
            'amount' => $amount,
            'fee' => $withdrawalFee,
            'total_deducted' => $totalAmount,
            'payment_method' => $paymentMethod,
            'status' => 'pending',
            'estimated_processing_time' => getSetting('withdrawal_processing_time', 24) . ' hours'
        ]
    ]);
}

function handleTransactionsRequest($method) {
    global $currentUser;
    
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    $page = intval($_GET['page'] ?? 1);
    $limit = intval($_GET['limit'] ?? 20);
    $type = $_GET['type'] ?? 'all';
    $status = $_GET['status'] ?? 'all';
    
    $offset = ($page - 1) * $limit;
    
    // Build query conditions
    $conditions = ["user_id = ?"];
    $params = [$currentUser['id']];
    
    if ($type !== 'all') {
        $conditions[] = "type = ?";
        $params[] = $type;
    }
    
    if ($status !== 'all') {
        $conditions[] = "status = ?";
        $params[] = $status;
    }
    
    $whereClause = implode(' AND ', $conditions);
    
    // Get total count
    $totalCount = fetchRow(
        "SELECT COUNT(*) as count FROM transactions WHERE $whereClause",
        $params
    )['count'];
    
    // Get transactions
    $transactions = fetchAll(
        "SELECT * FROM transactions 
         WHERE $whereClause 
         ORDER BY created_at DESC 
         LIMIT $limit OFFSET $offset",
        $params
    );
    
    echo json_encode([
        'success' => true,
        'data' => [
            'transactions' => $transactions,
            'pagination' => [
                'current_page' => $page,
                'total_pages' => ceil($totalCount / $limit),
                'total_count' => $totalCount,
                'per_page' => $limit
            ]
        ]
    ]);
}

function handlePaymentMethodsRequest($method) {
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    $type = $_GET['type'] ?? 'all'; // deposit, withdrawal, all
    
    // Mock payment methods - in production, this would be configurable
    $paymentMethods = [
        'deposit' => [
            [
                'id' => 'credit_card',
                'name' => 'Credit/Debit Card',
                'icon' => 'fas fa-credit-card',
                'min_amount' => 10.00,
                'max_amount' => 5000.00,
                'fee' => 0.00,
                'processing_time' => 'Instant',
                'currencies' => ['USD', 'EUR', 'GBP']
            ],
            [
                'id' => 'bank_transfer',
                'name' => 'Bank Transfer',
                'icon' => 'fas fa-university',
                'min_amount' => 50.00,
                'max_amount' => 10000.00,
                'fee' => 0.00,
                'processing_time' => '1-3 business days',
                'currencies' => ['USD', 'EUR']
            ],
            [
                'id' => 'crypto',
                'name' => 'Cryptocurrency',
                'icon' => 'fab fa-bitcoin',
                'min_amount' => 20.00,
                'max_amount' => 50000.00,
                'fee' => 0.00,
                'processing_time' => '10-60 minutes',
                'currencies' => ['BTC', 'ETH', 'USDT']
            ]
        ],
        'withdrawal' => [
            [
                'id' => 'bank_transfer',
                'name' => 'Bank Transfer',
                'icon' => 'fas fa-university',
                'min_amount' => 20.00,
                'max_amount' => 5000.00,
                'fee' => 5.00,
                'processing_time' => '1-3 business days',
                'currencies' => ['USD', 'EUR']
            ],
            [
                'id' => 'crypto',
                'name' => 'Cryptocurrency',
                'icon' => 'fab fa-bitcoin',
                'min_amount' => 20.00,
                'max_amount' => 10000.00,
                'fee' => 0.00,
                'processing_time' => '10-60 minutes',
                'currencies' => ['BTC', 'ETH', 'USDT']
            ]
        ]
    ];
    
    $result = [];
    if ($type === 'all') {
        $result = $paymentMethods;
    } else {
        $result = $paymentMethods[$type] ?? [];
    }
    
    echo json_encode([
        'success' => true,
        'data' => $result
    ]);
}

function handleLimitsRequest($method) {
    global $currentUser;
    
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    // Get user's current limits and usage
    $today = date('Y-m-d');
    $thisMonth = date('Y-m');
    
    $dailyDeposits = fetchRow(
        "SELECT COALESCE(SUM(amount), 0) as total 
         FROM transactions 
         WHERE user_id = ? AND type = 'deposit' AND status = 'completed' AND DATE(created_at) = ?",
        [$currentUser['id'], $today]
    )['total'];
    
    $monthlyDeposits = fetchRow(
        "SELECT COALESCE(SUM(amount), 0) as total 
         FROM transactions 
         WHERE user_id = ? AND type = 'deposit' AND status = 'completed' AND DATE_FORMAT(created_at, '%Y-%m') = ?",
        [$currentUser['id'], $thisMonth]
    )['total'];
    
    $dailyWithdrawals = fetchRow(
        "SELECT COALESCE(SUM(amount), 0) as total 
         FROM transactions 
         WHERE user_id = ? AND type = 'withdrawal' AND status IN ('completed', 'pending') AND DATE(created_at) = ?",
        [$currentUser['id'], $today]
    )['total'];
    
    $monthlyWithdrawals = fetchRow(
        "SELECT COALESCE(SUM(amount), 0) as total 
         FROM transactions 
         WHERE user_id = ? AND type = 'withdrawal' AND status IN ('completed', 'pending') AND DATE_FORMAT(created_at, '%Y-%m') = ?",
        [$currentUser['id'], $thisMonth]
    )['total'];
    
    // Default limits (would be configurable per user level)
    $limits = [
        'deposit' => [
            'min_amount' => floatval(getSetting('min_deposit', 10.00)),
            'max_amount' => floatval(getSetting('max_deposit', 10000.00)),
            'daily_limit' => 5000.00,
            'monthly_limit' => 50000.00,
            'daily_used' => floatval($dailyDeposits),
            'monthly_used' => floatval($monthlyDeposits)
        ],
        'withdrawal' => [
            'min_amount' => floatval(getSetting('min_withdrawal', 20.00)),
            'max_amount' => floatval(getSetting('max_withdrawal', 5000.00)),
            'daily_limit' => 2000.00,
            'monthly_limit' => 20000.00,
            'daily_used' => floatval($dailyWithdrawals),
            'monthly_used' => floatval($monthlyWithdrawals)
        ]
    ];
    
    // Calculate remaining limits
    $limits['deposit']['daily_remaining'] = max(0, $limits['deposit']['daily_limit'] - $limits['deposit']['daily_used']);
    $limits['deposit']['monthly_remaining'] = max(0, $limits['deposit']['monthly_limit'] - $limits['deposit']['monthly_used']);
    
    $limits['withdrawal']['daily_remaining'] = max(0, $limits['withdrawal']['daily_limit'] - $limits['withdrawal']['daily_used']);
    $limits['withdrawal']['monthly_remaining'] = max(0, $limits['withdrawal']['monthly_limit'] - $limits['withdrawal']['monthly_used']);
    
    echo json_encode([
        'success' => true,
        'data' => $limits
    ]);
}

function generatePaymentUrl($paymentMethod, $amount, $reference) {
    // Mock payment URL generation
    // In production, integrate with actual payment processors
    
    switch ($paymentMethod) {
        case 'credit_card':
            return "https://payment.crashify.com/card?ref=$reference&amount=$amount";
        case 'bank_transfer':
            return "https://payment.crashify.com/bank?ref=$reference&amount=$amount";
        case 'crypto':
            return "https://payment.crashify.com/crypto?ref=$reference&amount=$amount";
        default:
            return "https://payment.crashify.com/generic?ref=$reference&amount=$amount";
    }
}

function getSetting($key, $default = null) {
    $setting = fetchRow("SELECT setting_value FROM site_settings WHERE setting_key = ?", [$key]);
    return $setting ? $setting['setting_value'] : $default;
}
?>
