<?php
// pages/login.php - Login Page
// صفحه ورود

require_once '../includes/functions.php';

$page_title = 'ورود';
$current_page = 'login';

// Redirect if already logged in
if (isLoggedIn()) {
    header('Location: /pages/dashboard.php');
    exit;
}

$error_message = '';
$success_message = '';

// Handle login form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = sanitizeInput($_POST['email']);
    $password = $_POST['password'];
    $remember_me = isset($_POST['remember_me']);
    
    if (empty($email) || empty($password)) {
        $error_message = 'لطفاً تمام فیلدها را پر کنید';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error_message = 'ایمیل نامعتبر است';
    } else {
        // Check user credentials
        $stmt = $pdo->prepare("SELECT * FROM users WHERE email = ? AND status = 'active'");
        $stmt->execute([$email]);
        $user = $stmt->fetch();
        
        if ($user && password_verify($password, $user['password'])) {
            // Check if account is banned
            if ($user['status'] === 'banned') {
                $error_message = 'حساب کاربری شما مسدود شده است';
            } else {
                // Login successful
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['user_email'] = $user['email'];
                $_SESSION['user_name'] = $user['first_name'] . ' ' . $user['last_name'];
                $_SESSION['user_role'] = $user['role'];
                $_SESSION['user_language'] = $user['language'] ?: 'fa';
                
                // Update last login
                $stmt = $pdo->prepare("UPDATE users SET last_login = NOW(), last_ip = ? WHERE id = ?");
                $stmt->execute([$_SERVER['REMOTE_ADDR'], $user['id']]);
                
                // Handle remember me
                if ($remember_me) {
                    $token = bin2hex(random_bytes(32));
                    $expires = date('Y-m-d H:i:s', strtotime('+30 days'));
                    
                    // Store remember token
                    $stmt = $pdo->prepare("
                        INSERT INTO user_sessions (user_id, token, expires_at, created_at) 
                        VALUES (?, ?, ?, NOW())
                        ON DUPLICATE KEY UPDATE token = VALUES(token), expires_at = VALUES(expires_at)
                    ");
                    $stmt->execute([$user['id'], hash('sha256', $token), $expires]);
                    
                    // Set cookie
                    setcookie('remember_token', $token, strtotime('+30 days'), '/', '', true, true);
                }
                
                // Log login activity
                $stmt = $pdo->prepare("
                    INSERT INTO user_activity (user_id, activity_type, description, ip_address, user_agent, created_at)
                    VALUES (?, 'login', 'User logged in', ?, ?, NOW())
                ");
                $stmt->execute([$user['id'], $_SERVER['REMOTE_ADDR'], $_SERVER['HTTP_USER_AGENT']]);
                
                // Redirect to intended page or dashboard
                $redirect_url = $_SESSION['redirect_after_login'] ?? '/pages/dashboard.php';
                unset($_SESSION['redirect_after_login']);
                
                header('Location: ' . $redirect_url);
                exit;
            }
        } else {
            $error_message = 'ایمیل یا رمز عبور اشتباه است';
            
            // Log failed login attempt
            $stmt = $pdo->prepare("
                INSERT INTO login_attempts (email, ip_address, user_agent, success, created_at)
                VALUES (?, ?, ?, 0, NOW())
            ");
            $stmt->execute([$email, $_SERVER['REMOTE_ADDR'], $_SERVER['HTTP_USER_AGENT']]);
        }
    }
}

// Check for password reset success message
if (isset($_GET['reset']) && $_GET['reset'] === 'success') {
    $success_message = 'رمز عبور با موفقیت تغییر کرد. اکنون می‌توانید وارد شوید';
}

// Check for registration success message
if (isset($_GET['registered']) && $_GET['registered'] === 'success') {
    $success_message = 'ثبت نام با موفقیت انجام شد. اکنون می‌توانید وارد شوید';
}

include '../includes/header.php';
?>

<div class="auth-container">
    <div class="auth-wrapper">
        <div class="auth-card">
            <div class="auth-header">
                <div class="auth-logo">
                    <img src="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'assets/images/logo.png" alt="Crashify" class="logo-img">
                    <h2>Crashify</h2>
                </div>
                <h3>ورود به حساب کاربری</h3>
                <p class="text-muted">به دنیای هیجان‌انگیز شرط‌بندی خوش آمدید</p>
            </div>
            
            <div class="auth-body">
                <?php if ($error_message): ?>
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-circle"></i>
                        <?php echo $error_message; ?>
                    </div>
                <?php endif; ?>
                
                <?php if ($success_message): ?>
                    <div class="alert alert-success">
                        <i class="fas fa-check-circle"></i>
                        <?php echo $success_message; ?>
                    </div>
                <?php endif; ?>
                
                <form method="POST" class="auth-form" id="loginForm">
                    <div class="form-group">
                        <label for="email">
                            <i class="fas fa-envelope"></i>
                            ایمیل
                        </label>
                        <input type="email" class="form-control" id="email" name="email" 
                               value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>" 
                               placeholder="ایمیل خود را وارد کنید" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="password">
                            <i class="fas fa-lock"></i>
                            رمز عبور
                        </label>
                        <div class="password-input">
                            <input type="password" class="form-control" id="password" name="password" 
                                   placeholder="رمز عبور خود را وارد کنید" required>
                            <button type="button" class="password-toggle" onclick="togglePassword('password')">
                                <i class="fas fa-eye"></i>
                            </button>
                        </div>
                    </div>
                    
                    <div class="form-options">
                        <div class="custom-control custom-checkbox">
                            <input type="checkbox" class="custom-control-input" id="remember_me" name="remember_me">
                            <label class="custom-control-label" for="remember_me">
                                مرا به خاطر بسپار
                            </label>
                        </div>
                        
                        <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'auth/forgot-password.php" class="forgot-password-link">
                            رمز عبور را فراموش کرده‌اید؟
                        </a>
                    </div>
                    
                    <button type="submit" class="btn btn-primary btn-block auth-submit-btn">
                        <i class="fas fa-sign-in-alt"></i>
                        ورود
                    </button>
                </form>
                
                <!-- Social Login -->
                <div class="social-login">
                    <div class="divider">
                        <span>یا</span>
                    </div>
                    
                    <div class="social-buttons">
                        <button class="btn btn-social btn-google" onclick="socialLogin('google')">
                            <i class="fab fa-google"></i>
                            ورود با گوگل
                        </button>
                        
                        <button class="btn btn-social btn-facebook" onclick="socialLogin('facebook')">
                            <i class="fab fa-facebook-f"></i>
                            ورود با فیسبوک
                        </button>
                    </div>
                </div>
                
                <!-- Quick Demo Login -->
                <div class="demo-login">
                    <h6>ورود سریع برای تست:</h6>
                    <div class="demo-buttons">
                        <button class="btn btn-outline-secondary btn-sm" onclick="demoLogin('user')">
                            <i class="fas fa-user"></i>
                            کاربر عادی
                        </button>
                        <button class="btn btn-outline-warning btn-sm" onclick="demoLogin('admin')">
                            <i class="fas fa-user-shield"></i>
                            مدیر
                        </button>
                    </div>
                </div>
            </div>
            
            <div class="auth-footer">
                <p>
                    حساب کاربری ندارید؟
                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'auth/register.php" class="register-link">
                        <strong>ثبت نام کنید</strong>
                    </a>
                </p>
                
                <div class="auth-links">
                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/support.php">پشتیبانی</a>
                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/terms.php">قوانین</a>
                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/privacy.php">حریم خصوصی</a>
                </div>
            </div>
        </div>
        
        <!-- Features Sidebar -->
        <div class="features-sidebar">
            <div class="features-content">
                <h4>چرا Crashify؟</h4>
                
                <div class="feature-item">
                    <div class="feature-icon">
                        <i class="fas fa-shield-alt"></i>
                    </div>
                    <div class="feature-text">
                        <h6>امنیت بالا</h6>
                        <p>حفاظت کامل از اطلاعات و وجوه شما</p>
                    </div>
                </div>
                
                <div class="feature-item">
                    <div class="feature-icon">
                        <i class="fas fa-bolt"></i>
                    </div>
                    <div class="feature-text">
                        <h6>پردازش سریع</h6>
                        <p>واریز و برداشت فوری</p>
                    </div>
                </div>
                
                <div class="feature-item">
                    <div class="feature-icon">
                        <i class="fas fa-headset"></i>
                    </div>
                    <div class="feature-text">
                        <h6>پشتیبانی 24/7</h6>
                        <p>همیشه در کنار شما هستیم</p>
                    </div>
                </div>
                
                <div class="feature-item">
                    <div class="feature-icon">
                        <i class="fas fa-gift"></i>
                    </div>
                    <div class="feature-text">
                        <h6>جوایز ویژه</h6>
                        <p>بونوس‌های جذاب برای کاربران</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.auth-container {
    min-height: 100vh;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    padding: 2rem 1rem;
}

.auth-wrapper {
    display: flex;
    max-width: 1200px;
    width: 100%;
    background: white;
    border-radius: 20px;
    box-shadow: 0 20px 40px rgba(0,0,0,0.1);
    overflow: hidden;
}

.auth-card {
    flex: 1;
    padding: 3rem;
    max-width: 500px;
}

.auth-header {
    text-align: center;
    margin-bottom: 2rem;
}

.auth-logo {
    display: flex;
    align-items: center;
    justify-content: center;
    margin-bottom: 1rem;
}

.logo-img {
    width: 40px;
    height: 40px;
    margin-right: 0.5rem;
}

.auth-logo h2 {
    color: #667eea;
    margin: 0;
    font-weight: bold;
}

.auth-header h3 {
    color: #2c3e50;
    margin-bottom: 0.5rem;
}

.auth-body {
    margin-bottom: 2rem;
}

.form-group {
    margin-bottom: 1.5rem;
}

.form-group label {
    display: flex;
    align-items: center;
    margin-bottom: 0.5rem;
    color: #495057;
    font-weight: 500;
}

.form-group label i {
    margin-right: 0.5rem;
    width: 16px;
}

.form-control {
    border: 2px solid #e9ecef;
    border-radius: 10px;
    padding: 0.75rem 1rem;
    font-size: 1rem;
    transition: border-color 0.3s, box-shadow 0.3s;
}

.form-control:focus {
    border-color: #667eea;
    box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
}

.password-input {
    position: relative;
}

.password-toggle {
    position: absolute;
    right: 1rem;
    top: 50%;
    transform: translateY(-50%);
    background: none;
    border: none;
    color: #6c757d;
    cursor: pointer;
    padding: 0;
}

.password-toggle:hover {
    color: #495057;
}

.form-options {
    display: flex;
    justify-content: between;
    align-items: center;
    margin-bottom: 2rem;
}

.forgot-password-link {
    color: #667eea;
    text-decoration: none;
    font-size: 0.9rem;
}

.forgot-password-link:hover {
    text-decoration: underline;
}

.auth-submit-btn {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border: none;
    border-radius: 10px;
    padding: 0.75rem;
    font-size: 1.1rem;
    font-weight: 600;
    transition: transform 0.2s, box-shadow 0.2s;
}

.auth-submit-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
}

.social-login {
    margin: 2rem 0;
}

.divider {
    text-align: center;
    position: relative;
    margin: 1.5rem 0;
}

.divider:before {
    content: '';
    position: absolute;
    top: 50%;
    left: 0;
    right: 0;
    height: 1px;
    background: #dee2e6;
}

.divider span {
    background: white;
    padding: 0 1rem;
    color: #6c757d;
    font-size: 0.9rem;
}

.social-buttons {
    display: flex;
    gap: 1rem;
}

.btn-social {
    flex: 1;
    border-radius: 10px;
    padding: 0.75rem;
    font-weight: 500;
    transition: transform 0.2s;
}

.btn-social:hover {
    transform: translateY(-2px);
}

.btn-google {
    background: #db4437;
    border-color: #db4437;
    color: white;
}

.btn-facebook {
    background: #3b5998;
    border-color: #3b5998;
    color: white;
}

.demo-login {
    background: #f8f9fa;
    border-radius: 10px;
    padding: 1.5rem;
    text-align: center;
    margin-top: 2rem;
}

.demo-login h6 {
    color: #495057;
    margin-bottom: 1rem;
}

.demo-buttons {
    display: flex;
    gap: 1rem;
    justify-content: center;
}

.demo-buttons .btn {
    border-radius: 8px;
}

.auth-footer {
    text-align: center;
    padding-top: 2rem;
    border-top: 1px solid #dee2e6;
}

.register-link {
    color: #667eea;
    text-decoration: none;
}

.register-link:hover {
    text-decoration: underline;
}

.auth-links {
    margin-top: 1rem;
    display: flex;
    justify-content: center;
    gap: 1rem;
}

.auth-links a {
    color: #6c757d;
    text-decoration: none;
    font-size: 0.9rem;
}

.auth-links a:hover {
    color: #495057;
    text-decoration: underline;
}

.features-sidebar {
    flex: 1;
    background: linear-gradient(135deg, #2c3e50 0%, #34495e 100%);
    color: white;
    padding: 3rem;
    display: flex;
    align-items: center;
}

.features-content h4 {
    margin-bottom: 2rem;
    font-weight: bold;
}

.feature-item {
    display: flex;
    align-items: center;
    margin-bottom: 2rem;
}

.feature-icon {
    width: 60px;
    height: 60px;
    background: rgba(255,255,255,0.1);
    border-radius: 15px;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-right: 1rem;
    font-size: 1.5rem;
}

.feature-text h6 {
    margin-bottom: 0.25rem;
    font-weight: 600;
}

.feature-text p {
    margin: 0;
    opacity: 0.8;
    font-size: 0.9rem;
}

@media (max-width: 768px) {
    .auth-wrapper {
        flex-direction: column;
    }
    
    .features-sidebar {
        order: -1;
        padding: 2rem;
    }
    
    .features-content {
        text-align: center;
    }
    
    .feature-item {
        flex-direction: column;
        text-align: center;
    }
    
    .feature-icon {
        margin-right: 0;
        margin-bottom: 1rem;
    }
    
    .auth-card {
        padding: 2rem;
    }
    
    .social-buttons {
        flex-direction: column;
    }
    
    .demo-buttons {
        flex-direction: column;
        align-items: center;
    }
    
    .demo-buttons .btn {
        width: 200px;
    }
}
</style>

<script>
// Toggle password visibility
function togglePassword(fieldId) {
    const field = document.getElementById(fieldId);
    const toggle = field.nextElementSibling.querySelector('i');
    
    if (field.type === 'password') {
        field.type = 'text';
        toggle.classList.remove('fa-eye');
        toggle.classList.add('fa-eye-slash');
    } else {
        field.type = 'password';
        toggle.classList.remove('fa-eye-slash');
        toggle.classList.add('fa-eye');
    }
}

// Social login (placeholder)
function socialLogin(provider) {
    alert(`ورود با ${provider} به زودی فعال خواهد شد`);
}

// Demo login
function demoLogin(type) {
    const form = document.getElementById('loginForm');
    const emailField = document.getElementById('email');
    const passwordField = document.getElementById('password');
    
    if (type === 'user') {
        emailField.value = 'demo@crashify.com';
        passwordField.value = 'demo123';
    } else if (type === 'admin') {
        emailField.value = 'admin@crashify.com';
        passwordField.value = 'admin123';
    }
    
    // Auto submit after a short delay
    setTimeout(() => {
        form.submit();
    }, 500);
}

// Form validation
document.getElementById('loginForm').addEventListener('submit', function(e) {
    const email = document.getElementById('email').value;
    const password = document.getElementById('password').value;
    
    if (!email || !password) {
        e.preventDefault();
        alert('لطفاً تمام فیلدها را پر کنید');
        return;
    }
    
    if (!isValidEmail(email)) {
        e.preventDefault();
        alert('ایمیل نامعتبر است');
        return;
    }
    
    // Show loading state
    const submitBtn = this.querySelector('.auth-submit-btn');
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> در حال ورود...';
});

function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

// Auto-focus first empty field
document.addEventListener('DOMContentLoaded', function() {
    const emailField = document.getElementById('email');
    const passwordField = document.getElementById('password');
    
    if (!emailField.value) {
        emailField.focus();
    } else if (!passwordField.value) {
        passwordField.focus();
    }
});
</script>

<?php include '../includes/footer.php'; ?>
