<?php
// pages/profile.php - User Profile Page
// صفحه پروفایل کاربر

require_once '../includes/auth.php';
require_once '../includes/functions.php';

// Check if user is logged in
if (!isLoggedIn()) {
    header('Location: /auth/login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$page_title = 'پروفایل کاربری';
$current_page = 'profile';

// Handle form submissions
$success_message = '';
$error_message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['update_profile'])) {
        // Update profile information
        $first_name = sanitizeInput($_POST['first_name']);
        $last_name = sanitizeInput($_POST['last_name']);
        $email = sanitizeInput($_POST['email']);
        $phone = sanitizeInput($_POST['phone']);
        $birth_date = sanitizeInput($_POST['birth_date']);
        $country = sanitizeInput($_POST['country']);
        $city = sanitizeInput($_POST['city']);
        
        // Validate email
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $error_message = 'ایمیل نامعتبر است';
        } else {
            // Check if email already exists for another user
            $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ? AND id != ?");
            $stmt->execute([$email, $user_id]);
            
            if ($stmt->rowCount() > 0) {
                $error_message = 'این ایمیل قبلاً استفاده شده است';
            } else {
                // Update user profile
                $stmt = $pdo->prepare("
                    UPDATE users SET 
                        first_name = ?, last_name = ?, email = ?, phone = ?, 
                        birth_date = ?, country = ?, city = ?, updated_at = NOW()
                    WHERE id = ?
                ");
                
                if ($stmt->execute([$first_name, $last_name, $email, $phone, $birth_date, $country, $city, $user_id])) {
                    $success_message = 'اطلاعات پروفایل با موفقیت به‌روزرسانی شد';
                    
                    // Update session data
                    $_SESSION['user_name'] = $first_name . ' ' . $last_name;
                    $_SESSION['user_email'] = $email;
                } else {
                    $error_message = 'خطا در به‌روزرسانی اطلاعات';
                }
            }
        }
    }
    
    if (isset($_POST['change_password'])) {
        // Change password
        $current_password = $_POST['current_password'];
        $new_password = $_POST['new_password'];
        $confirm_password = $_POST['confirm_password'];
        
        // Validate passwords
        if (strlen($new_password) < 6) {
            $error_message = 'رمز عبور جدید باید حداقل 6 کاراکتر باشد';
        } elseif ($new_password !== $confirm_password) {
            $error_message = 'تکرار رمز عبور مطابقت ندارد';
        } else {
            // Verify current password
            $stmt = $pdo->prepare("SELECT password FROM users WHERE id = ?");
            $stmt->execute([$user_id]);
            $user = $stmt->fetch();
            
            if (!password_verify($current_password, $user['password'])) {
                $error_message = 'رمز عبور فعلی اشتباه است';
            } else {
                // Update password
                $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                $stmt = $pdo->prepare("UPDATE users SET password = ?, updated_at = NOW() WHERE id = ?");
                
                if ($stmt->execute([$hashed_password, $user_id])) {
                    $success_message = 'رمز عبور با موفقیت تغییر کرد';
                } else {
                    $error_message = 'خطا در تغییر رمز عبور';
                }
            }
        }
    }
    
    if (isset($_POST['update_preferences'])) {
        // Update user preferences
        $language = sanitizeInput($_POST['language']);
        $timezone = sanitizeInput($_POST['timezone']);
        $currency = sanitizeInput($_POST['currency']);
        $notifications_email = isset($_POST['notifications_email']) ? 1 : 0;
        $notifications_sms = isset($_POST['notifications_sms']) ? 1 : 0;
        $notifications_push = isset($_POST['notifications_push']) ? 1 : 0;
        $two_factor_enabled = isset($_POST['two_factor_enabled']) ? 1 : 0;
        
        // Update preferences
        $stmt = $pdo->prepare("
            UPDATE users SET 
                language = ?, timezone = ?, currency = ?, 
                notifications_email = ?, notifications_sms = ?, notifications_push = ?,
                two_factor_enabled = ?, updated_at = NOW()
            WHERE id = ?
        ");
        
        if ($stmt->execute([$language, $timezone, $currency, $notifications_email, $notifications_sms, $notifications_push, $two_factor_enabled, $user_id])) {
            $success_message = 'تنظیمات با موفقیت به‌روزرسانی شد';
            $_SESSION['user_language'] = $language;
        } else {
            $error_message = 'خطا در به‌روزرسانی تنظیمات';
        }
    }
    
    if (isset($_POST['upload_avatar'])) {
        // Handle avatar upload
        if (isset($_FILES['avatar']) && $_FILES['avatar']['error'] === UPLOAD_ERR_OK) {
            $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
            $max_size = 2 * 1024 * 1024; // 2MB
            
            if (!in_array($_FILES['avatar']['type'], $allowed_types)) {
                $error_message = 'فرمت فایل مجاز نیست. فقط JPG، PNG و GIF مجاز است';
            } elseif ($_FILES['avatar']['size'] > $max_size) {
                $error_message = 'حجم فایل نباید بیشتر از 2 مگابایت باشد';
            } else {
                $upload_dir = '../uploads/avatars/';
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }
                
                $file_extension = pathinfo($_FILES['avatar']['name'], PATHINFO_EXTENSION);
                $new_filename = 'avatar_' . $user_id . '_' . time() . '.' . $file_extension;
                $upload_path = $upload_dir . $new_filename;
                
                if (move_uploaded_file($_FILES['avatar']['tmp_name'], $upload_path)) {
                    // Delete old avatar if exists
                    $stmt = $pdo->prepare("SELECT avatar FROM users WHERE id = ?");
                    $stmt->execute([$user_id]);
                    $old_avatar = $stmt->fetchColumn();
                    
                    if ($old_avatar && file_exists('../uploads/avatars/' . $old_avatar)) {
                        unlink('../uploads/avatars/' . $old_avatar);
                    }
                    
                    // Update database
                    $stmt = $pdo->prepare("UPDATE users SET avatar = ?, updated_at = NOW() WHERE id = ?");
                    if ($stmt->execute([$new_filename, $user_id])) {
                        $success_message = 'تصویر پروفایل با موفقیت به‌روزرسانی شد';
                    } else {
                        $error_message = 'خطا در به‌روزرسانی تصویر پروفایل';
                    }
                } else {
                    $error_message = 'خطا در آپلود فایل';
                }
            }
        } else {
            $error_message = 'لطفاً فایل تصویر را انتخاب کنید';
        }
    }
}

// Get user data
$stmt = $pdo->prepare("
    SELECT u.*, 
           COALESCE(SUM(CASE WHEN t.type = 'deposit' AND t.status = 'completed' THEN t.amount ELSE 0 END), 0) as total_deposits,
           COALESCE(SUM(CASE WHEN t.type = 'withdrawal' AND t.status = 'completed' THEN t.amount ELSE 0 END), 0) as total_withdrawals,
           COALESCE(COUNT(DISTINCT b.id), 0) as total_bets,
           COALESCE(SUM(CASE WHEN b.status = 'won' THEN b.payout_amount - b.bet_amount ELSE 0 END), 0) as total_winnings
    FROM users u
    LEFT JOIN transactions t ON u.id = t.user_id
    LEFT JOIN bets b ON u.id = b.user_id
    WHERE u.id = ?
    GROUP BY u.id
");
$stmt->execute([$user_id]);
$user = $stmt->fetch();

// Get recent activity
$stmt = $pdo->prepare("
    (SELECT 'bet' as type, id, bet_amount as amount, game_type as description, status, created_at 
     FROM bets WHERE user_id = ? ORDER BY created_at DESC LIMIT 5)
    UNION ALL
    (SELECT 'transaction' as type, id, amount, type as description, status, created_at 
     FROM transactions WHERE user_id = ? ORDER BY created_at DESC LIMIT 5)
    ORDER BY created_at DESC LIMIT 10
");
$stmt->execute([$user_id, $user_id]);
$recent_activity = $stmt->fetchAll();

// Get verification status
$stmt = $pdo->prepare("SELECT * FROM user_verifications WHERE user_id = ?");
$stmt->execute([$user_id]);
$verification = $stmt->fetch();

include '../includes/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <!-- Profile Sidebar -->
        <div class="col-lg-3 col-md-4">
            <div class="card profile-sidebar">
                <div class="card-body text-center">
                    <div class="profile-avatar mb-3">
                        <?php if ($user['avatar']): ?>
                            <img src="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'uploads/avatars/<?php echo htmlspecialchars($user['avatar']); ?>" 
                                 alt="Avatar" class="rounded-circle" width="120" height="120">
                        <?php else: ?>
                            <div class="avatar-placeholder rounded-circle d-flex align-items-center justify-content-center">
                                <i class="fas fa-user fa-3x text-muted"></i>
                            </div>
                        <?php endif; ?>
                        <div class="avatar-upload mt-2">
                            <form method="POST" enctype="multipart/form-data" class="d-inline">
                                <input type="file" name="avatar" id="avatar-input" accept="image/*" style="display: none;">
                                <label for="avatar-input" class="btn btn-sm btn-outline-primary">
                                    <i class="fas fa-camera"></i> تغییر تصویر
                                </label>
                                <button type="submit" name="upload_avatar" class="btn btn-sm btn-primary" id="upload-btn" style="display: none;">
                                    آپلود
                                </button>
                            </form>
                        </div>
                    </div>
                    
                    <h4><?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?></h4>
                    <p class="text-muted"><?php echo htmlspecialchars($user['email']); ?></p>
                    
                    <div class="profile-stats mt-4">
                        <div class="row text-center">
                            <div class="col-6">
                                <div class="stat-item">
                                    <h5 class="mb-0">$<?php echo number_format($user['balance'], 2); ?></h5>
                                    <small class="text-muted">موجودی</small>
                                </div>
                            </div>
                            <div class="col-6">
                                <div class="stat-item">
                                    <h5 class="mb-0"><?php echo $user['total_bets']; ?></h5>
                                    <small class="text-muted">کل شرط‌ها</small>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Verification Status -->
                    <div class="verification-status mt-4">
                        <h6>وضعیت تأیید هویت</h6>
                        <div class="verification-items">
                            <div class="verification-item d-flex justify-content-between align-items-center">
                                <span>ایمیل</span>
                                <?php if ($user['email_verified']): ?>
                                    <span class="badge badge-success"><i class="fas fa-check"></i> تأیید شده</span>
                                <?php else: ?>
                                    <span class="badge badge-warning"><i class="fas fa-clock"></i> در انتظار</span>
                                <?php endif; ?>
                            </div>
                            <div class="verification-item d-flex justify-content-between align-items-center">
                                <span>شماره تلفن</span>
                                <?php if ($user['phone_verified']): ?>
                                    <span class="badge badge-success"><i class="fas fa-check"></i> تأیید شده</span>
                                <?php else: ?>
                                    <span class="badge badge-warning"><i class="fas fa-clock"></i> در انتظار</span>
                                <?php endif; ?>
                            </div>
                            <div class="verification-item d-flex justify-content-between align-items-center">
                                <span>مدارک هویتی</span>
                                <?php if ($verification && $verification['identity_status'] === 'approved'): ?>
                                    <span class="badge badge-success"><i class="fas fa-check"></i> تأیید شده</span>
                                <?php elseif ($verification && $verification['identity_status'] === 'pending'): ?>
                                    <span class="badge badge-warning"><i class="fas fa-clock"></i> در انتظار</span>
                                <?php else: ?>
                                    <span class="badge badge-danger"><i class="fas fa-times"></i> تأیید نشده</span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Profile Content -->
        <div class="col-lg-9 col-md-8">
            <?php if ($success_message): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    <?php echo $success_message; ?>
                    <button type="button" class="close" data-dismiss="alert">&times;</button>
                </div>
            <?php endif; ?>
            
            <?php if ($error_message): ?>
                <div class="alert alert-danger alert-dismissible fade show">
                    <?php echo $error_message; ?>
                    <button type="button" class="close" data-dismiss="alert">&times;</button>
                </div>
            <?php endif; ?>
            
            <!-- Profile Tabs -->
            <div class="card">
                <div class="card-header">
                    <ul class="nav nav-tabs card-header-tabs" role="tablist">
                        <li class="nav-item">
                            <a class="nav-link active" data-toggle="tab" href="#profile-info" role="tab">
                                <i class="fas fa-user"></i> اطلاعات شخصی
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" data-toggle="tab" href="#security" role="tab">
                                <i class="fas fa-shield-alt"></i> امنیت
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" data-toggle="tab" href="#preferences" role="tab">
                                <i class="fas fa-cog"></i> تنظیمات
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" data-toggle="tab" href="#activity" role="tab">
                                <i class="fas fa-history"></i> فعالیت‌ها
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" data-toggle="tab" href="#statistics" role="tab">
                                <i class="fas fa-chart-bar"></i> آمار
                            </a>
                        </li>
                    </ul>
                </div>
                
                <div class="card-body">
                    <div class="tab-content">
                        <!-- Profile Information Tab -->
                        <div class="tab-pane fade show active" id="profile-info" role="tabpanel">
                            <form method="POST" class="profile-form">
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label for="first_name">نام</label>
                                            <input type="text" class="form-control" id="first_name" name="first_name" 
                                                   value="<?php echo htmlspecialchars($user['first_name']); ?>" required>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label for="last_name">نام خانوادگی</label>
                                            <input type="text" class="form-control" id="last_name" name="last_name" 
                                                   value="<?php echo htmlspecialchars($user['last_name']); ?>" required>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label for="email">ایمیل</label>
                                            <input type="email" class="form-control" id="email" name="email" 
                                                   value="<?php echo htmlspecialchars($user['email']); ?>" required>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label for="phone">شماره تلفن</label>
                                            <input type="tel" class="form-control" id="phone" name="phone" 
                                                   value="<?php echo htmlspecialchars($user['phone']); ?>">
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-4">
                                        <div class="form-group">
                                            <label for="birth_date">تاریخ تولد</label>
                                            <input type="date" class="form-control" id="birth_date" name="birth_date" 
                                                   value="<?php echo $user['birth_date']; ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="form-group">
                                            <label for="country">کشور</label>
                                            <select class="form-control" id="country" name="country">
                                                <option value="">انتخاب کنید</option>
                                                <option value="IR" <?php echo $user['country'] === 'IR' ? 'selected' : ''; ?>>ایران</option>
                                                <option value="US" <?php echo $user['country'] === 'US' ? 'selected' : ''; ?>>آمریکا</option>
                                                <option value="GB" <?php echo $user['country'] === 'GB' ? 'selected' : ''; ?>>انگلستان</option>
                                                <option value="DE" <?php echo $user['country'] === 'DE' ? 'selected' : ''; ?>>آلمان</option>
                                                <option value="FR" <?php echo $user['country'] === 'FR' ? 'selected' : ''; ?>>فرانسه</option>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="form-group">
                                            <label for="city">شهر</label>
                                            <input type="text" class="form-control" id="city" name="city" 
                                                   value="<?php echo htmlspecialchars($user['city']); ?>">
                                        </div>
                                    </div>
                                </div>
                                
                                <button type="submit" name="update_profile" class="btn btn-primary">
                                    <i class="fas fa-save"></i> ذخیره تغییرات
                                </button>
                            </form>
                        </div>
                        
                        <!-- Security Tab -->
                        <div class="tab-pane fade" id="security" role="tabpanel">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="card">
                                        <div class="card-header">
                                            <h5><i class="fas fa-key"></i> تغییر رمز عبور</h5>
                                        </div>
                                        <div class="card-body">
                                            <form method="POST" class="password-form">
                                                <div class="form-group">
                                                    <label for="current_password">رمز عبور فعلی</label>
                                                    <input type="password" class="form-control" id="current_password" 
                                                           name="current_password" required>
                                                </div>
                                                <div class="form-group">
                                                    <label for="new_password">رمز عبور جدید</label>
                                                    <input type="password" class="form-control" id="new_password" 
                                                           name="new_password" required minlength="6">
                                                </div>
                                                <div class="form-group">
                                                    <label for="confirm_password">تکرار رمز عبور جدید</label>
                                                    <input type="password" class="form-control" id="confirm_password" 
                                                           name="confirm_password" required minlength="6">
                                                </div>
                                                <button type="submit" name="change_password" class="btn btn-warning">
                                                    <i class="fas fa-lock"></i> تغییر رمز عبور
                                                </button>
                                            </form>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="card">
                                        <div class="card-header">
                                            <h5><i class="fas fa-shield-alt"></i> تنظیمات امنیتی</h5>
                                        </div>
                                        <div class="card-body">
                                            <div class="security-item d-flex justify-content-between align-items-center mb-3">
                                                <div>
                                                    <strong>احراز هویت دو مرحله‌ای</strong>
                                                    <br><small class="text-muted">افزایش امنیت حساب کاربری</small>
                                                </div>
                                                <div class="custom-control custom-switch">
                                                    <input type="checkbox" class="custom-control-input" id="two-factor" 
                                                           <?php echo $user['two_factor_enabled'] ? 'checked' : ''; ?>>
                                                    <label class="custom-control-label" for="two-factor"></label>
                                                </div>
                                            </div>
                                            
                                            <div class="security-item d-flex justify-content-between align-items-center mb-3">
                                                <div>
                                                    <strong>اعلان ورود</strong>
                                                    <br><small class="text-muted">اطلاع از ورود به حساب</small>
                                                </div>
                                                <div class="custom-control custom-switch">
                                                    <input type="checkbox" class="custom-control-input" id="login-alerts" checked>
                                                    <label class="custom-control-label" for="login-alerts"></label>
                                                </div>
                                            </div>
                                            
                                            <div class="recent-logins mt-4">
                                                <h6>آخرین ورودها</h6>
                                                <div class="login-item d-flex justify-content-between">
                                                    <span>دستگاه فعلی</span>
                                                    <span class="text-success">الان</span>
                                                </div>
                                                <div class="login-item d-flex justify-content-between">
                                                    <span>Chrome - Windows</span>
                                                    <span class="text-muted">2 ساعت پیش</span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Preferences Tab -->
                        <div class="tab-pane fade" id="preferences" role="tabpanel">
                            <form method="POST" class="preferences-form">
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="card">
                                            <div class="card-header">
                                                <h5><i class="fas fa-globe"></i> تنظیمات عمومی</h5>
                                            </div>
                                            <div class="card-body">
                                                <div class="form-group">
                                                    <label for="language">زبان</label>
                                                    <select class="form-control" id="language" name="language">
                                                        <option value="fa" <?php echo $user['language'] === 'fa' ? 'selected' : ''; ?>>فارسی</option>
                                                        <option value="en" <?php echo $user['language'] === 'en' ? 'selected' : ''; ?>>English</option>
                                                    </select>
                                                </div>
                                                
                                                <div class="form-group">
                                                    <label for="timezone">منطقه زمانی</label>
                                                    <select class="form-control" id="timezone" name="timezone">
                                                        <option value="Asia/Tehran" <?php echo $user['timezone'] === 'Asia/Tehran' ? 'selected' : ''; ?>>تهران</option>
                                                        <option value="UTC" <?php echo $user['timezone'] === 'UTC' ? 'selected' : ''; ?>>UTC</option>
                                                        <option value="America/New_York" <?php echo $user['timezone'] === 'America/New_York' ? 'selected' : ''; ?>>نیویورک</option>
                                                    </select>
                                                </div>
                                                
                                                <div class="form-group">
                                                    <label for="currency">واحد پول</label>
                                                    <select class="form-control" id="currency" name="currency">
                                                        <option value="USD" <?php echo $user['currency'] === 'USD' ? 'selected' : ''; ?>>دلار آمریکا (USD)</option>
                                                        <option value="EUR" <?php echo $user['currency'] === 'EUR' ? 'selected' : ''; ?>>یورو (EUR)</option>
                                                        <option value="IRR" <?php echo $user['currency'] === 'IRR' ? 'selected' : ''; ?>>ریال ایران (IRR)</option>
                                                    </select>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="col-md-6">
                                        <div class="card">
                                            <div class="card-header">
                                                <h5><i class="fas fa-bell"></i> اعلان‌ها</h5>
                                            </div>
                                            <div class="card-body">
                                                <div class="form-group">
                                                    <div class="custom-control custom-checkbox">
                                                        <input type="checkbox" class="custom-control-input" id="notifications_email" 
                                                               name="notifications_email" <?php echo $user['notifications_email'] ? 'checked' : ''; ?>>
                                                        <label class="custom-control-label" for="notifications_email">
                                                            اعلان‌های ایمیل
                                                        </label>
                                                    </div>
                                                </div>
                                                
                                                <div class="form-group">
                                                    <div class="custom-control custom-checkbox">
                                                        <input type="checkbox" class="custom-control-input" id="notifications_sms" 
                                                               name="notifications_sms" <?php echo $user['notifications_sms'] ? 'checked' : ''; ?>>
                                                        <label class="custom-control-label" for="notifications_sms">
                                                            اعلان‌های پیامکی
                                                        </label>
                                                    </div>
                                                </div>
                                                
                                                <div class="form-group">
                                                    <div class="custom-control custom-checkbox">
                                                        <input type="checkbox" class="custom-control-input" id="notifications_push" 
                                                               name="notifications_push" <?php echo $user['notifications_push'] ? 'checked' : ''; ?>>
                                                        <label class="custom-control-label" for="notifications_push">
                                                            اعلان‌های فوری
                                                        </label>
                                                    </div>
                                                </div>
                                                
                                                <div class="form-group">
                                                    <div class="custom-control custom-checkbox">
                                                        <input type="checkbox" class="custom-control-input" id="two_factor_enabled" 
                                                               name="two_factor_enabled" <?php echo $user['two_factor_enabled'] ? 'checked' : ''; ?>>
                                                        <label class="custom-control-label" for="two_factor_enabled">
                                                            احراز هویت دو مرحله‌ای
                                                        </label>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="text-center mt-3">
                                    <button type="submit" name="update_preferences" class="btn btn-primary">
                                        <i class="fas fa-save"></i> ذخیره تنظیمات
                                    </button>
                                </div>
                            </form>
                        </div>
                        
                        <!-- Activity Tab -->
                        <div class="tab-pane fade" id="activity" role="tabpanel">
                            <div class="activity-timeline">
                                <h5><i class="fas fa-history"></i> فعالیت‌های اخیر</h5>
                                
                                <?php if (empty($recent_activity)): ?>
                                    <p class="text-muted text-center">فعالیتی یافت نشد</p>
                                <?php else: ?>
                                    <?php foreach ($recent_activity as $activity): ?>
                                        <div class="activity-item">
                                            <div class="activity-icon">
                                                <?php if ($activity['type'] === 'bet'): ?>
                                                    <i class="fas fa-dice text-primary"></i>
                                                <?php else: ?>
                                                    <i class="fas fa-exchange-alt text-success"></i>
                                                <?php endif; ?>
                                            </div>
                                            <div class="activity-content">
                                                <div class="activity-title">
                                                    <?php if ($activity['type'] === 'bet'): ?>
                                                        شرط <?php echo htmlspecialchars($activity['description']); ?>
                                                    <?php else: ?>
                                                        تراکنش <?php echo htmlspecialchars($activity['description']); ?>
                                                    <?php endif; ?>
                                                </div>
                                                <div class="activity-details">
                                                    <span class="amount">$<?php echo number_format($activity['amount'], 2); ?></span>
                                                    <span class="status badge badge-<?php echo $activity['status'] === 'completed' || $activity['status'] === 'won' ? 'success' : ($activity['status'] === 'pending' ? 'warning' : 'danger'); ?>">
                                                        <?php echo htmlspecialchars($activity['status']); ?>
                                                    </span>
                                                </div>
                                                <div class="activity-time text-muted">
                                                    <?php echo timeAgo($activity['created_at']); ?>
                                                </div>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Statistics Tab -->
                        <div class="tab-pane fade" id="statistics" role="tabpanel">
                            <div class="row">
                                <div class="col-md-3">
                                    <div class="stat-card text-center">
                                        <div class="stat-icon">
                                            <i class="fas fa-arrow-down text-success"></i>
                                        </div>
                                        <div class="stat-value">$<?php echo number_format($user['total_deposits'], 2); ?></div>
                                        <div class="stat-label">کل واریزها</div>
                                    </div>
                                </div>
                                
                                <div class="col-md-3">
                                    <div class="stat-card text-center">
                                        <div class="stat-icon">
                                            <i class="fas fa-arrow-up text-danger"></i>
                                        </div>
                                        <div class="stat-value">$<?php echo number_format($user['total_withdrawals'], 2); ?></div>
                                        <div class="stat-label">کل برداشت‌ها</div>
                                    </div>
                                </div>
                                
                                <div class="col-md-3">
                                    <div class="stat-card text-center">
                                        <div class="stat-icon">
                                            <i class="fas fa-dice text-primary"></i>
                                        </div>
                                        <div class="stat-value"><?php echo $user['total_bets']; ?></div>
                                        <div class="stat-label">کل شرط‌ها</div>
                                    </div>
                                </div>
                                
                                <div class="col-md-3">
                                    <div class="stat-card text-center">
                                        <div class="stat-icon">
                                            <i class="fas fa-trophy text-warning"></i>
                                        </div>
                                        <div class="stat-value">$<?php echo number_format($user['total_winnings'], 2); ?></div>
                                        <div class="stat-label">کل برد</div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row mt-4">
                                <div class="col-md-6">
                                    <div class="card">
                                        <div class="card-header">
                                            <h5><i class="fas fa-chart-pie"></i> توزیع بازی‌ها</h5>
                                        </div>
                                        <div class="card-body">
                                            <canvas id="gamesChart" width="400" height="200"></canvas>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="card">
                                        <div class="card-header">
                                            <h5><i class="fas fa-chart-line"></i> روند موجودی</h5>
                                        </div>
                                        <div class="card-body">
                                            <canvas id="balanceChart" width="400" height="200"></canvas>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.profile-sidebar {
    position: sticky;
    top: 20px;
}

.avatar-placeholder {
    width: 120px;
    height: 120px;
    background: #f8f9fa;
    border: 2px dashed #dee2e6;
    margin: 0 auto;
}

.profile-stats .stat-item {
    padding: 1rem 0;
    border-bottom: 1px solid #eee;
}

.profile-stats .stat-item:last-child {
    border-bottom: none;
}

.verification-items .verification-item {
    padding: 0.5rem 0;
    border-bottom: 1px solid #f8f9fa;
}

.verification-items .verification-item:last-child {
    border-bottom: none;
}

.activity-timeline {
    max-height: 500px;
    overflow-y: auto;
}

.activity-item {
    display: flex;
    align-items: flex-start;
    padding: 1rem 0;
    border-bottom: 1px solid #f8f9fa;
}

.activity-item:last-child {
    border-bottom: none;
}

.activity-icon {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: #f8f9fa;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-right: 1rem;
}

.activity-content {
    flex: 1;
}

.activity-title {
    font-weight: 500;
    margin-bottom: 0.25rem;
}

.activity-details {
    margin-bottom: 0.25rem;
}

.activity-details .amount {
    font-weight: 600;
    margin-right: 0.5rem;
}

.stat-card {
    background: #fff;
    border: 1px solid #dee2e6;
    border-radius: 0.5rem;
    padding: 1.5rem;
    margin-bottom: 1rem;
    transition: transform 0.2s;
}

.stat-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
}

.stat-icon {
    font-size: 2rem;
    margin-bottom: 0.5rem;
}

.stat-value {
    font-size: 1.5rem;
    font-weight: 600;
    margin-bottom: 0.25rem;
}

.stat-label {
    color: #6c757d;
    font-size: 0.875rem;
}

.security-item {
    padding: 1rem 0;
    border-bottom: 1px solid #f8f9fa;
}

.security-item:last-child {
    border-bottom: none;
}

.login-item {
    padding: 0.5rem 0;
    border-bottom: 1px solid #f8f9fa;
}

.login-item:last-child {
    border-bottom: none;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Avatar upload preview
    document.getElementById('avatar-input').addEventListener('change', function(e) {
        if (e.target.files && e.target.files[0]) {
            document.getElementById('upload-btn').style.display = 'inline-block';
        }
    });
    
    // Password confirmation validation
    document.getElementById('confirm_password').addEventListener('input', function() {
        const newPassword = document.getElementById('new_password').value;
        const confirmPassword = this.value;
        
        if (newPassword !== confirmPassword) {
            this.setCustomValidity('رمز عبور مطابقت ندارد');
        } else {
            this.setCustomValidity('');
        }
    });
});
</script>

<?php include '../includes/footer.php'; ?>
