
<?php
// admin/support.php - Support Management
// مدیریت پشتیبانی

require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

// Require admin authentication
requireAuth();
requireAdmin();

// Page settings
$pageTitle = 'Support Management - Admin';
$bodyClass = 'admin-support';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $ticketId = intval($_POST['ticket_id'] ?? 0);
    
    switch ($action) {
        case 'reply':
            $message = $_POST['message'] ?? '';
            if ($message && $ticketId) {
                // Add reply
                execute(
                    "INSERT INTO support_messages (ticket_id, user_id, message, is_admin, created_at) VALUES (?, ?, ?, 1, NOW())",
                    [$ticketId, $currentUser['id'], $message]
                );
                
                // Update ticket status and last response
                execute(
                    "UPDATE support_tickets SET status = 'replied', last_response_at = NOW(), assigned_to = ? WHERE id = ?",
                    [$currentUser['id'], $ticketId]
                );
                
                $successMessage = "Reply sent successfully";
            }
            break;
            
        case 'update_status':
            $status = $_POST['status'] ?? '';
            if (in_array($status, ['open', 'pending', 'replied', 'closed'])) {
                execute("UPDATE support_tickets SET status = ?, updated_at = NOW() WHERE id = ?", [$status, $ticketId]);
                $successMessage = "Ticket status updated successfully";
            }
            break;
            
        case 'assign':
            $assignTo = intval($_POST['assign_to'] ?? 0);
            execute("UPDATE support_tickets SET assigned_to = ?, updated_at = NOW() WHERE id = ?", [$assignTo, $ticketId]);
            $successMessage = "Ticket assigned successfully";
            break;
            
        case 'update_priority':
            $priority = $_POST['priority'] ?? '';
            if (in_array($priority, ['low', 'medium', 'high', 'urgent'])) {
                execute("UPDATE support_tickets SET priority = ?, updated_at = NOW() WHERE id = ?", [$priority, $ticketId]);
                $successMessage = "Priority updated successfully";
            }
            break;
    }
}

// Get filter parameters
$search = $_GET['search'] ?? '';
$status = $_GET['status'] ?? 'all';
$priority = $_GET['priority'] ?? 'all';
$category = $_GET['category'] ?? 'all';
$assigned = $_GET['assigned'] ?? 'all';
$sortBy = $_GET['sort'] ?? 'created_at';
$sortOrder = $_GET['order'] ?? 'DESC';
$page = max(1, intval($_GET['page'] ?? 1));
$limit = 25;
$offset = ($page - 1) * $limit;

// Build WHERE clause
$whereConditions = [];
$params = [];

if ($search) {
    $whereConditions[] = "(st.subject LIKE ? OR u.username LIKE ? OR u.email LIKE ?)";
    $searchTerm = "%$search%";
    $params = array_merge($params, [$searchTerm, $searchTerm, $searchTerm]);
}

if ($status !== 'all') {
    $whereConditions[] = "st.status = ?";
    $params[] = $status;
}

if ($priority !== 'all') {
    $whereConditions[] = "st.priority = ?";
    $params[] = $priority;
}

if ($category !== 'all') {
    $whereConditions[] = "st.category = ?";
    $params[] = $category;
}

if ($assigned !== 'all') {
    if ($assigned === 'unassigned') {
        $whereConditions[] = "st.assigned_to IS NULL";
    } else {
        $whereConditions[] = "st.assigned_to = ?";
        $params[] = $assigned;
    }
}

$whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';

// Validate sort parameters
$allowedSorts = ['id', 'subject', 'priority', 'status', 'created_at', 'updated_at'];
$sortBy = in_array($sortBy, $allowedSorts) ? "st.$sortBy" : 'st.created_at';
$sortOrder = strtoupper($sortOrder) === 'ASC' ? 'ASC' : 'DESC';

// Get total count
$totalTickets = fetchValue("SELECT COUNT(*) FROM support_tickets st JOIN users u ON st.user_id = u.id $whereClause", $params);

// Get tickets
$tickets = fetchAll(
    "SELECT st.*, u.username, u.email,
        (SELECT username FROM users WHERE id = st.assigned_to) as assigned_username,
        (SELECT COUNT(*) FROM support_messages WHERE ticket_id = st.id) as message_count,
        (SELECT COUNT(*) FROM support_messages WHERE ticket_id = st.id AND is_admin = 0 AND created_at > COALESCE(st.last_response_at, '1970-01-01')) as unread_count
     FROM support_tickets st 
     JOIN users u ON st.user_id = u.id 
     $whereClause 
     ORDER BY $sortBy $sortOrder 
     LIMIT $limit OFFSET $offset",
    $params
);

// Calculate pagination
$totalPages = ceil($totalTickets / $limit);

// Get support statistics
$supportStats = fetchRow(
    "SELECT 
        COUNT(*) as total_tickets,
        COUNT(CASE WHEN status = 'open' THEN 1 END) as open_tickets,
        COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_tickets,
        COUNT(CASE WHEN status = 'replied' THEN 1 END) as replied_tickets,
        COUNT(CASE WHEN status = 'closed' THEN 1 END) as closed_tickets,
        COUNT(CASE WHEN DATE(created_at) = CURDATE() THEN 1 END) as today_tickets,
        COUNT(CASE WHEN assigned_to IS NULL THEN 1 END) as unassigned_tickets,
        AVG(CASE WHEN status = 'closed' AND last_response_at IS NOT NULL THEN TIMESTAMPDIFF(HOUR, created_at, last_response_at) END) as avg_response_time
     FROM support_tickets"
);

// Get admin users for assignment
$adminUsers = fetchAll("SELECT id, username FROM users WHERE role = 'admin' AND status = 'active'");

include '../includes/header.php';
?>

<!-- Admin Support Management -->
<div class="admin-container">
    <?php include 'sidebar.php'; ?>
    
    <main class="admin-main">
        <!-- Header -->
        <header class="admin-header">
            <div class="header-left">
                <h1 class="page-title">Support Management</h1>
                <p class="page-subtitle">Manage customer support tickets and inquiries</p>
            </div>
            
            <div class="header-right">
                <button class="btn btn-outline" onclick="exportTickets()">
                    <i class="fas fa-download"></i>
                    Export CSV
                </button>
                <button class="btn btn-primary" onclick="refreshTickets()">
                    <i class="fas fa-sync-alt"></i>
                    Refresh
                </button>
            </div>
        </header>
        
        <!-- Support Statistics -->
        <section class="stats-section">
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-ticket-alt"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format($supportStats['total_tickets']); ?></div>
                        <div class="stat-label">Total Tickets</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon danger">
                        <i class="fas fa-exclamation-circle"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format($supportStats['open_tickets']); ?></div>
                        <div class="stat-label">Open Tickets</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon warning">
                        <i class="fas fa-clock"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format($supportStats['pending_tickets']); ?></div>
                        <div class="stat-label">Pending</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon success">
                        <i class="fas fa-reply"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format($supportStats['replied_tickets']); ?></div>
                        <div class="stat-label">Replied</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-user-slash"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format($supportStats['unassigned_tickets']); ?></div>
                        <div class="stat-label">Unassigned</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-stopwatch"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo round($supportStats['avg_response_time'] ?: 0, 1); ?>h</div>
                        <div class="stat-label">Avg Response Time</div>
                    </div>
                </div>
            </div>
        </section>
        
        <!-- Filters and Search -->
        <section class="filters-section">
            <form class="filters-form" method="GET">
                <div class="filter-group">
                    <div class="search-box">
                        <i class="fas fa-search"></i>
                        <input type="text" 
                               name="search" 
                               value="<?php echo htmlspecialchars($search); ?>" 
                               placeholder="Search tickets..."
                               class="search-input">
                    </div>
                </div>
                
                <div class="filter-group">
                    <select name="status" onchange="this.form.submit()">
                        <option value="all" <?php echo $status === 'all' ? 'selected' : ''; ?>>All Status</option>
                        <option value="open" <?php echo $status === 'open' ? 'selected' : ''; ?>>Open</option>
                        <option value="pending" <?php echo $status === 'pending' ? 'selected' : ''; ?>>Pending</option>
                        <option value="replied" <?php echo $status === 'replied' ? 'selected' : ''; ?>>Replied</option>
                        <option value="closed" <?php echo $status === 'closed' ? 'selected' : ''; ?>>Closed</option>
                    </select>
                </div>
                
                <div class="filter-group">
                    <select name="priority" onchange="this.form.submit()">
                        <option value="all" <?php echo $priority === 'all' ? 'selected' : ''; ?>>All Priority</option>
                        <option value="low" <?php echo $priority === 'low' ? 'selected' : ''; ?>>Low</option>
                        <option value="medium" <?php echo $priority === 'medium' ? 'selected' : ''; ?>>Medium</option>
                        <option value="high" <?php echo $priority === 'high' ? 'selected' : ''; ?>>High</option>
                        <option value="urgent" <?php echo $priority === 'urgent' ? 'selected' : ''; ?>>Urgent</option>
                    </select>
                </div>
                
                <div class="filter-group">
                    <select name="assigned" onchange="this.form.submit()">
                        <option value="all" <?php echo $assigned === 'all' ? 'selected' : ''; ?>>All Assigned</option>
                        <option value="unassigned" <?php echo $assigned === 'unassigned' ? 'selected' : ''; ?>>Unassigned</option>
                        <?php foreach ($adminUsers as $admin): ?>
                        <option value="<?php echo $admin['id']; ?>" <?php echo $assigned == $admin['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($admin['username']); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-filter"></i>
                    Filter
                </button>
                
                <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'admin/support.php" class="btn btn-outline">
                    <i class="fas fa-times"></i>
                    Clear
                </a>
            </form>
            
            <div class="results-info">
                Showing <?php echo count($tickets); ?> of <?php echo number_format($totalTickets); ?> tickets
            </div>
        </section>
        
        <!-- Tickets Table -->
        <section class="table-section">
            <div class="table-container">
                <table class="admin-table">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Subject</th>
                            <th>User</th>
                            <th>Category</th>
                            <th>Priority</th>
                            <th>Status</th>
                            <th>Assigned</th>
                            <th>Messages</th>
                            <th>Created</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($tickets as $ticket): ?>
                        <tr class="ticket-row <?php echo $ticket['unread_count'] > 0 ? 'has-unread' : ''; ?>" data-ticket-id="<?php echo $ticket['id']; ?>">
                            <td>
                                <div class="ticket-id">
                                    <strong>#<?php echo $ticket['id']; ?></strong>
                                    <?php if ($ticket['unread_count'] > 0): ?>
                                    <span class="unread-badge"><?php echo $ticket['unread_count']; ?></span>
                                    <?php endif; ?>
                                </div>
                            </td>
                            
                            <td>
                                <div class="ticket-subject">
                                    <a href="#" onclick="viewTicket(<?php echo $ticket['id']; ?>)" class="subject-link">
                                        <?php echo htmlspecialchars($ticket['subject']); ?>
                                    </a>
                                </div>
                            </td>
                            
                            <td>
                                <div class="user-info">
                                    <div class="username"><?php echo htmlspecialchars($ticket['username']); ?></div>
                                    <div class="email"><?php echo htmlspecialchars($ticket['email']); ?></div>
                                </div>
                            </td>
                            
                            <td>
                                <span class="category-badge">
                                    <?php echo ucfirst($ticket['category']); ?>
                                </span>
                            </td>
                            
                            <td>
                                <span class="priority-badge priority-<?php echo $ticket['priority']; ?>">
                                    <?php echo ucfirst($ticket['priority']); ?>
                                </span>
                            </td>
                            
                            <td>
                                <span class="status-badge status-<?php echo $ticket['status']; ?>">
                                    <?php echo ucfirst($ticket['status']); ?>
                                </span>
                            </td>
                            
                            <td>
                                <div class="assigned-info">
                                    <?php if ($ticket['assigned_username']): ?>
                                    <span class="assigned-user"><?php echo htmlspecialchars($ticket['assigned_username']); ?></span>
                                    <?php else: ?>
                                    <span class="unassigned">Unassigned</span>
                                    <?php endif; ?>
                                </div>
                            </td>
                            
                            <td>
                                <div class="message-count">
                                    <i class="fas fa-comments"></i>
                                    <?php echo $ticket['message_count']; ?>
                                </div>
                            </td>
                            
                            <td>
                                <div class="date-info">
                                    <div class="date"><?php echo date('M j, Y', strtotime($ticket['created_at'])); ?></div>
                                    <div class="time"><?php echo date('H:i', strtotime($ticket['created_at'])); ?></div>
                                </div>
                            </td>
                            
                            <td>
                                <div class="action-buttons">
                                    <button class="btn btn-primary btn-xs" onclick="viewTicket(<?php echo $ticket['id']; ?>)">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                    
                                    <div class="dropdown">
                                        <button class="btn btn-outline btn-xs dropdown-toggle">
                                            <i class="fas fa-ellipsis-v"></i>
                                        </button>
                                        <div class="dropdown-menu">
                                            <a href="#" onclick="assignTicket(<?php echo $ticket['id']; ?>)">Assign</a>
                                            <a href="#" onclick="updatePriority(<?php echo $ticket['id']; ?>)">Change Priority</a>
                                            <a href="#" onclick="updateStatus(<?php echo $ticket['id']; ?>, 'closed')">Close Ticket</a>
                                            <div class="dropdown-divider"></div>
                                            <a href="#" onclick="viewUserTickets(<?php echo $ticket['user_id']; ?>)">User's Tickets</a>
                                        </div>
                                    </div>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            <?php if ($totalPages > 1): ?>
            <div class="pagination-container">
                <nav class="pagination">
                    <?php if ($page > 1): ?>
                    <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>" class="page-link prev">
                        <i class="fas fa-chevron-left"></i>
                        Previous
                    </a>
                    <?php endif; ?>
                    
                    <?php
                    $startPage = max(1, $page - 2);
                    $endPage = min($totalPages, $page + 2);
                    
                    for ($i = $startPage; $i <= $endPage; $i++): ?>
                    <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>" 
                       class="page-link <?php echo $i === $page ? 'active' : ''; ?>">
                        <?php echo $i; ?>
                    </a>
                    <?php endfor; ?>
                    
                    <?php if ($page < $totalPages): ?>
                    <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>" class="page-link next">
                        Next
                        <i class="fas fa-chevron-right"></i>
                    </a>
                    <?php endif; ?>
                </nav>
            </div>
            <?php endif; ?>
        </section>
    </main>
</div>

<!-- Ticket Details Modal -->
<div class="modal-backdrop" id="ticketModal">
    <div class="modal ticket-modal large">
        <div class="modal-header">
            <h3 class="modal-title">Ticket Details</h3>
            <button class="modal-close" onclick="closeModal('ticketModal')">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="modal-body" id="ticketModalContent">
            <!-- Ticket details will be loaded here -->
        </div>
    </div>
</div>

<script>
// Support management functions
function viewTicket(ticketId) {
    fetch(`/api/admin/support/${ticketId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showTicketDetails(data.ticket, data.messages);
            }
        })
        .catch(error => {
            console.error('Error:', error);
        });
}

function showTicketDetails(ticket, messages) {
    const content = `
        <div class="ticket-details">
            <div class="ticket-header">
                <div class="ticket-info">
                    <h4>${ticket.subject}</h4>
                    <div class="ticket-meta">
                        <span class="ticket-id">#${ticket.id}</span>
                        <span class="status-badge status-${ticket.status}">${ticket.status}</span>
                        <span class="priority-badge priority-${ticket.priority}">${ticket.priority}</span>
                        <span class="category-badge">${ticket.category}</span>
                    </div>
                </div>
                <div class="ticket-actions">
                    <select onchange="updateTicketStatus(${ticket.id}, this.value)">
                        <option value="">Change Status</option>
                        <option value="open">Open</option>
                        <option value="pending">Pending</option>
                        <option value="replied">Replied</option>
                        <option value="closed">Closed</option>
                    </select>
                </div>
            </div>
            
            <div class="messages-container">
                ${messages.map(message => `
                    <div class="message ${message.is_admin ? 'admin-message' : 'user-message'}">
                        <div class="message-header">
                            <div class="message-author">
                                <i class="fas fa-${message.is_admin ? 'user-shield' : 'user'}"></i>
                                ${message.username}
                            </div>
                            <div class="message-time">
                                ${new Date(message.created_at).toLocaleString()}
                            </div>
                        </div>
                        <div class="message-content">
                            ${message.message.replace(/\n/g, '<br>')}
                        </div>
                    </div>
                `).join('')}
            </div>
            
            <div class="reply-form">
                <form onsubmit="sendReply(event, ${ticket.id})">
                    <div class="form-group">
                        <textarea name="message" rows="4" placeholder="Type your reply..." required></textarea>
                    </div>
                    <div class="form-actions">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-reply"></i>
                            Send Reply
                        </button>
                    </div>
                </form>
            </div>
        </div>
    `;
    
    document.getElementById('ticketModalContent').innerHTML = content;
    openModal('ticketModal');
}

function sendReply(event, ticketId) {
    event.preventDefault();
    
    const formData = new FormData(event.target);
    formData.append('action', 'reply');
    formData.append('ticket_id', ticketId);
    
    fetch('/admin/support.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.text())
    .then(() => {
        // Refresh ticket details
        viewTicket(ticketId);
        // Refresh page to update ticket list
        setTimeout(() => window.location.reload(), 1000);
    })
    .catch(error => {
        console.error('Error:', error);
    });
}

function updateTicketStatus(ticketId, status) {
    if (!status) return;
    
    const formData = new FormData();
    formData.append('action', 'update_status');
    formData.append('ticket_id', ticketId);
    formData.append('status', status);
    
    fetch('/admin/support.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.text())
    .then(() => {
        window.location.reload();
    })
    .catch(error => {
        console.error('Error:', error);
    });
}

function assignTicket(ticketId) {
    // Show assignment modal or dropdown
    console.log('Assign ticket:', ticketId);
}

function updatePriority(ticketId) {
    const priority = prompt('Enter new priority (low, medium, high, urgent):');
    if (priority && ['low', 'medium', 'high', 'urgent'].includes(priority)) {
        const formData = new FormData();
        formData.append('action', 'update_priority');
        formData.append('ticket_id', ticketId);
        formData.append('priority', priority);
        
        fetch('/admin/support.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.text())
        .then(() => {
            window.location.reload();
        })
        .catch(error => {
            console.error('Error:', error);
        });
    }
}

function updateStatus(ticketId, status) {
    updateTicketStatus(ticketId, status);
}

function exportTickets() {
    const params = new URLSearchParams(window.location.search);
    params.set('export', 'csv');
    window.open(`/api/admin/support/export?${params.toString()}`, '_blank');
}

function refreshTickets() {
    window.location.reload();
}
</script>

<?php include '../includes/footer.php'; ?>
