<?php
// admin/index.php - Admin Dashboard
// داشبورد مدیریت

require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

// Require admin authentication
requireAuth();
requireAdmin();

// Page settings
$pageTitle = 'Admin Dashboard - Crashify';
$bodyClass = 'admin-dashboard';

// Get dashboard statistics
$stats = [
    'users' => [
        'total' => fetchValue("SELECT COUNT(*) FROM users"),
        'active' => fetchValue("SELECT COUNT(*) FROM users WHERE status = 'active'"),
        'new_today' => fetchValue("SELECT COUNT(*) FROM users WHERE DATE(created_at) = CURDATE()"),
        'online' => fetchValue("SELECT COUNT(*) FROM users WHERE last_activity > DATE_SUB(NOW(), INTERVAL 15 MINUTE)")
    ],
    'transactions' => [
        'total' => fetchValue("SELECT COUNT(*) FROM transactions"),
        'today' => fetchValue("SELECT COUNT(*) FROM transactions WHERE DATE(created_at) = CURDATE()"),
        'volume_today' => fetchValue("SELECT SUM(amount) FROM transactions WHERE DATE(created_at) = CURDATE() AND status = 'completed'") ?: 0,
        'pending' => fetchValue("SELECT COUNT(*) FROM transactions WHERE status = 'pending'")
    ],
    'bets' => [
        'total' => fetchValue("SELECT COUNT(*) FROM user_bets"),
        'today' => fetchValue("SELECT COUNT(*) FROM user_bets WHERE DATE(created_at) = CURDATE()"),
        'volume_today' => fetchValue("SELECT SUM(bet_amount) FROM user_bets WHERE DATE(created_at) = CURDATE()") ?: 0,
        'active' => fetchValue("SELECT COUNT(*) FROM user_bets WHERE status = 'pending'")
    ],
    'support' => [
        'total_tickets' => fetchValue("SELECT COUNT(*) FROM support_tickets"),
        'open_tickets' => fetchValue("SELECT COUNT(*) FROM support_tickets WHERE status IN ('open', 'pending')"),
        'new_today' => fetchValue("SELECT COUNT(*) FROM support_tickets WHERE DATE(created_at) = CURDATE()"),
        'avg_response_time' => fetchValue("SELECT AVG(TIMESTAMPDIFF(HOUR, created_at, updated_at)) FROM support_tickets WHERE status = 'closed'") ?: 0
    ]
];

// Get recent activities
$recentUsers = fetchAll(
    "SELECT id, username, email, created_at, status 
     FROM users 
     ORDER BY created_at DESC 
     LIMIT 10"
);

$recentTransactions = fetchAll(
    "SELECT t.*, u.username 
     FROM transactions t
     JOIN users u ON t.user_id = u.id
     ORDER BY t.created_at DESC 
     LIMIT 10"
);

$recentTickets = fetchAll(
    "SELECT st.*, u.username 
     FROM support_tickets st
     JOIN users u ON st.user_id = u.id
     ORDER BY st.created_at DESC 
     LIMIT 10"
);

// Get system status
$systemStatus = [
    'database' => checkDatabaseConnection(),
    'cache' => checkCacheStatus(),
    'storage' => checkStorageSpace(),
    'memory' => getMemoryUsage()
];

// Get revenue chart data (last 30 days)
$revenueData = fetchAll(
    "SELECT 
        DATE(created_at) as date,
        SUM(CASE WHEN type = 'deposit' THEN amount ELSE 0 END) as deposits,
        SUM(CASE WHEN type = 'withdrawal' THEN amount ELSE 0 END) as withdrawals,
        COUNT(*) as transactions
     FROM transactions 
     WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
     AND status = 'completed'
     GROUP BY DATE(created_at)
     ORDER BY date DESC"
);

include '../includes/header.php';
?>

<!-- Admin Dashboard -->
<div class="admin-container">
    <!-- Admin Sidebar -->
    <aside class="admin-sidebar">
        <div class="sidebar-header">
            <div class="admin-logo">
                <i class="fas fa-shield-alt"></i>
                <span>Admin Panel</span>
            </div>
        </div>
        
        <nav class="sidebar-nav">
            <ul class="nav-menu">
                <li class="nav-item active">
                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'admin/index.php" class="nav-link">
                        <i class="fas fa-tachometer-alt"></i>
                        <span>Dashboard</span>
                    </a>
                </li>
                <li class="nav-item">
                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'admin/users.php" class="nav-link">
                        <i class="fas fa-users"></i>
                        <span>Users</span>
                        <span class="nav-badge"><?php echo number_format($stats['users']['total']); ?></span>
                    </a>
                </li>
                <li class="nav-item">
                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'admin/transactions.php" class="nav-link">
                        <i class="fas fa-exchange-alt"></i>
                        <span>Transactions</span>
                        <?php if ($stats['transactions']['pending'] > 0): ?>
                        <span class="nav-badge warning"><?php echo $stats['transactions']['pending']; ?></span>
                        <?php endif; ?>
                    </a>
                </li>
                <li class="nav-item">
                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'admin/support.php" class="nav-link">
                        <i class="fas fa-headset"></i>
                        <span>Support</span>
                        <?php if ($stats['support']['open_tickets'] > 0): ?>
                        <span class="nav-badge danger"><?php echo $stats['support']['open_tickets']; ?></span>
                        <?php endif; ?>
                    </a>
                </li>
                <li class="nav-item">
                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'admin/content.php" class="nav-link">
                        <i class="fas fa-edit"></i>
                        <span>Content</span>
                    </a>
                </li>
                <li class="nav-item">
                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'admin/settings.php" class="nav-link">
                        <i class="fas fa-cog"></i>
                        <span>Settings</span>
                    </a>
                </li>
                <li class="nav-item">
                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'admin/reports.php" class="nav-link">
                        <i class="fas fa-chart-line"></i>
                        <span>Reports</span>
                    </a>
                </li>
            </ul>
        </nav>
        
        <div class="sidebar-footer">
            <div class="admin-user">
                <div class="user-avatar">
                    <i class="fas fa-user-shield"></i>
                </div>
                <div class="user-info">
                    <div class="user-name"><?php echo htmlspecialchars($currentUser['username']); ?></div>
                    <div class="user-role">Administrator</div>
                </div>
            </div>
        </div>
    </aside>
    
    <!-- Main Content -->
    <main class="admin-main">
        <!-- Top Bar -->
        <header class="admin-header">
            <div class="header-left">
                <button class="sidebar-toggle" onclick="toggleSidebar()">
                    <i class="fas fa-bars"></i>
                </button>
                <h1 class="page-title">Dashboard</h1>
            </div>
            
            <div class="header-right">
                <div class="header-stats">
                    <div class="stat-item">
                        <span class="stat-label">Online Users:</span>
                        <span class="stat-value"><?php echo number_format($stats['users']['online']); ?></span>
                    </div>
                    <div class="stat-item">
                        <span class="stat-label">Server Time:</span>
                        <span class="stat-value" id="serverTime"><?php echo date('H:i:s'); ?></span>
                    </div>
                </div>
                
                <div class="header-actions">
                    <button class="btn btn-outline" onclick="refreshDashboard()">
                        <i class="fas fa-sync-alt"></i>
                        Refresh
                    </button>
                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'pages/dashboard.php" class="btn btn-primary">
                        <i class="fas fa-external-link-alt"></i>
                        View Site
                    </a>
                </div>
            </div>
        </header>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <!-- Statistics Cards -->
            <section class="stats-section">
                <div class="stats-grid">
                    <!-- Users Stats -->
                    <div class="stat-card">
                        <div class="card-header">
                            <div class="card-icon users">
                                <i class="fas fa-users"></i>
                            </div>
                            <div class="card-info">
                                <h3>Users</h3>
                                <div class="card-value"><?php echo number_format($stats['users']['total']); ?></div>
                            </div>
                        </div>
                        <div class="card-details">
                            <div class="detail-item">
                                <span class="label">Active:</span>
                                <span class="value"><?php echo number_format($stats['users']['active']); ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="label">New Today:</span>
                                <span class="value success">+<?php echo number_format($stats['users']['new_today']); ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="label">Online:</span>
                                <span class="value"><?php echo number_format($stats['users']['online']); ?></span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Transactions Stats -->
                    <div class="stat-card">
                        <div class="card-header">
                            <div class="card-icon transactions">
                                <i class="fas fa-exchange-alt"></i>
                            </div>
                            <div class="card-info">
                                <h3>Transactions</h3>
                                <div class="card-value"><?php echo number_format($stats['transactions']['total']); ?></div>
                            </div>
                        </div>
                        <div class="card-details">
                            <div class="detail-item">
                                <span class="label">Today:</span>
                                <span class="value"><?php echo number_format($stats['transactions']['today']); ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="label">Volume:</span>
                                <span class="value">$<?php echo number_format($stats['transactions']['volume_today'], 2); ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="label">Pending:</span>
                                <span class="value warning"><?php echo number_format($stats['transactions']['pending']); ?></span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Bets Stats -->
                    <div class="stat-card">
                        <div class="card-header">
                            <div class="card-icon bets">
                                <i class="fas fa-dice"></i>
                            </div>
                            <div class="card-info">
                                <h3>Bets</h3>
                                <div class="card-value"><?php echo number_format($stats['bets']['total']); ?></div>
                            </div>
                        </div>
                        <div class="card-details">
                            <div class="detail-item">
                                <span class="label">Today:</span>
                                <span class="value"><?php echo number_format($stats['bets']['today']); ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="label">Volume:</span>
                                <span class="value">$<?php echo number_format($stats['bets']['volume_today'], 2); ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="label">Active:</span>
                                <span class="value"><?php echo number_format($stats['bets']['active']); ?></span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Support Stats -->
                    <div class="stat-card">
                        <div class="card-header">
                            <div class="card-icon support">
                                <i class="fas fa-headset"></i>
                            </div>
                            <div class="card-info">
                                <h3>Support</h3>
                                <div class="card-value"><?php echo number_format($stats['support']['total_tickets']); ?></div>
                            </div>
                        </div>
                        <div class="card-details">
                            <div class="detail-item">
                                <span class="label">Open:</span>
                                <span class="value danger"><?php echo number_format($stats['support']['open_tickets']); ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="label">New Today:</span>
                                <span class="value"><?php echo number_format($stats['support']['new_today']); ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="label">Avg Response:</span>
                                <span class="value"><?php echo round($stats['support']['avg_response_time'], 1); ?>h</span>
                            </div>
                        </div>
                    </div>
                </div>
            </section>
            
            <!-- Charts Section -->
            <section class="charts-section">
                <div class="charts-grid">
                    <!-- Revenue Chart -->
                    <div class="chart-card">
                        <div class="card-header">
                            <h3>Revenue Overview (Last 30 Days)</h3>
                            <div class="chart-controls">
                                <select id="chartPeriod" onchange="updateChart()">
                                    <option value="30">Last 30 Days</option>
                                    <option value="7">Last 7 Days</option>
                                    <option value="1">Today</option>
                                </select>
                            </div>
                        </div>
                        <div class="chart-container">
                            <canvas id="revenueChart" width="800" height="300"></canvas>
                        </div>
                    </div>
                    
                    <!-- System Status -->
                    <div class="status-card">
                        <div class="card-header">
                            <h3>System Status</h3>
                            <div class="status-indicator <?php echo $systemStatus['database'] ? 'online' : 'offline'; ?>">
                                <i class="fas fa-circle"></i>
                                <?php echo $systemStatus['database'] ? 'Online' : 'Offline'; ?>
                            </div>
                        </div>
                        <div class="status-list">
                            <div class="status-item">
                                <div class="status-info">
                                    <i class="fas fa-database"></i>
                                    <span>Database</span>
                                </div>
                                <div class="status-value <?php echo $systemStatus['database'] ? 'success' : 'danger'; ?>">
                                    <?php echo $systemStatus['database'] ? 'Connected' : 'Disconnected'; ?>
                                </div>
                            </div>
                            
                            <div class="status-item">
                                <div class="status-info">
                                    <i class="fas fa-memory"></i>
                                    <span>Memory Usage</span>
                                </div>
                                <div class="status-value">
                                    <?php echo $systemStatus['memory']['percentage']; ?>%
                                </div>
                            </div>
                            
                            <div class="status-item">
                                <div class="status-info">
                                    <i class="fas fa-hdd"></i>
                                    <span>Storage</span>
                                </div>
                                <div class="status-value">
                                    <?php echo $systemStatus['storage']['percentage']; ?>%
                                </div>
                            </div>
                            
                            <div class="status-item">
                                <div class="status-info">
                                    <i class="fas fa-tachometer-alt"></i>
                                    <span>Cache</span>
                                </div>
                                <div class="status-value <?php echo $systemStatus['cache'] ? 'success' : 'warning'; ?>">
                                    <?php echo $systemStatus['cache'] ? 'Active' : 'Inactive'; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </section>
            
            <!-- Recent Activities -->
            <section class="activities-section">
                <div class="activities-grid">
                    <!-- Recent Users -->
                    <div class="activity-card">
                        <div class="card-header">
                            <h3>Recent Users</h3>
                            <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'admin/users.php" class="view-all">View All</a>
                        </div>
                        <div class="activity-list">
                            <?php foreach ($recentUsers as $user): ?>
                            <div class="activity-item">
                                <div class="activity-avatar">
                                    <i class="fas fa-user"></i>
                                </div>
                                <div class="activity-info">
                                    <div class="activity-title"><?php echo htmlspecialchars($user['username']); ?></div>
                                    <div class="activity-meta"><?php echo htmlspecialchars($user['email']); ?></div>
                                </div>
                                <div class="activity-time">
                                    <?php echo timeAgo($user['created_at']); ?>
                                </div>
                                <div class="activity-status">
                                    <span class="status-badge status-<?php echo $user['status']; ?>">
                                        <?php echo ucfirst($user['status']); ?>
                                    </span>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    
                    <!-- Recent Transactions -->
                    <div class="activity-card">
                        <div class="card-header">
                            <h3>Recent Transactions</h3>
                            <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'admin/transactions.php" class="view-all">View All</a>
                        </div>
                        <div class="activity-list">
                            <?php foreach ($recentTransactions as $transaction): ?>
                            <div class="activity-item">
                                <div class="activity-avatar">
                                    <i class="fas fa-<?php echo $transaction['type'] === 'deposit' ? 'arrow-down' : 'arrow-up'; ?>"></i>
                                </div>
                                <div class="activity-info">
                                    <div class="activity-title"><?php echo htmlspecialchars($transaction['username']); ?></div>
                                    <div class="activity-meta"><?php echo ucfirst($transaction['type']); ?> - $<?php echo number_format($transaction['amount'], 2); ?></div>
                                </div>
                                <div class="activity-time">
                                    <?php echo timeAgo($transaction['created_at']); ?>
                                </div>
                                <div class="activity-status">
                                    <span class="status-badge status-<?php echo $transaction['status']; ?>">
                                        <?php echo ucfirst($transaction['status']); ?>
                                    </span>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    
                    <!-- Recent Support Tickets -->
                    <div class="activity-card">
                        <div class="card-header">
                            <h3>Recent Support Tickets</h3>
                            <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '') . '/'admin/support.php" class="view-all">View All</a>
                        </div>
                        <div class="activity-list">
                            <?php foreach ($recentTickets as $ticket): ?>
                            <div class="activity-item">
                                <div class="activity-avatar">
                                    <i class="fas fa-ticket-alt"></i>
                                </div>
                                <div class="activity-info">
                                    <div class="activity-title"><?php echo htmlspecialchars($ticket['subject']); ?></div>
                                    <div class="activity-meta">by <?php echo htmlspecialchars($ticket['username']); ?></div>
                                </div>
                                <div class="activity-time">
                                    <?php echo timeAgo($ticket['created_at']); ?>
                                </div>
                                <div class="activity-status">
                                    <span class="status-badge status-<?php echo $ticket['status']; ?>">
                                        <?php echo ucfirst($ticket['status']); ?>
                                    </span>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </section>
        </div>
    </main>
</div>

<script>
// Initialize dashboard
document.addEventListener('DOMContentLoaded', function() {
    initializeCharts();
    startRealTimeUpdates();
    updateServerTime();
});

// Initialize charts
function initializeCharts() {
    const ctx = document.getElementById('revenueChart').getContext('2d');
    
    const revenueData = <?php echo json_encode($revenueData); ?>;
    
    new Chart(ctx, {
        type: 'line',
        data: {
            labels: revenueData.map(item => item.date),
            datasets: [{
                label: 'Deposits',
                data: revenueData.map(item => item.deposits),
                borderColor: '#00ff88',
                backgroundColor: 'rgba(0, 255, 136, 0.1)',
                tension: 0.4
            }, {
                label: 'Withdrawals',
                data: revenueData.map(item => item.withdrawals),
                borderColor: '#ff4757',
                backgroundColor: 'rgba(255, 71, 87, 0.1)',
                tension: 0.4
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function(value) {
                            return '$' + value.toLocaleString();
                        }
                    }
                }
            },
            plugins: {
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            return context.dataset.label + ': $' + context.parsed.y.toLocaleString();
                        }
                    }
                }
            }
        }
    });
}

// Start real-time updates
function startRealTimeUpdates() {
    // Update every 30 seconds
    setInterval(() => {
        updateDashboardStats();
    }, 30000);
}

// Update server time
function updateServerTime() {
    setInterval(() => {
        const now = new Date();
        document.getElementById('serverTime').textContent = now.toLocaleTimeString();
    }, 1000);
}

// Update dashboard statistics
function updateDashboardStats() {
    fetch('/api/admin/stats')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                updateStatsDisplay(data.stats);
            }
        })
        .catch(error => {
            console.error('Failed to update stats:', error);
        });
}

// Update stats display
function updateStatsDisplay(stats) {
    // Update online users count
    const onlineUsersElement = document.querySelector('.header-stats .stat-value');
    if (onlineUsersElement) {
        onlineUsersElement.textContent = stats.users.online.toLocaleString();
    }
    
    // Update other stats as needed
    // This would update various elements on the page
}

// Refresh dashboard
function refreshDashboard() {
    window.location.reload();
}

// Toggle sidebar
function toggleSidebar() {
    document.querySelector('.admin-container').classList.toggle('sidebar-collapsed');
}

// Update chart period
function updateChart() {
    const period = document.getElementById('chartPeriod').value;
    
    fetch(`/api/admin/revenue-data?period=${period}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Update chart with new data
                updateChartData(data.chartData);
            }
        })
        .catch(error => {
            console.error('Failed to update chart:', error);
        });
}

function updateChartData(newData) {
    // This would update the chart with new data
    console.log('Updating chart with:', newData);
}
</script>

<?php
// Helper functions for system status
function checkDatabaseConnection() {
    try {
        global $pdo;
        $pdo->query("SELECT 1");
        return true;
    } catch (Exception $e) {
        return false;
    }
}

function checkCacheStatus() {
    // Check if cache is working (Redis, Memcached, etc.)
    return true; // Simplified for demo
}

function checkStorageSpace() {
    $bytes = disk_free_space(".");
    $total = disk_total_space(".");
    $used = $total - $bytes;
    $percentage = round(($used / $total) * 100, 1);
    
    return [
        'free' => $bytes,
        'total' => $total,
        'used' => $used,
        'percentage' => $percentage
    ];
}

function getMemoryUsage() {
    $memory = memory_get_usage(true);
    $peak = memory_get_peak_usage(true);
    $limit = ini_get('memory_limit');
    
    // Convert limit to bytes
    $limitBytes = convertToBytes($limit);
    $percentage = round(($memory / $limitBytes) * 100, 1);
    
    return [
        'current' => $memory,
        'peak' => $peak,
        'limit' => $limitBytes,
        'percentage' => $percentage
    ];
}

function convertToBytes($val) {
    $val = trim($val);
    $last = strtolower($val[strlen($val)-1]);
    $val = (int) $val;
    
    switch($last) {
        case 'g':
            $val *= 1024;
        case 'm':
            $val *= 1024;
        case 'k':
            $val *= 1024;
    }
    
    return $val;
}

function timeAgo($datetime) {
    $time = time() - strtotime($datetime);
    
    if ($time < 60) return 'just now';
    if ($time < 3600) return floor($time/60) . 'm ago';
    if ($time < 86400) return floor($time/3600) . 'h ago';
    if ($time < 2592000) return floor($time/86400) . 'd ago';
    
    return date('M j, Y', strtotime($datetime));
}

include '../includes/footer.php';
?>
