<?php
// admin/content.php - Content Management
// مدیریت محتوا

require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

// Require admin authentication
requireAuth();
requireAdmin();

// Page settings
$pageTitle = 'Content Management - Admin';
$bodyClass = 'admin-content';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'update_page':
            $pageId = $_POST['page_id'] ?? '';
            $title = $_POST['title'] ?? '';
            $content = $_POST['content'] ?? '';
            $metaDescription = $_POST['meta_description'] ?? '';
            $status = $_POST['status'] ?? 'draft';
            
            if ($pageId && $title && $content) {
                execute(
                    "INSERT INTO cms_pages (page_id, title, content, meta_description, status, updated_at, updated_by) 
                     VALUES (?, ?, ?, ?, ?, NOW(), ?)
                     ON DUPLICATE KEY UPDATE 
                     title = VALUES(title), 
                     content = VALUES(content), 
                     meta_description = VALUES(meta_description),
                     status = VALUES(status),
                     updated_at = NOW(), 
                     updated_by = VALUES(updated_by)",
                    [$pageId, $title, $content, $metaDescription, $status, $currentUser['id']]
                );
                $successMessage = "Page updated successfully";
            }
            break;
            
        case 'add_promotion':
            $title = $_POST['promo_title'] ?? '';
            $description = $_POST['promo_description'] ?? '';
            $type = $_POST['promo_type'] ?? '';
            $value = $_POST['promo_value'] ?? '';
            $code = $_POST['promo_code'] ?? '';
            $startDate = $_POST['start_date'] ?? '';
            $endDate = $_POST['end_date'] ?? '';
            $status = $_POST['promo_status'] ?? 'active';
            
            if ($title && $description && $type && $value) {
                execute(
                    "INSERT INTO promotions (title, description, type, value, code, start_date, end_date, status, created_at, created_by) 
                     VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), ?)",
                    [$title, $description, $type, $value, $code, $startDate, $endDate, $status, $currentUser['id']]
                );
                $successMessage = "Promotion added successfully";
            }
            break;
            
        case 'add_news':
            $title = $_POST['news_title'] ?? '';
            $content = $_POST['news_content'] ?? '';
            $status = $_POST['news_status'] ?? 'draft';
            
            if ($title && $content) {
                execute(
                    "INSERT INTO news (title, content, status, created_at, created_by) 
                     VALUES (?, ?, ?, NOW(), ?)",
                    [$title, $content, $status, $currentUser['id']]
                );
                $successMessage = "News article added successfully";
            }
            break;
    }
}

// Get current tab
$tab = $_GET['tab'] ?? 'pages';

// Get CMS pages
$cmsPages = fetchAll(
    "SELECT cp.*, u.username as updated_by_username 
     FROM cms_pages cp 
     LEFT JOIN users u ON cp.updated_by = u.id 
     ORDER BY cp.updated_at DESC"
);

// Get promotions
$promotions = fetchAll(
    "SELECT p.*, u.username as created_by_username 
     FROM promotions p 
     LEFT JOIN users u ON p.created_by = u.id 
     ORDER BY p.created_at DESC"
);

// Get news/announcements
$news = fetchAll(
    "SELECT n.*, u.username as created_by_username 
     FROM news n 
     LEFT JOIN users u ON n.created_by = u.id 
     ORDER BY n.created_at DESC 
     LIMIT 20"
);

// Predefined pages
$predefinedPages = [
    'home' => 'Home Page',
    'about' => 'About Us',
    'terms' => 'Terms of Service',
    'privacy' => 'Privacy Policy',
    'faq' => 'FAQ',
    'contact' => 'Contact Us',
    'responsible-gambling' => 'Responsible Gambling',
    'help' => 'Help Center'
];

include '../includes/header.php';
?>

<!-- Admin Content Management -->
<div class="admin-container">
    <?php include 'sidebar.php'; ?>
    
    <main class="admin-main">
        <!-- Header -->
        <header class="admin-header">
            <div class="header-left">
                <h1 class="page-title">Content Management</h1>
                <p class="page-subtitle">Manage website content, pages, and promotions</p>
            </div>
            
            <div class="header-right">
                <button class="btn btn-outline" onclick="previewSite()">
                    <i class="fas fa-external-link-alt"></i>
                    Preview Site
                </button>
            </div>
        </header>
        
        <!-- Content Tabs -->
        <section class="tabs-section">
            <nav class="content-tabs">
                <a href="?tab=pages" class="tab-link <?php echo $tab === 'pages' ? 'active' : ''; ?>">
                    <i class="fas fa-file-alt"></i>
                    Pages
                </a>
                <a href="?tab=promotions" class="tab-link <?php echo $tab === 'promotions' ? 'active' : ''; ?>">
                    <i class="fas fa-tags"></i>
                    Promotions
                </a>
                <a href="?tab=news" class="tab-link <?php echo $tab === 'news' ? 'active' : ''; ?>">
                    <i class="fas fa-newspaper"></i>
                    News
                </a>
                <a href="?tab=banners" class="tab-link <?php echo $tab === 'banners' ? 'active' : ''; ?>">
                    <i class="fas fa-images"></i>
                    Banners
                </a>
            </nav>
        </section>
        
        <!-- Pages Tab -->
        <?php if ($tab === 'pages'): ?>
        <section class="content-section">
            <div class="section-header">
                <h3>Website Pages</h3>
                <button class="btn btn-primary" onclick="openPageEditor()">
                    <i class="fas fa-plus"></i>
                    Add New Page
                </button>
            </div>
            
            <div class="pages-grid">
                <?php foreach ($predefinedPages as $pageId => $pageName): ?>
                <?php 
                $existingPage = null;
                foreach ($cmsPages as $page) {
                    if ($page['page_id'] === $pageId) {
                        $existingPage = $page;
                        break;
                    }
                }
                ?>
                <div class="page-card">
                    <div class="page-header">
                        <div class="page-info">
                            <h4><?php echo $pageName; ?></h4>
                            <p class="page-id"><?php echo $pageId; ?></p>
                        </div>
                        <div class="page-status">
                            <?php if ($existingPage): ?>
                            <span class="status-badge status-<?php echo $existingPage['status']; ?>">
                                <?php echo ucfirst($existingPage['status']); ?>
                            </span>
                            <?php else: ?>
                            <span class="status-badge status-empty">Not Created</span>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div class="page-meta">
                        <?php if ($existingPage): ?>
                        <div class="meta-item">
                            <i class="fas fa-user"></i>
                            <?php echo htmlspecialchars($existingPage['updated_by_username'] ?? 'Unknown'); ?>
                        </div>
                        <div class="meta-item">
                            <i class="fas fa-clock"></i>
                            <?php echo date('M j, Y', strtotime($existingPage['updated_at'])); ?>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="page-actions">
                        <button class="btn btn-primary btn-sm" onclick="editPage('<?php echo $pageId; ?>')">
                            <i class="fas fa-edit"></i>
                            <?php echo $existingPage ? 'Edit' : 'Create'; ?>
                        </button>
                        <?php if ($existingPage): ?>
                        <button class="btn btn-outline btn-sm" onclick="previewPage('<?php echo $pageId; ?>')">
                            <i class="fas fa-eye"></i>
                            Preview
                        </button>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </section>
        <?php endif; ?>
        
        <!-- Promotions Tab -->
        <?php if ($tab === 'promotions'): ?>
        <section class="content-section">
            <div class="section-header">
                <h3>Promotions & Bonuses</h3>
                <button class="btn btn-primary" onclick="openPromotionModal()">
                    <i class="fas fa-plus"></i>
                    Add Promotion
                </button>
            </div>
            
            <div class="table-container">
                <table class="admin-table">
                    <thead>
                        <tr>
                            <th>Title</th>
                            <th>Type</th>
                            <th>Value</th>
                            <th>Code</th>
                            <th>Period</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($promotions as $promo): ?>
                        <tr>
                            <td>
                                <div class="promo-info">
                                    <div class="promo-title"><?php echo htmlspecialchars($promo['title']); ?></div>
                                    <div class="promo-description"><?php echo htmlspecialchars(substr($promo['description'], 0, 100)); ?>...</div>
                                </div>
                            </td>
                            <td>
                                <span class="type-badge type-<?php echo $promo['type']; ?>">
                                    <?php echo ucfirst($promo['type']); ?>
                                </span>
                            </td>
                            <td>
                                <div class="promo-value">
                                    <?php if ($promo['type'] === 'percentage'): ?>
                                    <?php echo $promo['value']; ?>%
                                    <?php else: ?>
                                    $<?php echo number_format($promo['value'], 2); ?>
                                    <?php endif; ?>
                                </div>
                            </td>
                            <td>
                                <?php if ($promo['code']): ?>
                                <code class="promo-code"><?php echo htmlspecialchars($promo['code']); ?></code>
                                <?php else: ?>
                                <span class="text-muted">No code</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <div class="date-range">
                                    <?php if ($promo['start_date']): ?>
                                    <div class="start-date"><?php echo date('M j', strtotime($promo['start_date'])); ?></div>
                                    <?php endif; ?>
                                    <?php if ($promo['end_date']): ?>
                                    <div class="end-date">to <?php echo date('M j', strtotime($promo['end_date'])); ?></div>
                                    <?php endif; ?>
                                </div>
                            </td>
                            <td>
                                <span class="status-badge status-<?php echo $promo['status']; ?>">
                                    <?php echo ucfirst($promo['status']); ?>
                                </span>
                            </td>
                            <td>
                                <div class="action-buttons">
                                    <button class="btn btn-outline btn-xs" onclick="editPromotion(<?php echo $promo['id']; ?>)">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <button class="btn btn-danger btn-xs" onclick="deletePromotion(<?php echo $promo['id']; ?>)">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </section>
        <?php endif; ?>
        
        <!-- News Tab -->
        <?php if ($tab === 'news'): ?>
        <section class="content-section">
            <div class="section-header">
                <h3>News & Announcements</h3>
                <button class="btn btn-primary" onclick="openNewsModal()">
                    <i class="fas fa-plus"></i>
                    Add News
                </button>
            </div>
            
            <div class="news-grid">
                <?php foreach ($news as $article): ?>
                <div class="news-card">
                    <div class="news-header">
                        <h4><?php echo htmlspecialchars($article['title']); ?></h4>
                        <span class="status-badge status-<?php echo $article['status']; ?>">
                            <?php echo ucfirst($article['status']); ?>
                        </span>
                    </div>
                    
                    <div class="news-content">
                        <?php echo htmlspecialchars(substr($article['content'], 0, 200)); ?>...
                    </div>
                    
                    <div class="news-meta">
                        <div class="meta-item">
                            <i class="fas fa-user"></i>
                            <?php echo htmlspecialchars($article['created_by_username']); ?>
                        </div>
                        <div class="meta-item">
                            <i class="fas fa-clock"></i>
                            <?php echo date('M j, Y', strtotime($article['created_at'])); ?>
                        </div>
                    </div>
                    
                    <div class="news-actions">
                        <button class="btn btn-primary btn-sm" onclick="editNews(<?php echo $article['id']; ?>)">
                            <i class="fas fa-edit"></i>
                            Edit
                        </button>
                        <button class="btn btn-outline btn-sm" onclick="previewNews(<?php echo $article['id']; ?>)">
                            <i class="fas fa-eye"></i>
                            Preview
                        </button>
                        <button class="btn btn-danger btn-sm" onclick="deleteNews(<?php echo $article['id']; ?>)">
                            <i class="fas fa-trash"></i>
                            Delete
                        </button>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </section>
        <?php endif; ?>
        
        <!-- Banners Tab -->
        <?php if ($tab === 'banners'): ?>
        <section class="content-section">
            <div class="section-header">
                <h3>Website Banners</h3>
                <button class="btn btn-primary" onclick="openBannerModal()">
                    <i class="fas fa-plus"></i>
                    Add Banner
                </button>
            </div>
            
            <div class="banners-container">
                <div class="banner-placeholder">
                    <i class="fas fa-image"></i>
                    <p>No banners configured yet</p>
                    <button class="btn btn-primary" onclick="openBannerModal()">Add First Banner</button>
                </div>
            </div>
        </section>
        <?php endif; ?>
    </main>
</div>

<!-- Page Editor Modal -->
<div class="modal-backdrop" id="pageEditorModal">
    <div class="modal page-editor-modal large">
        <div class="modal-header">
            <h3 class="modal-title">Page Editor</h3>
            <button class="modal-close" onclick="closeModal('pageEditorModal')">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="modal-body">
            <form id="pageEditorForm" method="POST">
                <input type="hidden" name="action" value="update_page">
                <input type="hidden" name="page_id" id="pageId">
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="pageTitle">Page Title</label>
                        <input type="text" name="title" id="pageTitle" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="pageStatus">Status</label>
                        <select name="status" id="pageStatus">
                            <option value="draft">Draft</option>
                            <option value="published">Published</option>
                        </select>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="metaDescription">Meta Description</label>
                    <textarea name="meta_description" id="metaDescription" rows="2" placeholder="SEO meta description..."></textarea>
                </div>
                
                <div class="form-group">
                    <label for="pageContent">Content</label>
                    <textarea name="content" id="pageContent" rows="15" required placeholder="Enter page content..."></textarea>
                </div>
                
                <div class="modal-actions">
                    <button type="button" class="btn btn-outline" onclick="closeModal('pageEditorModal')">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Page</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Promotion Modal -->
<div class="modal-backdrop" id="promotionModal">
    <div class="modal promotion-modal">
        <div class="modal-header">
            <h3 class="modal-title">Add Promotion</h3>
            <button class="modal-close" onclick="closeModal('promotionModal')">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="modal-body">
            <form id="promotionForm" method="POST">
                <input type="hidden" name="action" value="add_promotion">
                
                <div class="form-group">
                    <label for="promoTitle">Title</label>
                    <input type="text" name="promo_title" id="promoTitle" required>
                </div>
                
                <div class="form-group">
                    <label for="promoDescription">Description</label>
                    <textarea name="promo_description" id="promoDescription" rows="3" required></textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="promoType">Type</label>
                        <select name="promo_type" id="promoType" required>
                            <option value="percentage">Percentage</option>
                            <option value="fixed">Fixed Amount</option>
                            <option value="free_spins">Free Spins</option>
                            <option value="cashback">Cashback</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="promoValue">Value</label>
                        <input type="number" name="promo_value" id="promoValue" step="0.01" required>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="promoCode">Promo Code (Optional)</label>
                    <input type="text" name="promo_code" id="promoCode" placeholder="e.g., WELCOME100">
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="startDate">Start Date</label>
                        <input type="date" name="start_date" id="startDate">
                    </div>
                    
                    <div class="form-group">
                        <label for="endDate">End Date</label>
                        <input type="date" name="end_date" id="endDate">
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="promoStatus">Status</label>
                    <select name="promo_status" id="promoStatus">
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                    </select>
                </div>
                
                <div class="modal-actions">
                    <button type="button" class="btn btn-outline" onclick="closeModal('promotionModal')">Cancel</button>
                    <button type="submit" class="btn btn-primary">Add Promotion</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- News Modal -->
<div class="modal-backdrop" id="newsModal">
    <div class="modal news-modal">
        <div class="modal-header">
            <h3 class="modal-title">Add News Article</h3>
            <button class="modal-close" onclick="closeModal('newsModal')">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="modal-body">
            <form id="newsForm" method="POST">
                <input type="hidden" name="action" value="add_news">
                
                <div class="form-group">
                    <label for="newsTitle">Title</label>
                    <input type="text" name="news_title" id="newsTitle" required>
                </div>
                
                <div class="form-group">
                    <label for="newsContent">Content</label>
                    <textarea name="news_content" id="newsContent" rows="10" required></textarea>
                </div>
                
                <div class="form-group">
                    <label for="newsStatus">Status</label>
                    <select name="news_status" id="newsStatus">
                        <option value="draft">Draft</option>
                        <option value="published">Published</option>
                    </select>
                </div>
                
                <div class="modal-actions">
                    <button type="button" class="btn btn-outline" onclick="closeModal('newsModal')">Cancel</button>
                    <button type="submit" class="btn btn-primary">Add News</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// Content management functions
function editPage(pageId) {
    // Load existing page data if available
    fetch(`/api/admin/pages/${pageId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success && data.page) {
                document.getElementById('pageId').value = pageId;
                document.getElementById('pageTitle').value = data.page.title || '';
                document.getElementById('pageContent').value = data.page.content || '';
                document.getElementById('metaDescription').value = data.page.meta_description || '';
                document.getElementById('pageStatus').value = data.page.status || 'draft';
            } else {
                // New page
                document.getElementById('pageId').value = pageId;
                document.getElementById('pageTitle').value = '';
                document.getElementById('pageContent').value = '';
                document.getElementById('metaDescription').value = '';
                document.getElementById('pageStatus').value = 'draft';
            }
            openModal('pageEditorModal');
        })
        .catch(error => {
            console.error('Error:', error);
            // Open modal anyway for new page
            document.getElementById('pageId').value = pageId;
            openModal('pageEditorModal');
        });
}

function openPageEditor() {
    const pageId = prompt('Enter page ID (e.g., custom-page):');
    if (pageId) {
        editPage(pageId);
    }
}

function previewPage(pageId) {
    window.open(`/pages/${pageId}`, '_blank');
}

function openPromotionModal() {
    document.getElementById('promotionForm').reset();
    openModal('promotionModal');
}

function editPromotion(promoId) {
    // Load promotion data and open modal
    console.log('Edit promotion:', promoId);
}

function deletePromotion(promoId) {
    if (confirm('Are you sure you want to delete this promotion?')) {
        // Delete promotion
        console.log('Delete promotion:', promoId);
    }
}

function openNewsModal() {
    document.getElementById('newsForm').reset();
    openModal('newsModal');
}

function editNews(newsId) {
    // Load news data and open modal
    console.log('Edit news:', newsId);
}

function previewNews(newsId) {
    window.open(`/news/${newsId}`, '_blank');
}

function deleteNews(newsId) {
    if (confirm('Are you sure you want to delete this news article?')) {
        // Delete news
        console.log('Delete news:', newsId);
    }
}

function openBannerModal() {
    // Open banner management modal
    console.log('Open banner modal');
}

function previewSite() {
    window.open('/', '_blank');
}
</script>

<?php include '../includes/footer.php'; ?>
