
<?php
// admin/users.php - User Management
// مدیریت کاربران

require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

// Require admin authentication
requireAuth();
requireAdmin();

// Page settings
$pageTitle = 'User Management - Admin';
$bodyClass = 'admin-users';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $userId = intval($_POST['user_id'] ?? 0);
    
    switch ($action) {
        case 'update_status':
            $status = $_POST['status'] ?? '';
            if (in_array($status, ['active', 'suspended', 'banned'])) {
                execute("UPDATE users SET status = ? WHERE id = ?", [$status, $userId]);
                $message = "User status updated successfully";
            }
            break;
            
        case 'update_balance':
            $amount = floatval($_POST['amount'] ?? 0);
            $type = $_POST['type'] ?? 'add'; // add or subtract
            
            if ($amount > 0) {
                if ($type === 'add') {
                    execute("UPDATE users SET balance = balance + ? WHERE id = ?", [$amount, $userId]);
                    execute("INSERT INTO transactions (user_id, type, amount, status, description, created_at) VALUES (?, 'admin_credit', ?, 'completed', 'Admin balance adjustment', NOW())", [$userId, $amount]);
                } else {
                    execute("UPDATE users SET balance = GREATEST(0, balance - ?) WHERE id = ?", [$amount, $userId]);
                    execute("INSERT INTO transactions (user_id, type, amount, status, description, created_at) VALUES (?, 'admin_debit', ?, 'completed', 'Admin balance adjustment', NOW())", [$userId, $amount]);
                }
                $message = "User balance updated successfully";
            }
            break;
            
        case 'send_message':
            $messageText = $_POST['message'] ?? '';
            if ($messageText) {
                execute("INSERT INTO admin_messages (user_id, admin_id, message, created_at) VALUES (?, ?, ?, NOW())", [$userId, $currentUser['id'], $messageText]);
                $message = "Message sent successfully";
            }
            break;
    }
}

// Get filter parameters
$search = $_GET['search'] ?? '';
$status = $_GET['status'] ?? 'all';
$sortBy = $_GET['sort'] ?? 'created_at';
$sortOrder = $_GET['order'] ?? 'DESC';
$page = max(1, intval($_GET['page'] ?? 1));
$limit = 25;
$offset = ($page - 1) * $limit;

// Build WHERE clause
$whereConditions = [];
$params = [];

if ($search) {
    $whereConditions[] = "(username LIKE ? OR email LIKE ? OR first_name LIKE ? OR last_name LIKE ?)";
    $searchTerm = "%$search%";
    $params = array_merge($params, [$searchTerm, $searchTerm, $searchTerm, $searchTerm]);
}

if ($status !== 'all') {
    $whereConditions[] = "status = ?";
    $params[] = $status;
}

$whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';

// Validate sort parameters
$allowedSorts = ['id', 'username', 'email', 'balance', 'created_at', 'last_activity'];
$sortBy = in_array($sortBy, $allowedSorts) ? $sortBy : 'created_at';
$sortOrder = strtoupper($sortOrder) === 'ASC' ? 'ASC' : 'DESC';

// Get total count
$totalUsers = fetchValue("SELECT COUNT(*) FROM users $whereClause", $params);

// Get users
$users = fetchAll(
    "SELECT u.*, 
        (SELECT COUNT(*) FROM user_bets WHERE user_id = u.id) as total_bets,
        (SELECT SUM(bet_amount) FROM user_bets WHERE user_id = u.id) as total_wagered,
        (SELECT COUNT(*) FROM transactions WHERE user_id = u.id AND type = 'deposit') as total_deposits
     FROM users u 
     $whereClause 
     ORDER BY $sortBy $sortOrder 
     LIMIT $limit OFFSET $offset",
    $params
);

// Calculate pagination
$totalPages = ceil($totalUsers / $limit);

// Get user statistics
$userStats = fetchRow(
    "SELECT 
        COUNT(*) as total_users,
        COUNT(CASE WHEN status = 'active' THEN 1 END) as active_users,
        COUNT(CASE WHEN status = 'suspended' THEN 1 END) as suspended_users,
        COUNT(CASE WHEN status = 'banned' THEN 1 END) as banned_users,
        COUNT(CASE WHEN DATE(created_at) = CURDATE() THEN 1 END) as new_today,
        COUNT(CASE WHEN last_activity > DATE_SUB(NOW(), INTERVAL 15 MINUTE) THEN 1 END) as online_now
     FROM users"
);

include '../includes/header.php';
?>

<!-- Admin Users Management -->
<div class="admin-container">
    <?php include 'sidebar.php'; ?>
    
    <main class="admin-main">
        <!-- Header -->
        <header class="admin-header">
            <div class="header-left">
                <h1 class="page-title">User Management</h1>
                <p class="page-subtitle">Manage user accounts, balances, and permissions</p>
            </div>
            
            <div class="header-right">
                <button class="btn btn-outline" onclick="exportUsers()">
                    <i class="fas fa-download"></i>
                    Export CSV
                </button>
                <button class="btn btn-primary" onclick="openAddUserModal()">
                    <i class="fas fa-plus"></i>
                    Add User
                </button>
            </div>
        </header>
        
        <!-- User Statistics -->
        <section class="stats-section">
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format($userStats['total_users']); ?></div>
                        <div class="stat-label">Total Users</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon active">
                        <i class="fas fa-user-check"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format($userStats['active_users']); ?></div>
                        <div class="stat-label">Active Users</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon warning">
                        <i class="fas fa-user-clock"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format($userStats['suspended_users']); ?></div>
                        <div class="stat-label">Suspended</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon danger">
                        <i class="fas fa-user-slash"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format($userStats['banned_users']); ?></div>
                        <div class="stat-label">Banned</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon success">
                        <i class="fas fa-user-plus"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format($userStats['new_today']); ?></div>
                        <div class="stat-label">New Today</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon online">
                        <i class="fas fa-circle"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format($userStats['online_now']); ?></div>
                        <div class="stat-label">Online Now</div>
                    </div>
                </div>
            </div>
        </section>
        
        <!-- Filters and Search -->
        <section class="filters-section">
            <form class="filters-form" method="GET">
                <div class="filter-group">
                    <div class="search-box">
                        <i class="fas fa-search"></i>
                        <input type="text" 
                               name="search" 
                               value="<?php echo htmlspecialchars($search); ?>" 
                               placeholder="Search users..."
                               class="search-input">
                    </div>
                </div>
                
                <div class="filter-group">
                    <select name="status" onchange="this.form.submit()">
                        <option value="all" <?php echo $status === 'all' ? 'selected' : ''; ?>>All Status</option>
                        <option value="active" <?php echo $status === 'active' ? 'selected' : ''; ?>>Active</option>
                        <option value="suspended" <?php echo $status === 'suspended' ? 'selected' : ''; ?>>Suspended</option>
                        <option value="banned" <?php echo $status === 'banned' ? 'selected' : ''; ?>>Banned</option>
                    </select>
                </div>
                
                <div class="filter-group">
                    <select name="sort" onchange="this.form.submit()">
                        <option value="created_at" <?php echo $sortBy === 'created_at' ? 'selected' : ''; ?>>Registration Date</option>
                        <option value="username" <?php echo $sortBy === 'username' ? 'selected' : ''; ?>>Username</option>
                        <option value="balance" <?php echo $sortBy === 'balance' ? 'selected' : ''; ?>>Balance</option>
                        <option value="last_activity" <?php echo $sortBy === 'last_activity' ? 'selected' : ''; ?>>Last Activity</option>
                    </select>
                </div>
                
                <div class="filter-group">
                    <select name="order" onchange="this.form.submit()">
                        <option value="DESC" <?php echo $sortOrder === 'DESC' ? 'selected' : ''; ?>>Descending</option>
                        <option value="ASC" <?php echo $sortOrder === 'ASC' ? 'selected' : ''; ?>>Ascending</option>
                    </select>
                </div>
                
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-filter"></i>
                    Filter
                </button>
                
                <a href="/admin/users.php" class="btn btn-outline">
                    <i class="fas fa-times"></i>
                    Clear
                </a>
            </form>
            
            <div class="results-info">
                Showing <?php echo count($users); ?> of <?php echo number_format($totalUsers); ?> users
            </div>
        </section>
        
        <!-- Users Table -->
        <section class="table-section">
            <div class="table-container">
                <table class="admin-table">
                    <thead>
                        <tr>
                            <th>
                                <input type="checkbox" id="selectAll" onchange="toggleSelectAll()">
                            </th>
                            <th>User</th>
                            <th>Contact</th>
                            <th>Balance</th>
                            <th>Activity</th>
                            <th>Registration</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users as $user): ?>
                        <tr class="user-row" data-user-id="<?php echo $user['id']; ?>">
                            <td>
                                <input type="checkbox" class="user-checkbox" value="<?php echo $user['id']; ?>">
                            </td>
                            
                            <td>
                                <div class="user-info">
                                    <div class="user-avatar">
                                        <?php if ($user['avatar']): ?>
                                        <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="Avatar">
                                        <?php else: ?>
                                        <i class="fas fa-user"></i>
                                        <?php endif; ?>
                                        <?php if (strtotime($user['last_activity']) > strtotime('-15 minutes')): ?>
                                        <span class="online-indicator"></span>
                                        <?php endif; ?>
                                    </div>
                                    <div class="user-details">
                                        <div class="username"><?php echo htmlspecialchars($user['username']); ?></div>
                                        <div class="user-id">ID: <?php echo $user['id']; ?></div>
                                        <?php if ($user['first_name'] || $user['last_name']): ?>
                                        <div class="full-name"><?php echo htmlspecialchars(trim($user['first_name'] . ' ' . $user['last_name'])); ?></div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </td>
                            
                            <td>
                                <div class="contact-info">
                                    <div class="email">
                                        <i class="fas fa-envelope"></i>
                                        <?php echo htmlspecialchars($user['email']); ?>
                                    </div>
                                    <?php if ($user['phone']): ?>
                                    <div class="phone">
                                        <i class="fas fa-phone"></i>
                                        <?php echo htmlspecialchars($user['phone']); ?>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            </td>
                            
                            <td>
                                <div class="balance-info">
                                    <div class="balance-amount">$<?php echo number_format($user['balance'], 2); ?></div>
                                    <div class="balance-stats">
                                        <small>Bets: <?php echo number_format($user['total_bets'] ?: 0); ?></small>
                                        <small>Wagered: $<?php echo number_format($user['total_wagered'] ?: 0, 2); ?></small>
                                    </div>
                                </div>
                            </td>
                            
                            <td>
                                <div class="activity-info">
                                    <div class="last-activity">
                                        <?php if ($user['last_activity']): ?>
                                        <?php echo timeAgo($user['last_activity']); ?>
                                        <?php else: ?>
                                        Never
                                        <?php endif; ?>
                                    </div>
                                    <div class="activity-stats">
                                        <small>Deposits: <?php echo number_format($user['total_deposits'] ?: 0); ?></small>
                                    </div>
                                </div>
                            </td>
                            
                            <td>
                                <div class="registration-info">
                                    <div class="reg-date"><?php echo date('M j, Y', strtotime($user['created_at'])); ?></div>
                                    <div class="reg-time"><?php echo date('H:i', strtotime($user['created_at'])); ?></div>
                                </div>
                            </td>
                            
                            <td>
                                <span class="status-badge status-<?php echo $user['status']; ?>">
                                    <?php echo ucfirst($user['status']); ?>
                                </span>
                            </td>
                            
                            <td>
                                <div class="action-buttons">
                                    <button class="btn btn-outline btn-xs" onclick="viewUser(<?php echo $user['id']; ?>)">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                    <button class="btn btn-primary btn-xs" onclick="editUser(<?php echo $user['id']; ?>)">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <div class="dropdown">
                                        <button class="btn btn-outline btn-xs dropdown-toggle">
                                            <i class="fas fa-ellipsis-v"></i>
                                        </button>
                                        <div class="dropdown-menu">
                                            <a href="#" onclick="adjustBalance(<?php echo $user['id']; ?>)">Adjust Balance</a>
                                            <a href="#" onclick="sendMessage(<?php echo $user['id']; ?>)">Send Message</a>
                                            <a href="#" onclick="viewTransactions(<?php echo $user['id']; ?>)">View Transactions</a>
                                            <a href="#" onclick="loginAsUser(<?php echo $user['id']; ?>)">Login as User</a>
                                            <div class="dropdown-divider"></div>
                                            <?php if ($user['status'] !== 'suspended'): ?>
                                            <a href="#" onclick="suspendUser(<?php echo $user['id']; ?>)" class="text-warning">Suspend</a>
                                            <?php endif; ?>
                                            <?php if ($user['status'] !== 'banned'): ?>
                                            <a href="#" onclick="banUser(<?php echo $user['id']; ?>)" class="text-danger">Ban</a>
                                            <?php endif; ?>
                                            <?php if ($user['status'] !== 'active'): ?>
                                            <a href="#" onclick="activateUser(<?php echo $user['id']; ?>)" class="text-success">Activate</a>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            <?php if ($totalPages > 1): ?>
            <div class="pagination-container">
                <nav class="pagination">
                    <?php if ($page > 1): ?>
                    <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>" class="page-link prev">
                        <i class="fas fa-chevron-left"></i>
                        Previous
                    </a>
                    <?php endif; ?>
                    
                    <?php
                    $startPage = max(1, $page - 2);
                    $endPage = min($totalPages, $page + 2);
                    
                    for ($i = $startPage; $i <= $endPage; $i++): ?>
                    <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>" 
                       class="page-link <?php echo $i === $page ? 'active' : ''; ?>">
                        <?php echo $i; ?>
                    </a>
                    <?php endfor; ?>
                    
                    <?php if ($page < $totalPages): ?>
                    <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>" class="page-link next">
                        Next
                        <i class="fas fa-chevron-right"></i>
                    </a>
                    <?php endif; ?>
                </nav>
            </div>
            <?php endif; ?>
        </section>
        
        <!-- Bulk Actions -->
        <section class="bulk-actions" id="bulkActions" style="display: none;">
            <div class="bulk-actions-bar">
                <div class="selected-count">
                    <span id="selectedCount">0</span> users selected
                </div>
                <div class="bulk-buttons">
                    <button class="btn btn-outline" onclick="bulkAction('activate')">
                        <i class="fas fa-check"></i>
                        Activate
                    </button>
                    <button class="btn btn-warning" onclick="bulkAction('suspend')">
                        <i class="fas fa-pause"></i>
                        Suspend
                    </button>
                    <button class="btn btn-danger" onclick="bulkAction('ban')">
                        <i class="fas fa-ban"></i>
                        Ban
                    </button>
                    <button class="btn btn-outline" onclick="bulkExport()">
                        <i class="fas fa-download"></i>
                        Export Selected
                    </button>
                </div>
            </div>
        </section>
    </main>
</div>

<!-- User Details Modal -->
<div class="modal-backdrop" id="userModal">
    <div class="modal user-modal">
        <div class="modal-header">
            <h3 class="modal-title">User Details</h3>
            <button class="modal-close" onclick="closeModal('userModal')">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="modal-body" id="userModalContent">
            <!-- User details will be loaded here -->
        </div>
    </div>
</div>

<!-- Balance Adjustment Modal -->
<div class="modal-backdrop" id="balanceModal">
    <div class="modal balance-modal">
        <div class="modal-header">
            <h3 class="modal-title">Adjust Balance</h3>
            <button class="modal-close" onclick="closeModal('balanceModal')">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="modal-body">
            <form id="balanceForm" method="POST">
                <input type="hidden" name="action" value="update_balance">
                <input type="hidden" name="user_id" id="balanceUserId">
                
                <div class="form-group">
                    <label>Current Balance</label>
                    <div class="current-balance" id="currentBalance">$0.00</div>
                </div>
                
                <div class="form-group">
                    <label for="balanceType">Action</label>
                    <select name="type" id="balanceType" required>
                        <option value="add">Add to Balance</option>
                        <option value="subtract">Subtract from Balance</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="balanceAmount">Amount</label>
                    <input type="number" name="amount" id="balanceAmount" step="0.01" min="0.01" required>
                </div>
                
                <div class="form-group">
                    <label for="balanceReason">Reason</label>
                    <textarea name="reason" id="balanceReason" rows="3" placeholder="Reason for balance adjustment..."></textarea>
                </div>
                
                <div class="modal-actions">
                    <button type="button" class="btn btn-outline" onclick="closeModal('balanceModal')">Cancel</button>
                    <button type="submit" class="btn btn-primary">Update Balance</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// User management functions
function viewUser(userId) {
    fetch(`/api/admin/users/${userId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showUserDetails(data.user);
            }
        })
        .catch(error => {
            console.error('Error:', error);
        });
}

function editUser(userId) {
    window.location.href = `/admin/user-edit.php?id=${userId}`;
}

function adjustBalance(userId) {
    fetch(`/api/admin/users/${userId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                document.getElementById('balanceUserId').value = userId;
                document.getElementById('currentBalance').textContent = `$${parseFloat(data.user.balance).toFixed(2)}`;
                openModal('balanceModal');
            }
        });
}

function suspendUser(userId) {
    if (confirm('Are you sure you want to suspend this user?')) {
        updateUserStatus(userId, 'suspended');
    }
}

function banUser(userId) {
    if (confirm('Are you sure you want to ban this user?')) {
        updateUserStatus(userId, 'banned');
    }
}

function activateUser(userId) {
    updateUserStatus(userId, 'active');
}

function updateUserStatus(userId, status) {
    const formData = new FormData();
    formData.append('action', 'update_status');
    formData.append('user_id', userId);
    formData.append('status', status);
    
    fetch('/admin/users.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.text())
    .then(() => {
        window.location.reload();
    })
    .catch(error => {
        console.error('Error:', error);
    });
}

// Bulk actions
function toggleSelectAll() {
    const selectAll = document.getElementById('selectAll');
    const checkboxes = document.querySelectorAll('.user-checkbox');
    
    checkboxes.forEach(checkbox => {
        checkbox.checked = selectAll.checked;
    });
    
    updateBulkActions();
}

function updateBulkActions() {
    const checkboxes = document.querySelectorAll('.user-checkbox:checked');
    const bulkActions = document.getElementById('bulkActions');
    const selectedCount = document.getElementById('selectedCount');
    
    if (checkboxes.length > 0) {
        bulkActions.style.display = 'block';
        selectedCount.textContent = checkboxes.length;
    } else {
        bulkActions.style.display = 'none';
    }
}

// Add event listeners to checkboxes
document.addEventListener('DOMContentLoaded', function() {
    const checkboxes = document.querySelectorAll('.user-checkbox');
    checkboxes.forEach(checkbox => {
        checkbox.addEventListener('change', updateBulkActions);
    });
});

function bulkAction(action) {
    const selectedUsers = Array.from(document.querySelectorAll('.user-checkbox:checked'))
        .map(cb => cb.value);
    
    if (selectedUsers.length === 0) return;
    
    if (confirm(`Are you sure you want to ${action} ${selectedUsers.length} users?`)) {
        // Implement bulk action
        console.log(`Bulk ${action} for users:`, selectedUsers);
    }
}

function exportUsers() {
    const params = new URLSearchParams(window.location.search);
    params.set('export', 'csv');
    window.open(`/api/admin/users/export?${params.toString()}`, '_blank');
}
</script>

<?php include '../includes/footer.php'; ?>
