<?php
// pages/crash-game.php - Main Crash Game Page
// صفحه اصلی بازی کرش

require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

// Require authentication
requireAuth();

// Page settings
$pageTitle = 'Crash Game - Crashify';
$bodyClass = 'crash-game-page';
$breadcrumbs = [
    ['title' => 'Games', 'url' => '/pages/casino.php'],
    ['title' => 'Crash Game']
];

// Get user's crash game stats
$userStats = fetchRow(
    "SELECT 
        COUNT(*) as total_games,
        SUM(CASE WHEN profit > 0 THEN 1 ELSE 0 END) as wins,
        SUM(bet_amount) as total_wagered,
        SUM(profit) as total_profit,
        MAX(multiplier) as highest_multiplier,
        AVG(multiplier) as avg_multiplier
     FROM crash_bets 
     WHERE user_id = ? AND status = 'completed'",
    [$currentUser['id']]
) ?: [
    'total_games' => 0,
    'wins' => 0,
    'total_wagered' => 0,
    'total_profit' => 0,
    'highest_multiplier' => 0,
    'avg_multiplier' => 0
];

// Get recent crash results
$recentResults = fetchAll(
    "SELECT multiplier, created_at 
     FROM crash_rounds 
     WHERE status = 'completed' 
     ORDER BY created_at DESC 
     LIMIT 20"
);

// Get current game state
$currentRound = fetchRow(
    "SELECT * FROM crash_rounds 
     WHERE status IN ('waiting', 'running') 
     ORDER BY created_at DESC 
     LIMIT 1"
);

// Get user's active bet for current round
$activeBet = null;
if ($currentRound) {
    $activeBet = fetchRow(
        "SELECT * FROM crash_bets 
         WHERE user_id = ? AND round_id = ? AND status = 'active'",
        [$currentUser['id'], $currentRound['id']]
    );
}

// Get top winners today
$topWinners = fetchAll(
    "SELECT u.username, cb.bet_amount, cb.multiplier, cb.profit
     FROM crash_bets cb
     JOIN users u ON cb.user_id = u.id
     WHERE DATE(cb.created_at) = CURDATE() 
     AND cb.profit > 0
     ORDER BY cb.profit DESC
     LIMIT 10"
);

$additionalScripts = [
    '/assets/js/crash-game.js',
    '/assets/js/chart.js'
];

include '../includes/header.php';
?>

<!-- Crash Game Container -->
<div class="crash-game-container">
    <!-- Game Header -->
    <section class="game-header">
        <div class="container">
            <div class="header-content">
                <div class="game-info">
                    <h1 class="game-title">
                        <i class="fas fa-rocket"></i>
                        Crash Game
                    </h1>
                    <p class="game-description">
                        Watch the multiplier rise and cash out before it crashes!
                    </p>
                </div>
                
                <div class="game-stats">
                    <div class="stat-item">
                        <div class="stat-value"><?php echo number_format($userStats['total_games']); ?></div>
                        <div class="stat-label">Games Played</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-value"><?php echo $userStats['total_games'] > 0 ? round(($userStats['wins'] / $userStats['total_games']) * 100, 1) : 0; ?>%</div>
                        <div class="stat-label">Win Rate</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-value"><?php echo $userStats['highest_multiplier'] > 0 ? number_format($userStats['highest_multiplier'], 2) : '0.00'; ?>x</div>
                        <div class="stat-label">Best Multiplier</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-value">$<?php echo number_format($userStats['total_profit'], 2); ?></div>
                        <div class="stat-label">Total Profit</div>
                    </div>
                </div>
            </div>
        </div>
    </section>
    
    <!-- Main Game Area -->
    <section class="game-area">
        <div class="container">
            <div class="game-layout">
                <!-- Game Chart -->
                <div class="game-chart-container">
                    <div class="game-status" id="gameStatus">
                        <div class="status-waiting" id="statusWaiting" style="display: none;">
                            <div class="waiting-content">
                                <div class="waiting-icon">
                                    <i class="fas fa-clock"></i>
                                </div>
                                <h3>Next Round Starting...</h3>
                                <div class="countdown" id="countdown">5</div>
                            </div>
                        </div>
                        
                        <div class="status-running" id="statusRunning" style="display: none;">
                            <div class="multiplier-display">
                                <div class="current-multiplier" id="currentMultiplier">1.00x</div>
                                <div class="multiplier-trend" id="multiplierTrend">
                                    <i class="fas fa-arrow-up"></i>
                                </div>
                            </div>
                        </div>
                        
                        <div class="status-crashed" id="statusCrashed" style="display: none;">
                            <div class="crashed-content">
                                <div class="crashed-icon">
                                    <i class="fas fa-bomb"></i>
                                </div>
                                <h3>CRASHED!</h3>
                                <div class="crash-multiplier" id="crashMultiplier">0.00x</div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Chart Canvas -->
                    <div class="chart-container">
                        <canvas id="crashChart" width="800" height="400"></canvas>
                    </div>
                    
                    <!-- Recent Results -->
                    <div class="recent-results">
                        <h4>Recent Results</h4>
                        <div class="results-list" id="recentResults">
                            <?php foreach ($recentResults as $result): ?>
                            <div class="result-item <?php echo $result['multiplier'] >= 2 ? 'high' : ($result['multiplier'] >= 1.5 ? 'medium' : 'low'); ?>">
                                <?php echo number_format($result['multiplier'], 2); ?>x
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
                
                <!-- Game Controls -->
                <div class="game-controls">
                    <!-- Betting Panel -->
                    <div class="betting-panel">
                        <div class="panel-header">
                            <h3>Place Your Bet</h3>
                            <div class="balance-display">
                                Balance: <span class="balance-amount">$<?php echo number_format($currentUser['balance'], 2); ?></span>
                            </div>
                        </div>
                        
                        <div class="betting-form" id="bettingForm">
                            <div class="bet-amount-section">
                                <label for="betAmount">Bet Amount</label>
                                <div class="amount-input">
                                    <span class="currency">$</span>
                                    <input type="number" 
                                           id="betAmount" 
                                           min="1" 
                                           max="<?php echo $currentUser['balance']; ?>" 
                                           step="0.01" 
                                           value="10.00"
                                           placeholder="0.00">
                                </div>
                                <div class="quick-amounts">
                                    <button class="quick-amount" onclick="setBetAmount(5)">$5</button>
                                    <button class="quick-amount" onclick="setBetAmount(10)">$10</button>
                                    <button class="quick-amount" onclick="setBetAmount(25)">$25</button>
                                    <button class="quick-amount" onclick="setBetAmount(50)">$50</button>
                                    <button class="quick-amount" onclick="setBetAmount(100)">$100</button>
                                    <button class="quick-amount max" onclick="setBetAmount(<?php echo $currentUser['balance']; ?>)">MAX</button>
                                </div>
                            </div>
                            
                            <div class="auto-cashout-section">
                                <label class="checkbox-label">
                                    <input type="checkbox" id="autoCashout">
                                    <span class="checkbox-custom"></span>
                                    <span class="checkbox-text">Auto Cash Out</span>
                                </label>
                                <div class="auto-cashout-input" id="autoCashoutInput" style="display: none;">
                                    <input type="number" 
                                           id="autoCashoutMultiplier" 
                                           min="1.01" 
                                           max="1000" 
                                           step="0.01" 
                                           value="2.00"
                                           placeholder="2.00">
                                    <span class="multiplier-suffix">x</span>
                                </div>
                            </div>
                            
                            <div class="betting-actions">
                                <button class="btn btn-primary btn-lg btn-place-bet" id="placeBetBtn" onclick="placeBet()">
                                    <i class="fas fa-rocket"></i>
                                    <span>Place Bet</span>
                                </button>
                                
                                <button class="btn btn-success btn-lg btn-cash-out" id="cashOutBtn" onclick="cashOut()" style="display: none;">
                                    <i class="fas fa-hand-paper"></i>
                                    <span>Cash Out</span>
                                    <div class="cash-out-amount" id="cashOutAmount">$0.00</div>
                                </button>
                            </div>
                        </div>
                        
                        <!-- Active Bet Display -->
                        <div class="active-bet" id="activeBet" style="display: none;">
                            <div class="bet-info">
                                <div class="bet-amount">
                                    <span class="label">Bet:</span>
                                    <span class="amount" id="activeBetAmount">$0.00</span>
                                </div>
                                <div class="potential-win">
                                    <span class="label">Potential Win:</span>
                                    <span class="amount" id="potentialWin">$0.00</span>
                                </div>
                            </div>
                            <div class="bet-status" id="betStatus">
                                <span class="status-text">Waiting for round...</span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Live Players -->
                    <div class="live-players-panel">
                        <div class="panel-header">
                            <h3>Live Players</h3>
                            <div class="players-count" id="playersCount">0 players</div>
                        </div>
                        
                        <div class="players-list" id="playersList">
                            <!-- Live players will be populated here -->
                        </div>
                    </div>
                    
                    <!-- Chat Panel -->
                    <div class="chat-panel">
                        <div class="panel-header">
                            <h3>Chat</h3>
                            <button class="chat-toggle" onclick="toggleChat()">
                                <i class="fas fa-comments"></i>
                            </button>
                        </div>
                        
                        <div class="chat-messages" id="chatMessages">
                            <!-- Chat messages will be populated here -->
                        </div>
                        
                        <div class="chat-input">
                            <input type="text" 
                                   id="chatMessageInput" 
                                   placeholder="Type a message..." 
                                   maxlength="200"
                                   onkeypress="handleChatKeyPress(event)">
                            <button class="send-btn" onclick="sendChatMessage()">
                                <i class="fas fa-paper-plane"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>
    
    <!-- Game Statistics -->
    <section class="game-statistics">
        <div class="container">
            <div class="stats-grid">
                <!-- Top Winners Today -->
                <div class="stats-widget">
                    <div class="widget-header">
                        <h3>
                            <i class="fas fa-trophy"></i>
                            Top Winners Today
                        </h3>
                    </div>
                    <div class="winners-list">
                        <?php if (empty($topWinners)): ?>
                        <div class="empty-state">
                            <p>No winners yet today</p>
                        </div>
                        <?php else: ?>
                        <?php foreach ($topWinners as $index => $winner): ?>
                        <div class="winner-item">
                            <div class="winner-rank">#<?php echo $index + 1; ?></div>
                            <div class="winner-info">
                                <div class="winner-name"><?php echo htmlspecialchars($winner['username']); ?></div>
                                <div class="winner-details">
                                    $<?php echo number_format($winner['bet_amount'], 2); ?> @ <?php echo number_format($winner['multiplier'], 2); ?>x
                                </div>
                            </div>
                            <div class="winner-profit">
                                +$<?php echo number_format($winner['profit'], 2); ?>
                            </div>
                        </div>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Game Statistics -->
                <div class="stats-widget">
                    <div class="widget-header">
                        <h3>
                            <i class="fas fa-chart-bar"></i>
                            Game Statistics
                        </h3>
                    </div>
                    <div class="stats-list">
                        <div class="stat-row">
                            <span class="stat-label">Average Multiplier:</span>
                            <span class="stat-value"><?php echo number_format(array_sum(array_column($recentResults, 'multiplier')) / max(count($recentResults), 1), 2); ?>x</span>
                        </div>
                        <div class="stat-row">
                            <span class="stat-label">Highest Today:</span>
                            <span class="stat-value"><?php echo !empty($recentResults) ? number_format(max(array_column($recentResults, 'multiplier')), 2) : '0.00'; ?>x</span>
                        </div>
                        <div class="stat-row">
                            <span class="stat-label">Games > 2x:</span>
                            <span class="stat-value"><?php echo count(array_filter($recentResults, fn($r) => $r['multiplier'] >= 2)); ?>/<?php echo count($recentResults); ?></span>
                        </div>
                        <div class="stat-row">
                            <span class="stat-label">Games > 10x:</span>
                            <span class="stat-value"><?php echo count(array_filter($recentResults, fn($r) => $r['multiplier'] >= 10)); ?>/<?php echo count($recentResults); ?></span>
                        </div>
                    </div>
                </div>
                
                <!-- How to Play -->
                <div class="stats-widget">
                    <div class="widget-header">
                        <h3>
                            <i class="fas fa-question-circle"></i>
                            How to Play
                        </h3>
                    </div>
                    <div class="how-to-play">
                        <div class="step">
                            <div class="step-number">1</div>
                            <div class="step-text">Place your bet before the round starts</div>
                        </div>
                        <div class="step">
                            <div class="step-number">2</div>
                            <div class="step-text">Watch the multiplier rise</div>
                        </div>
                        <div class="step">
                            <div class="step-number">3</div>
                            <div class="step-text">Cash out before it crashes!</div>
                        </div>
                        <div class="step">
                            <div class="step-number">4</div>
                            <div class="step-text">Win = Bet × Multiplier</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- Game Sounds -->
<audio id="soundBet" preload="auto">
    <source src="/assets/sounds/bet.mp3" type="audio/mpeg">
</audio>
<audio id="soundCashOut" preload="auto">
    <source src="/assets/sounds/cashout.mp3" type="audio/mpeg">
</audio>
<audio id="soundCrash" preload="auto">
    <source src="/assets/sounds/crash.mp3" type="audio/mpeg">
</audio>

<script>
// Initialize crash game
document.addEventListener('DOMContentLoaded', function() {
    initializeCrashGame();
});

// Game state
let gameState = {
    status: 'waiting', // waiting, running, crashed
    currentMultiplier: 1.00,
    startTime: null,
    activeBet: <?php echo $activeBet ? json_encode($activeBet) : 'null'; ?>,
    autoCashout: false,
    autoCashoutMultiplier: 2.00
};

function initializeCrashGame() {
    // Initialize WebSocket connection for real-time updates
    connectToGameServer();
    
    // Initialize chart
    initializeCrashChart();
    
    // Setup auto cashout toggle
    document.getElementById('autoCashout').addEventListener('change', function() {
        const autoCashoutInput = document.getElementById('autoCashoutInput');
        if (this.checked) {
            autoCashoutInput.style.display = 'block';
            gameState.autoCashout = true;
        } else {
            autoCashoutInput.style.display = 'none';
            gameState.autoCashout = false;
        }
    });
    
    // Setup auto cashout multiplier
    document.getElementById('autoCashoutMultiplier').addEventListener('input', function() {
        gameState.autoCashoutMultiplier = parseFloat(this.value) || 2.00;
    });
    
    // Load current game state
    loadGameState();
}

function connectToGameServer() {
    // This would connect to WebSocket server for real-time game updates
    // For demo purposes, we'll simulate with intervals
    
    // Simulate game rounds
    setInterval(() => {
        if (gameState.status === 'waiting') {
            startNewRound();
        }
    }, 10000); // New round every 10 seconds for demo
}

function loadGameState() {
    // Load current game state from server
    Crashify.api.request('/api/crash/state')
        .then(response => {
            if (response.success) {
                updateGameState(response.gameState);
            }
        })
        .catch(error => {
            console.error('Failed to load game state:', error);
        });
}

function updateGameState(newState) {
    gameState = { ...gameState, ...newState };
    
    // Update UI based on game state
    switch (gameState.status) {
        case 'waiting':
            showWaitingState();
            break;
        case 'running':
            showRunningState();
            break;
        case 'crashed':
            showCrashedState();
            break;
    }
}

function showWaitingState() {
    document.getElementById('statusWaiting').style.display = 'block';
    document.getElementById('statusRunning').style.display = 'none';
    document.getElementById('statusCrashed').style.display = 'none';
    
    // Enable betting
    document.getElementById('placeBetBtn').disabled = false;
    document.getElementById('cashOutBtn').style.display = 'none';
    
    // Start countdown
    startCountdown();
}

function showRunningState() {
    document.getElementById('statusWaiting').style.display = 'none';
    document.getElementById('statusRunning').style.display = 'block';
    document.getElementById('statusCrashed').style.display = 'none';
    
    // Disable betting, enable cash out if user has active bet
    document.getElementById('placeBetBtn').disabled = true;
    
    if (gameState.activeBet) {
        document.getElementById('cashOutBtn').style.display = 'block';
        updateCashOutButton();
    }
    
    // Start multiplier animation
    startMultiplierAnimation();
}

function showCrashedState() {
    document.getElementById('statusWaiting').style.display = 'none';
    document.getElementById('statusRunning').style.display = 'none';
    document.getElementById('statusCrashed').style.display = 'block';
    
    // Update crash multiplier
    document.getElementById('crashMultiplier').textContent = gameState.currentMultiplier.toFixed(2) + 'x';
    
    // Hide cash out button
    document.getElementById('cashOutBtn').style.display = 'none';
    
    // Play crash sound
    playSound('crash');
    
    // Show results after 3 seconds
    setTimeout(() => {
        showWaitingState();
    }, 3000);
}

function startCountdown() {
    let countdown = 5;
    const countdownElement = document.getElementById('countdown');
    
    const countdownInterval = setInterval(() => {
        countdownElement.textContent = countdown;
        countdown--;
        
        if (countdown < 0) {
            clearInterval(countdownInterval);
            // Start game round
            gameState.status = 'running';
            showRunningState();
        }
    }, 1000);
}

function startMultiplierAnimation() {
    gameState.startTime = Date.now();
    gameState.currentMultiplier = 1.00;
    
    const animationInterval = setInterval(() => {
        // Calculate multiplier based on time elapsed
        const elapsed = (Date.now() - gameState.startTime) / 1000;
        gameState.currentMultiplier = 1 + (elapsed * 0.1); // Simple linear growth
        
        // Update display
        document.getElementById('currentMultiplier').textContent = gameState.currentMultiplier.toFixed(2) + 'x';
        
        // Update chart
        updateCrashChart(gameState.currentMultiplier);
        
        // Check auto cashout
        if (gameState.autoCashout && gameState.activeBet && 
            gameState.currentMultiplier >= gameState.autoCashoutMultiplier) {
            cashOut();
        }
        
        // Random crash (for demo)
        if (Math.random() < 0.01 * elapsed) { // Increasing crash probability
            clearInterval(animationInterval);
            gameState.status = 'crashed';
            showCrashedState();
        }
        
        // Update cash out button
        if (gameState.activeBet) {
            updateCashOutButton();
        }
    }, 100);
}

function placeBet() {
    const betAmount = parseFloat(document.getElementById('betAmount').value);
    
    if (!betAmount || betAmount < 1) {
        Crashify.ui.toast.error('Please enter a valid bet amount');
        return;
    }
    
    if (betAmount > <?php echo $currentUser['balance']; ?>) {
        Crashify.ui.toast.error('Insufficient balance');
        return;
    }
    
    // Show loading
    const placeBetBtn = document.getElementById('placeBetBtn');
    placeBetBtn.classList.add('loading');
    placeBetBtn.disabled = true;
    
    // Place bet
    Crashify.api.request('/api/crash/bet', {
        method: 'POST',
        body: JSON.stringify({
            amount: betAmount,
            autoCashout: gameState.autoCashout,
            autoCashoutMultiplier: gameState.autoCashoutMultiplier
        })
    })
    .then(response => {
        if (response.success) {
            gameState.activeBet = response.bet;
            
            // Update UI
            showActiveBet();
            updateUserBalance(response.newBalance);
            
            // Play bet sound
            playSound('bet');
            
            Crashify.ui.toast.success('Bet placed successfully!');
        } else {
            throw new Error(response.message);
        }
    })
    .catch(error => {
        Crashify.ui.toast.error(error.message);
    })
    .finally(() => {
        placeBetBtn.classList.remove('loading');
        placeBetBtn.disabled = false;
    });
}

function cashOut() {
    if (!gameState.activeBet) return;
    
    // Show loading
    const cashOutBtn = document.getElementById('cashOutBtn');
    cashOutBtn.classList.add('loading');
    cashOutBtn.disabled = true;
    
    // Cash out
    Crashify.api.request('/api/crash/cashout', {
        method: 'POST',
        body: JSON.stringify({
            betId: gameState.activeBet.id,
            multiplier: gameState.currentMultiplier
        })
    })
    .then(response => {
        if (response.success) {
            const winAmount = response.winAmount;
            
            // Update UI
            hideActiveBet();
            updateUserBalance(response.newBalance);
            
            // Play cashout sound
            playSound('cashOut');
            
            // Show win notification
            Crashify.ui.toast.success(`Cashed out at ${gameState.currentMultiplier.toFixed(2)}x! Won $${winAmount.toFixed(2)}`);
            
            gameState.activeBet = null;
        } else {
            throw new Error(response.message);
        }
    })
    .catch(error => {
        Crashify.ui.toast.error(error.message);
    })
    .finally(() => {
        cashOutBtn.classList.remove('loading');
        cashOutBtn.disabled = false;
    });
}

function showActiveBet() {
    const activeBetElement = document.getElementById('activeBet');
    const bettingForm = document.getElementById('bettingForm');
    
    activeBetElement.style.display = 'block';
    bettingForm.style.display = 'none';
    
    document.getElementById('activeBetAmount').textContent = `$${gameState.activeBet.amount.toFixed(2)}`;
    updatePotentialWin();
}

function hideActiveBet() {
    const activeBetElement = document.getElementById('activeBet');
    const bettingForm = document.getElementById('bettingForm');
    
    activeBetElement.style.display = 'none';
    bettingForm.style.display = 'block';
}

function updatePotentialWin() {
    if (gameState.activeBet) {
        const potentialWin = gameState.activeBet.amount * gameState.currentMultiplier;
        document.getElementById('potentialWin').textContent = `$${potentialWin.toFixed(2)}`;
    }
}

function updateCashOutButton() {
    if (gameState.activeBet) {
        const cashOutAmount = gameState.activeBet.amount * gameState.currentMultiplier;
        document.getElementById('cashOutAmount').textContent = `$${cashOutAmount.toFixed(2)}`;
        updatePotentialWin();
    }
}

function setBetAmount(amount) {
    document.getElementById('betAmount').value = amount.toFixed(2);
}

function updateUserBalance(newBalance) {
    document.querySelector('.balance-amount').textContent = `$${newBalance.toFixed(2)}`;
    document.getElementById('headerBalance').textContent = `$${newBalance.toFixed(2)}`;
}

function playSound(soundName) {
    const audio = document.getElementById('sound' + soundName.charAt(0).toUpperCase() + soundName.slice(1));
    if (audio) {
        audio.currentTime = 0;
        audio.play().catch(e => console.log('Sound play failed:', e));
    }
}

// Chat functions
function toggleChat() {
    const chatPanel = document.querySelector('.chat-panel');
    chatPanel.classList.toggle('collapsed');
}

function handleChatKeyPress(event) {
    if (event.key === 'Enter') {
        sendChatMessage();
    }
}

function sendChatMessage() {
    const input = document.getElementById('chatMessageInput');
    const message = input.value.trim();
    
    if (!message) return;
    
    // Send message to server
    Crashify.api.request('/api/chat/send', {
        method: 'POST',
        body: JSON.stringify({ message })
    })
    .then(response => {
        if (response.success) {
            input.value = '';
        }
    })
    .catch(error => {
        Crashify.ui.toast.error('Failed to send message');
    });
}

// Chart functions (simplified)
function initializeCrashChart() {
    // Initialize chart canvas
    const canvas = document.getElementById('crashChart');
    const ctx = canvas.getContext('2d');
    
    // Basic chart setup
    ctx.strokeStyle = '#00ff88';
    ctx.lineWidth = 3;
    ctx.fillStyle = 'rgba(0, 255, 136, 0.1)';
}

function updateCrashChart(multiplier) {
    // Update chart with new multiplier point
    // This would be more complex in a real implementation
    console.log('Chart updated with multiplier:', multiplier);
}
</script>

<?php include '../includes/footer.php'; ?>
