// assets/js/casino.js - Casino Games JavaScript
// جاوااسکریپت بازی‌های کازینو

class CasinoManager {
    constructor() {
        this.games = new Map();
        this.activeGame = null;
        this.gameHistory = [];
        this.balance = 0;
        this.isPlaying = false;
        
        this.init();
    }
    
    init() {
        this.loadGames();
        this.setupEventListeners();
        this.loadBalance();
    }
    
    // Load available games
    async loadGames() {
        try {
            // Mock games data - in production, load from API
            const games = [
                {
                    id: 'slots',
                    name: 'اسلات',
                    type: 'slots',
                    minBet: 0.1,
                    maxBet: 100,
                    rtp: 96.5,
                    volatility: 'medium'
                },
                {
                    id: 'blackjack',
                    name: 'بلک جک',
                    type: 'table',
                    minBet: 1,
                    maxBet: 500,
                    rtp: 99.5,
                    volatility: 'low'
                },
                {
                    id: 'roulette',
                    name: 'رولت',
                    type: 'table',
                    minBet: 0.5,
                    maxBet: 1000,
                    rtp: 97.3,
                    volatility: 'medium'
                },
                {
                    id: 'baccarat',
                    name: 'باکارات',
                    type: 'table',
                    minBet: 2,
                    maxBet: 2000,
                    rtp: 98.9,
                    volatility: 'low'
                }
            ];
            
            games.forEach(game => {
                this.games.set(game.id, game);
            });
            
            this.renderGamesGrid();
        } catch (error) {
            console.error('Error loading games:', error);
        }
    }
    
    // Setup event listeners
    setupEventListeners() {
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('play-game-btn')) {
                this.startGame(e.target.dataset.gameId);
            }
            
            if (e.target.classList.contains('spin-btn')) {
                this.spinSlots();
            }
            
            if (e.target.classList.contains('deal-btn')) {
                this.dealCards();
            }
            
            if (e.target.classList.contains('hit-btn')) {
                this.hitCard();
            }
            
            if (e.target.classList.contains('stand-btn')) {
                this.standCards();
            }
            
            if (e.target.classList.contains('bet-red-btn')) {
                this.placeBet('red');
            }
            
            if (e.target.classList.contains('bet-black-btn')) {
                this.placeBet('black');
            }
            
            if (e.target.classList.contains('bet-number-btn')) {
                this.placeBet('number', e.target.dataset.number);
            }
        });
    }
    
    // Load user balance
    async loadBalance() {
        try {
            const response = await fetch('/api/wallet.php?endpoint=balance');
            const data = await response.json();
            
            if (data.success) {
                this.balance = data.data.balance;
                this.updateBalanceDisplay();
            }
        } catch (error) {
            console.error('Error loading balance:', error);
        }
    }
    
    // Render games grid
    renderGamesGrid() {
        const container = document.querySelector('.games-grid');
        if (!container) return;
        
        let html = '';
        this.games.forEach((game, gameId) => {
            html += `
                <div class="game-card" data-game-id="${gameId}">
                    <div class="game-image">
                        <img src="/assets/images/games/${gameId}.jpg" alt="${game.name}" 
                             onerror="this.src='/assets/images/games/default.jpg'">
                    </div>
                    <div class="game-info">
                        <h3 class="game-title">${game.name}</h3>
                        <div class="game-stats">
                            <span class="rtp">RTP: ${game.rtp}%</span>
                            <span class="volatility">${game.volatility}</span>
                        </div>
                        <div class="game-limits">
                            <span>حداقل: $${game.minBet}</span>
                            <span>حداکثر: $${game.maxBet}</span>
                        </div>
                        <button class="btn btn-primary play-game-btn" data-game-id="${gameId}">
                            بازی کنید
                        </button>
                    </div>
                </div>
            `;
        });
        
        container.innerHTML = html;
    }
    
    // Start a game
    async startGame(gameId) {
        const game = this.games.get(gameId);
        if (!game) return;
        
        this.activeGame = game;
        
        // Show game interface
        this.showGameInterface(game);
        
        // Initialize game state
        switch (game.type) {
            case 'slots':
                this.initSlots();
                break;
            case 'table':
                if (gameId === 'blackjack') {
                    this.initBlackjack();
                } else if (gameId === 'roulette') {
                    this.initRoulette();
                } else if (gameId === 'baccarat') {
                    this.initBaccarat();
                }
                break;
        }
    }
    
    // Show game interface
    showGameInterface(game) {
        const modal = document.createElement('div');
        modal.className = 'game-modal';
        modal.innerHTML = this.getGameHTML(game);
        
        document.body.appendChild(modal);
        
        // Add close functionality
        modal.querySelector('.close-game').addEventListener('click', () => {
            this.closeGame();
        });
        
        // Show modal
        setTimeout(() => modal.classList.add('show'), 10);
    }
    
    // Get game HTML
    getGameHTML(game) {
        switch (game.id) {
            case 'slots':
                return this.getSlotsHTML();
            case 'blackjack':
                return this.getBlackjackHTML();
            case 'roulette':
                return this.getRouletteHTML();
            case 'baccarat':
                return this.getBaccaratHTML();
            default:
                return '<div>Game not implemented</div>';
        }
    }
    
    // Slots game
    getSlotsHTML() {
        return `
            <div class="game-container slots-game">
                <div class="game-header">
                    <h2>اسلات ماشین</h2>
                    <button class="close-game">&times;</button>
                </div>
                <div class="game-content">
                    <div class="slots-machine">
                        <div class="reels">
                            <div class="reel" id="reel1">
                                <div class="symbol">🍒</div>
                                <div class="symbol">🍋</div>
                                <div class="symbol">🍊</div>
                            </div>
                            <div class="reel" id="reel2">
                                <div class="symbol">🍒</div>
                                <div class="symbol">🍋</div>
                                <div class="symbol">🍊</div>
                            </div>
                            <div class="reel" id="reel3">
                                <div class="symbol">🍒</div>
                                <div class="symbol">🍋</div>
                                <div class="symbol">🍊</div>
                            </div>
                        </div>
                        <div class="paylines"></div>
                    </div>
                    <div class="game-controls">
                        <div class="bet-controls">
                            <label>مبلغ شرط:</label>
                            <input type="number" class="bet-amount" min="0.1" max="100" value="1" step="0.1">
                        </div>
                        <button class="btn btn-primary spin-btn" ${this.isPlaying ? 'disabled' : ''}>
                            چرخاندن
                        </button>
                    </div>
                    <div class="game-info">
                        <div class="balance">موجودی: $<span class="balance-amount">${this.balance}</span></div>
                        <div class="last-win">آخرین برد: $<span class="last-win-amount">0</span></div>
                    </div>
                </div>
            </div>
        `;
    }
    
    // Initialize slots
    initSlots() {
        this.slotsSymbols = ['🍒', '🍋', '🍊', '🍇', '🍎', '💎', '⭐', '🔔'];
        this.slotsPaytable = {
            '🍒🍒🍒': 10,
            '🍋🍋🍋': 15,
            '🍊🍊🍊': 20,
            '🍇🍇🍇': 25,
            '🍎🍎🍎': 30,
            '💎💎💎': 100,
            '⭐⭐⭐': 200,
            '🔔🔔🔔': 500
        };
    }
    
    // Spin slots
    async spinSlots() {
        if (this.isPlaying) return;
        
        const betAmount = parseFloat(document.querySelector('.bet-amount').value);
        
        if (betAmount > this.balance) {
            this.showNotification('موجودی کافی نیست', 'error');
            return;
        }
        
        this.isPlaying = true;
        document.querySelector('.spin-btn').disabled = true;
        
        // Deduct bet amount
        this.balance -= betAmount;
        this.updateBalanceDisplay();
        
        // Animate reels
        const reels = document.querySelectorAll('.reel');
        reels.forEach((reel, index) => {
            this.animateReel(reel, index * 500);
        });
        
        // Wait for animation to complete
        setTimeout(() => {
            const result = this.getSpinResult();
            this.processSpinResult(result, betAmount);
            
            this.isPlaying = false;
            document.querySelector('.spin-btn').disabled = false;
        }, 2000);
    }
    
    // Animate reel
    animateReel(reel, delay) {
        setTimeout(() => {
            reel.classList.add('spinning');
            
            setTimeout(() => {
                // Set final symbol
                const randomSymbol = this.slotsSymbols[Math.floor(Math.random() * this.slotsSymbols.length)];
                reel.querySelector('.symbol').textContent = randomSymbol;
                reel.classList.remove('spinning');
            }, 1500);
        }, delay);
    }
    
    // Get spin result
    getSpinResult() {
        const reels = document.querySelectorAll('.reel .symbol');
        return Array.from(reels).map(symbol => symbol.textContent);
    }
    
    // Process spin result
    processSpinResult(result, betAmount) {
        const combination = result.join('');
        const payout = this.slotsPaytable[combination] || 0;
        const winAmount = payout * betAmount;
        
        if (winAmount > 0) {
            this.balance += winAmount;
            this.updateBalanceDisplay();
            document.querySelector('.last-win-amount').textContent = winAmount.toFixed(2);
            this.showNotification(`تبریک! شما $${winAmount.toFixed(2)} برنده شدید!`, 'success');
            
            // Highlight winning combination
            document.querySelectorAll('.reel').forEach(reel => {
                reel.classList.add('winner');
            });
            
            setTimeout(() => {
                document.querySelectorAll('.reel').forEach(reel => {
                    reel.classList.remove('winner');
                });
            }, 3000);
        }
        
        // Record game result
        this.recordGameResult('slots', betAmount, winAmount);
    }
    
    // Blackjack game
    getBlackjackHTML() {
        return `
            <div class="game-container blackjack-game">
                <div class="game-header">
                    <h2>بلک جک</h2>
                    <button class="close-game">&times;</button>
                </div>
                <div class="game-content">
                    <div class="dealer-area">
                        <h3>دیلر</h3>
                        <div class="dealer-cards"></div>
                        <div class="dealer-score">امتیاز: <span>0</span></div>
                    </div>
                    <div class="player-area">
                        <h3>شما</h3>
                        <div class="player-cards"></div>
                        <div class="player-score">امتیاز: <span>0</span></div>
                    </div>
                    <div class="game-controls">
                        <div class="bet-controls">
                            <label>مبلغ شرط:</label>
                            <input type="number" class="bet-amount" min="1" max="500" value="10" step="1">
                            <button class="btn btn-primary deal-btn">شروع بازی</button>
                        </div>
                        <div class="play-controls" style="display: none;">
                            <button class="btn btn-success hit-btn">کارت بگیر</button>
                            <button class="btn btn-warning stand-btn">بایست</button>
                        </div>
                    </div>
                    <div class="game-info">
                        <div class="balance">موجودی: $<span class="balance-amount">${this.balance}</span></div>
                        <div class="game-status"></div>
                    </div>
                </div>
            </div>
        `;
    }
    
    // Initialize blackjack
    initBlackjack() {
        this.deck = this.createDeck();
        this.playerCards = [];
        this.dealerCards = [];
        this.gameState = 'betting'; // betting, playing, finished
    }
    
    // Create deck of cards
    createDeck() {
        const suits = ['♠', '♥', '♦', '♣'];
        const ranks = ['A', '2', '3', '4', '5', '6', '7', '8', '9', '10', 'J', 'Q', 'K'];
        const deck = [];
        
        suits.forEach(suit => {
            ranks.forEach(rank => {
                deck.push({ suit, rank, value: this.getCardValue(rank) });
            });
        });
        
        return this.shuffleDeck(deck);
    }
    
    // Get card value for blackjack
    getCardValue(rank) {
        if (rank === 'A') return 11;
        if (['J', 'Q', 'K'].includes(rank)) return 10;
        return parseInt(rank);
    }
    
    // Shuffle deck
    shuffleDeck(deck) {
        for (let i = deck.length - 1; i > 0; i--) {
            const j = Math.floor(Math.random() * (i + 1));
            [deck[i], deck[j]] = [deck[j], deck[i]];
        }
        return deck;
    }
    
    // Deal initial cards
    dealCards() {
        const betAmount = parseFloat(document.querySelector('.bet-amount').value);
        
        if (betAmount > this.balance) {
            this.showNotification('موجودی کافی نیست', 'error');
            return;
        }
        
        // Deduct bet amount
        this.balance -= betAmount;
        this.updateBalanceDisplay();
        
        // Reset game
        this.playerCards = [];
        this.dealerCards = [];
        
        // Deal initial cards
        this.playerCards.push(this.drawCard());
        this.dealerCards.push(this.drawCard());
        this.playerCards.push(this.drawCard());
        this.dealerCards.push(this.drawCard());
        
        this.updateBlackjackDisplay();
        
        // Check for blackjack
        if (this.calculateScore(this.playerCards) === 21) {
            this.endBlackjackGame('blackjack', betAmount);
        } else {
            this.gameState = 'playing';
            document.querySelector('.bet-controls').style.display = 'none';
            document.querySelector('.play-controls').style.display = 'block';
        }
    }
    
    // Draw a card from deck
    drawCard() {
        if (this.deck.length === 0) {
            this.deck = this.createDeck();
        }
        return this.deck.pop();
    }
    
    // Hit (take another card)
    hitCard() {
        if (this.gameState !== 'playing') return;
        
        this.playerCards.push(this.drawCard());
        this.updateBlackjackDisplay();
        
        const playerScore = this.calculateScore(this.playerCards);
        if (playerScore > 21) {
            this.endBlackjackGame('bust', parseFloat(document.querySelector('.bet-amount').value));
        }
    }
    
    // Stand (end turn)
    standCards() {
        if (this.gameState !== 'playing') return;
        
        // Dealer plays
        while (this.calculateScore(this.dealerCards) < 17) {
            this.dealerCards.push(this.drawCard());
        }
        
        this.updateBlackjackDisplay();
        
        const playerScore = this.calculateScore(this.playerCards);
        const dealerScore = this.calculateScore(this.dealerCards);
        const betAmount = parseFloat(document.querySelector('.bet-amount').value);
        
        if (dealerScore > 21) {
            this.endBlackjackGame('dealer_bust', betAmount);
        } else if (playerScore > dealerScore) {
            this.endBlackjackGame('win', betAmount);
        } else if (playerScore < dealerScore) {
            this.endBlackjackGame('lose', betAmount);
        } else {
            this.endBlackjackGame('push', betAmount);
        }
    }
    
    // Calculate blackjack score
    calculateScore(cards) {
        let score = 0;
        let aces = 0;
        
        cards.forEach(card => {
            if (card.rank === 'A') {
                aces++;
                score += 11;
            } else {
                score += card.value;
            }
        });
        
        // Adjust for aces
        while (score > 21 && aces > 0) {
            score -= 10;
            aces--;
        }
        
        return score;
    }
    
    // Update blackjack display
    updateBlackjackDisplay() {
        const playerCardsEl = document.querySelector('.player-cards');
        const dealerCardsEl = document.querySelector('.dealer-cards');
        const playerScoreEl = document.querySelector('.player-score span');
        const dealerScoreEl = document.querySelector('.dealer-score span');
        
        // Show player cards
        playerCardsEl.innerHTML = this.playerCards.map(card => 
            `<div class="card">${card.rank}${card.suit}</div>`
        ).join('');
        
        // Show dealer cards (hide second card if game is still playing)
        if (this.gameState === 'playing') {
            dealerCardsEl.innerHTML = `
                <div class="card">${this.dealerCards[0].rank}${this.dealerCards[0].suit}</div>
                <div class="card card-back">?</div>
            `;
            dealerScoreEl.textContent = this.dealerCards[0].value;
        } else {
            dealerCardsEl.innerHTML = this.dealerCards.map(card => 
                `<div class="card">${card.rank}${card.suit}</div>`
            ).join('');
            dealerScoreEl.textContent = this.calculateScore(this.dealerCards);
        }
        
        playerScoreEl.textContent = this.calculateScore(this.playerCards);
    }
    
    // End blackjack game
    endBlackjackGame(result, betAmount) {
        this.gameState = 'finished';
        
        let winAmount = 0;
        let message = '';
        
        switch (result) {
            case 'blackjack':
                winAmount = betAmount * 2.5;
                message = 'بلک جک! تبریک!';
                break;
            case 'win':
            case 'dealer_bust':
                winAmount = betAmount * 2;
                message = 'شما برنده شدید!';
                break;
            case 'push':
                winAmount = betAmount;
                message = 'مساوی!';
                break;
            case 'lose':
            case 'bust':
                winAmount = 0;
                message = 'شما باختید!';
                break;
        }
        
        if (winAmount > 0) {
            this.balance += winAmount;
            this.updateBalanceDisplay();
        }
        
        document.querySelector('.game-status').textContent = message;
        document.querySelector('.play-controls').style.display = 'none';
        document.querySelector('.bet-controls').style.display = 'block';
        
        this.updateBlackjackDisplay();
        this.recordGameResult('blackjack', betAmount, winAmount - betAmount);
    }
    
    // Roulette game
    getRouletteHTML() {
        return `
            <div class="game-container roulette-game">
                <div class="game-header">
                    <h2>رولت</h2>
                    <button class="close-game">&times;</button>
                </div>
                <div class="game-content">
                    <div class="roulette-wheel">
                        <div class="wheel" id="roulette-wheel">
                            <div class="ball"></div>
                        </div>
                        <div class="winning-number">
                            <span id="winning-number">-</span>
                        </div>
                    </div>
                    <div class="betting-table">
                        <div class="color-bets">
                            <button class="bet-btn bet-red-btn" data-payout="2">قرمز (1:1)</button>
                            <button class="bet-btn bet-black-btn" data-payout="2">سیاه (1:1)</button>
                        </div>
                        <div class="number-grid">
                            ${Array.from({length: 37}, (_, i) => 
                                `<button class="bet-btn bet-number-btn" data-number="${i}" data-payout="36">${i}</button>`
                            ).join('')}
                        </div>
                    </div>
                    <div class="game-controls">
                        <div class="bet-controls">
                            <label>مبلغ شرط:</label>
                            <input type="number" class="bet-amount" min="0.5" max="1000" value="5" step="0.5">
                            <button class="btn btn-primary spin-roulette-btn">چرخاندن</button>
                        </div>
                        <div class="active-bets"></div>
                    </div>
                    <div class="game-info">
                        <div class="balance">موجودی: $<span class="balance-amount">${this.balance}</span></div>
                        <div class="total-bet">کل شرط: $<span class="total-bet-amount">0</span></div>
                    </div>
                </div>
            </div>
        `;
    }
    
    // Close game
    closeGame() {
        const modal = document.querySelector('.game-modal');
        if (modal) {
            modal.classList.remove('show');
            setTimeout(() => {
                document.body.removeChild(modal);
            }, 300);
        }
        
        this.activeGame = null;
        this.isPlaying = false;
    }
    
    // Record game result
    recordGameResult(gameType, betAmount, winAmount) {
        const result = {
            game: gameType,
            bet: betAmount,
            win: winAmount,
            profit: winAmount - betAmount,
            timestamp: Date.now()
        };
        
        this.gameHistory.push(result);
        
        // Keep only last 100 results
        if (this.gameHistory.length > 100) {
            this.gameHistory.shift();
        }
        
        // Send to server (optional)
        this.sendGameResult(result);
    }
    
    // Send game result to server
    async sendGameResult(result) {
        try {
            await fetch('/api/bets.php?endpoint=place', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    game_type: 'casino',
                    bet_amount: result.bet,
                    bet_data: {
                        game_id: result.game,
                        result: result
                    }
                })
            });
        } catch (error) {
            console.error('Error sending game result:', error);
        }
    }
    
    // Update balance display
    updateBalanceDisplay() {
        const balanceElements = document.querySelectorAll('.balance-amount');
        balanceElements.forEach(element => {
            element.textContent = this.balance.toFixed(2);
        });
    }
    
    // Show notification
    showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `alert alert-${type} notification`;
        notification.textContent = message;
        
        let container = document.querySelector('.notifications-container');
        if (!container) {
            container = document.createElement('div');
            container.className = 'notifications-container';
            document.body.appendChild(container);
        }
        
        container.appendChild(notification);
        
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 5000);
    }
}

// Initialize casino manager
document.addEventListener('DOMContentLoaded', () => {
    window.casinoManager = new CasinoManager();
});

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = CasinoManager;
}
