// assets/js/betting.js - Enhanced Betting JavaScript
// جاوااسکریپت پیشرفته شرط‌بندی

class BettingManager {
    constructor() {
        this.activeBets = new Map();
        this.betHistory = [];
        this.balance = 0;
        this.minBet = 1;
        this.maxBet = 10000;
        this.isConnected = false;
        this.eventListeners = new Map();
        
        this.init();
    }
    
    init() {
        this.loadUserData();
        this.setupEventListeners();
        this.connectWebSocket();
        this.startPeriodicUpdates();
    }
    
    // Load user data
    async loadUserData() {
        try {
            const response = await fetch('/api/wallet.php?endpoint=balance');
            const data = await response.json();
            
            if (data.success) {
                this.balance = data.data.balance;
                this.updateBalanceDisplay();
            }
        } catch (error) {
            console.error('Error loading user data:', error);
            this.showNotification('خطا در بارگذاری اطلاعات کاربر', 'error');
        }
    }
    
    // Setup event listeners
    setupEventListeners() {
        // Bet placement
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('place-bet-btn')) {
                this.handleBetPlacement(e.target);
            }
            
            if (e.target.classList.contains('cashout-btn')) {
                this.handleCashout(e.target);
            }
            
            if (e.target.classList.contains('cancel-bet-btn')) {
                this.handleBetCancellation(e.target);
            }
        });
        
        // Bet amount validation
        document.addEventListener('input', (e) => {
            if (e.target.classList.contains('bet-amount-input')) {
                this.validateBetAmount(e.target);
            }
        });
        
        // Quick bet buttons
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('quick-bet-btn')) {
                this.handleQuickBet(e.target);
            }
        });
    }
    
    // WebSocket connection for real-time updates
    connectWebSocket() {
        try {
            this.ws = new WebSocket('wss://crashify.com/ws');
            
            this.ws.onopen = () => {
                this.isConnected = true;
                console.log('WebSocket connected');
                this.showNotification('اتصال برقرار شد', 'success');
            };
            
            this.ws.onmessage = (event) => {
                const data = JSON.parse(event.data);
                this.handleWebSocketMessage(data);
            };
            
            this.ws.onclose = () => {
                this.isConnected = false;
                console.log('WebSocket disconnected');
                this.showNotification('اتصال قطع شد', 'warning');
                
                // Reconnect after 5 seconds
                setTimeout(() => this.connectWebSocket(), 5000);
            };
            
            this.ws.onerror = (error) => {
                console.error('WebSocket error:', error);
                this.showNotification('خطا در اتصال', 'error');
            };
        } catch (error) {
            console.error('Failed to connect WebSocket:', error);
        }
    }
    
    // Handle WebSocket messages
    handleWebSocketMessage(data) {
        switch (data.type) {
            case 'odds_update':
                this.updateOdds(data.odds);
                break;
            case 'match_update':
                this.updateMatchStatus(data.match);
                break;
            case 'bet_result':
                this.handleBetResult(data.result);
                break;
            case 'balance_update':
                this.updateBalance(data.balance);
                break;
            case 'crash_update':
                this.updateCrashMultiplier(data.multiplier);
                break;
        }
    }
    
    // Place a bet
    async placeBet(gameType, betData) {
        try {
            // Validate bet amount
            if (betData.bet_amount < this.minBet || betData.bet_amount > this.maxBet) {
                throw new Error(`مبلغ شرط باید بین ${this.minBet} تا ${this.maxBet} باشد`);
            }
            
            if (betData.bet_amount > this.balance) {
                throw new Error('موجودی کافی نیست');
            }
            
            // Show loading
            this.showLoading('در حال ثبت شرط...');
            
            const response = await fetch('/api/bets.php?endpoint=place', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    game_type: gameType,
                    bet_amount: betData.bet_amount,
                    bet_data: betData
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                // Add to active bets
                this.activeBets.set(result.data.bet_id, {
                    ...result.data,
                    timestamp: Date.now()
                });
                
                // Update balance
                this.balance -= betData.bet_amount;
                this.updateBalanceDisplay();
                
                // Update UI
                this.updateActiveBetsDisplay();
                this.showNotification('شرط با موفقیت ثبت شد', 'success');
                
                // Trigger custom event
                this.triggerEvent('betPlaced', result.data);
                
                return result.data;
            } else {
                throw new Error(result.error || 'خطا در ثبت شرط');
            }
        } catch (error) {
            console.error('Error placing bet:', error);
            this.showNotification(error.message, 'error');
            throw error;
        } finally {
            this.hideLoading();
        }
    }
    
    // Handle bet placement from UI
    async handleBetPlacement(button) {
        const betForm = button.closest('.bet-form');
        if (!betForm) return;
        
        const gameType = betForm.dataset.gameType;
        const betAmount = parseFloat(betForm.querySelector('.bet-amount-input').value);
        const betData = this.extractBetData(betForm);
        
        try {
            await this.placeBet(gameType, {
                bet_amount: betAmount,
                ...betData
            });
        } catch (error) {
            // Error already handled in placeBet
        }
    }
    
    // Extract bet data from form
    extractBetData(form) {
        const data = {};
        
        // Get all form inputs
        const inputs = form.querySelectorAll('input, select');
        inputs.forEach(input => {
            if (input.name && input.value) {
                data[input.name] = input.value;
            }
        });
        
        // Get selected odds
        const selectedOdds = form.querySelector('.odds-button.selected');
        if (selectedOdds) {
            data.odds = parseFloat(selectedOdds.dataset.odds);
            data.bet_type = selectedOdds.dataset.betType;
            data.match_id = selectedOdds.dataset.matchId;
        }
        
        return data;
    }
    
    // Cashout bet
    async cashoutBet(betId, multiplier = null) {
        try {
            this.showLoading('در حال تسویه...');
            
            const response = await fetch('/api/bets.php?endpoint=cashout', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    bet_id: betId,
                    multiplier: multiplier
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                // Remove from active bets
                this.activeBets.delete(betId);
                
                // Update balance
                this.balance += result.data.payout_amount;
                this.updateBalanceDisplay();
                
                // Update UI
                this.updateActiveBetsDisplay();
                this.showNotification(`تسویه موفق! برد: ${result.data.profit}`, 'success');
                
                // Trigger custom event
                this.triggerEvent('betCashedOut', result.data);
                
                return result.data;
            } else {
                throw new Error(result.error || 'خطا در تسویه');
            }
        } catch (error) {
            console.error('Error cashing out bet:', error);
            this.showNotification(error.message, 'error');
            throw error;
        } finally {
            this.hideLoading();
        }
    }
    
    // Handle cashout from UI
    async handleCashout(button) {
        const betId = parseInt(button.dataset.betId);
        const multiplier = button.dataset.multiplier ? parseFloat(button.dataset.multiplier) : null;
        
        try {
            await this.cashoutBet(betId, multiplier);
        } catch (error) {
            // Error already handled in cashoutBet
        }
    }
    
    // Cancel bet
    async cancelBet(betId) {
        try {
            this.showLoading('در حال لغو شرط...');
            
            const response = await fetch('/api/bets.php?endpoint=cancel', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    bet_id: betId
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                // Remove from active bets
                this.activeBets.delete(betId);
                
                // Update balance
                this.balance += result.data.refund_amount;
                this.updateBalanceDisplay();
                
                // Update UI
                this.updateActiveBetsDisplay();
                this.showNotification('شرط لغو شد', 'info');
                
                // Trigger custom event
                this.triggerEvent('betCancelled', result.data);
                
                return result.data;
            } else {
                throw new Error(result.error || 'خطا در لغو شرط');
            }
        } catch (error) {
            console.error('Error cancelling bet:', error);
            this.showNotification(error.message, 'error');
            throw error;
        } finally {
            this.hideLoading();
        }
    }
    
    // Handle bet cancellation from UI
    async handleBetCancellation(button) {
        const betId = parseInt(button.dataset.betId);
        
        if (confirm('آیا از لغو این شرط اطمینان دارید؟')) {
            try {
                await this.cancelBet(betId);
            } catch (error) {
                // Error already handled in cancelBet
            }
        }
    }
    
    // Quick bet functionality
    handleQuickBet(button) {
        const amount = parseFloat(button.dataset.amount);
        const betInput = button.closest('.bet-controls').querySelector('.bet-amount-input');
        
        if (betInput) {
            if (button.dataset.action === 'set') {
                betInput.value = amount;
            } else if (button.dataset.action === 'add') {
                const currentAmount = parseFloat(betInput.value) || 0;
                betInput.value = currentAmount + amount;
            } else if (button.dataset.action === 'multiply') {
                const currentAmount = parseFloat(betInput.value) || 0;
                betInput.value = currentAmount * amount;
            } else if (button.dataset.action === 'max') {
                betInput.value = Math.min(this.balance, this.maxBet);
            } else if (button.dataset.action === 'half') {
                const currentAmount = parseFloat(betInput.value) || 0;
                betInput.value = currentAmount / 2;
            }
            
            this.validateBetAmount(betInput);
        }
    }
    
    // Validate bet amount
    validateBetAmount(input) {
        const amount = parseFloat(input.value);
        const errorElement = input.parentElement.querySelector('.bet-error');
        
        let isValid = true;
        let errorMessage = '';
        
        if (isNaN(amount) || amount <= 0) {
            isValid = false;
            errorMessage = 'مبلغ نامعتبر';
        } else if (amount < this.minBet) {
            isValid = false;
            errorMessage = `حداقل مبلغ: ${this.minBet}`;
        } else if (amount > this.maxBet) {
            isValid = false;
            errorMessage = `حداکثر مبلغ: ${this.maxBet}`;
        } else if (amount > this.balance) {
            isValid = false;
            errorMessage = 'موجودی کافی نیست';
        }
        
        // Update UI
        input.classList.toggle('is-invalid', !isValid);
        input.classList.toggle('is-valid', isValid);
        
        if (errorElement) {
            errorElement.textContent = errorMessage;
            errorElement.style.display = errorMessage ? 'block' : 'none';
        }
        
        // Enable/disable bet button
        const betButton = input.closest('.bet-form').querySelector('.place-bet-btn');
        if (betButton) {
            betButton.disabled = !isValid;
        }
        
        return isValid;
    }
    
    // Update odds display
    updateOdds(oddsData) {
        oddsData.forEach(odd => {
            const oddsButton = document.querySelector(`[data-match-id="${odd.match_id}"][data-bet-type="${odd.bet_type}"]`);
            if (oddsButton) {
                const oldOdds = parseFloat(oddsButton.dataset.odds);
                const newOdds = odd.odds;
                
                oddsButton.dataset.odds = newOdds;
                oddsButton.querySelector('.odds-value').textContent = newOdds.toFixed(2);
                
                // Show change indicator
                if (newOdds > oldOdds) {
                    oddsButton.classList.add('odds-up');
                    setTimeout(() => oddsButton.classList.remove('odds-up'), 2000);
                } else if (newOdds < oldOdds) {
                    oddsButton.classList.add('odds-down');
                    setTimeout(() => oddsButton.classList.remove('odds-down'), 2000);
                }
            }
        });
    }
    
    // Update match status
    updateMatchStatus(matchData) {
        const matchElement = document.querySelector(`[data-match-id="${matchData.id}"]`);
        if (matchElement) {
            const statusElement = matchElement.querySelector('.match-status');
            const scoreElement = matchElement.querySelector('.match-score');
            const timeElement = matchElement.querySelector('.match-time');
            
            if (statusElement) statusElement.textContent = matchData.status;
            if (scoreElement) scoreElement.textContent = matchData.score;
            if (timeElement) timeElement.textContent = matchData.time;
        }
    }
    
    // Handle bet result
    handleBetResult(resultData) {
        const bet = this.activeBets.get(resultData.bet_id);
        if (bet) {
            this.activeBets.delete(resultData.bet_id);
            
            if (resultData.status === 'won') {
                this.balance += resultData.payout_amount;
                this.showNotification(`تبریک! شما برنده شدید: ${resultData.profit}`, 'success');
            } else if (resultData.status === 'lost') {
                this.showNotification('متاسفانه شرط شما باخت', 'error');
            }
            
            this.updateBalanceDisplay();
            this.updateActiveBetsDisplay();
            
            // Trigger custom event
            this.triggerEvent('betResult', resultData);
        }
    }
    
    // Update balance
    updateBalance(newBalance) {
        this.balance = newBalance;
        this.updateBalanceDisplay();
    }
    
    // Update crash multiplier
    updateCrashMultiplier(multiplier) {
        const multiplierElements = document.querySelectorAll('.crash-multiplier');
        multiplierElements.forEach(element => {
            element.textContent = multiplier.toFixed(2) + 'x';
        });
        
        // Update potential payout for active crash bets
        this.activeBets.forEach((bet, betId) => {
            if (bet.game_type === 'crash' && bet.status === 'active') {
                const potentialPayout = bet.bet_amount * multiplier;
                const payoutElement = document.querySelector(`[data-bet-id="${betId}"] .potential-payout`);
                if (payoutElement) {
                    payoutElement.textContent = potentialPayout.toFixed(2);
                }
            }
        });
    }
    
    // Update balance display
    updateBalanceDisplay() {
        const balanceElements = document.querySelectorAll('.user-balance');
        balanceElements.forEach(element => {
            element.textContent = this.formatCurrency(this.balance);
        });
    }
    
    // Update active bets display
    updateActiveBetsDisplay() {
        const container = document.querySelector('.active-bets-container');
        if (!container) return;
        
        if (this.activeBets.size === 0) {
            container.innerHTML = '<p class="text-muted text-center">شرط فعالی وجود ندارد</p>';
            return;
        }
        
        let html = '';
        this.activeBets.forEach((bet, betId) => {
            html += this.generateActiveBetHTML(bet, betId);
        });
        
        container.innerHTML = html;
    }
    
    // Generate HTML for active bet
    generateActiveBetHTML(bet, betId) {
        return `
            <div class="active-bet-item" data-bet-id="${betId}">
                <div class="bet-info">
                    <span class="bet-game">${bet.game_type}</span>
                    <span class="bet-amount">${this.formatCurrency(bet.bet_amount)}</span>
                </div>
                <div class="bet-actions">
                    ${bet.game_type === 'crash' ? 
                        `<button class="btn btn-sm btn-success cashout-btn" data-bet-id="${betId}">
                            تسویه
                        </button>` : 
                        `<button class="btn btn-sm btn-danger cancel-bet-btn" data-bet-id="${betId}">
                            لغو
                        </button>`
                    }
                </div>
            </div>
        `;
    }
    
    // Load bet history
    async loadBetHistory(page = 1, filters = {}) {
        try {
            const params = new URLSearchParams({
                page: page,
                limit: 20,
                ...filters
            });
            
            const response = await fetch(`/api/bets.php?endpoint=history&${params}`);
            const data = await response.json();
            
            if (data.success) {
                this.betHistory = data.data.bets;
                this.updateBetHistoryDisplay(data.data);
                return data.data;
            } else {
                throw new Error(data.error || 'خطا در بارگذاری تاریخچه');
            }
        } catch (error) {
            console.error('Error loading bet history:', error);
            this.showNotification(error.message, 'error');
            throw error;
        }
    }
    
    // Update bet history display
    updateBetHistoryDisplay(historyData) {
        const container = document.querySelector('.bet-history-container');
        if (!container) return;
        
        let html = '';
        historyData.bets.forEach(bet => {
            html += this.generateBetHistoryHTML(bet);
        });
        
        container.innerHTML = html;
        
        // Update pagination
        this.updatePagination(historyData.pagination);
    }
    
    // Generate HTML for bet history item
    generateBetHistoryHTML(bet) {
        const statusClass = {
            'won': 'success',
            'lost': 'danger',
            'pending': 'warning',
            'cancelled': 'secondary'
        }[bet.status] || 'secondary';
        
        return `
            <div class="bet-history-item">
                <div class="bet-details">
                    <div class="bet-game">${bet.game_type}</div>
                    <div class="bet-amount">${this.formatCurrency(bet.bet_amount)}</div>
                    <div class="bet-date">${this.formatDate(bet.created_at)}</div>
                </div>
                <div class="bet-result">
                    <span class="badge badge-${statusClass}">${bet.status}</span>
                    ${bet.payout_amount ? 
                        `<div class="payout">${this.formatCurrency(bet.payout_amount)}</div>` : 
                        ''
                    }
                </div>
            </div>
        `;
    }
    
    // Periodic updates
    startPeriodicUpdates() {
        // Update active bets every 5 seconds
        setInterval(() => {
            if (this.activeBets.size > 0) {
                this.updateActiveBets();
            }
        }, 5000);
        
        // Update balance every 30 seconds
        setInterval(() => {
            this.loadUserData();
        }, 30000);
    }
    
    // Update active bets from server
    async updateActiveBets() {
        try {
            const response = await fetch('/api/bets.php?endpoint=active');
            const data = await response.json();
            
            if (data.success) {
                // Update local active bets
                this.activeBets.clear();
                data.data.active_bets.forEach(bet => {
                    this.activeBets.set(bet.id, bet);
                });
                
                this.updateActiveBetsDisplay();
            }
        } catch (error) {
            console.error('Error updating active bets:', error);
        }
    }
    
    // Event system
    addEventListener(event, callback) {
        if (!this.eventListeners.has(event)) {
            this.eventListeners.set(event, []);
        }
        this.eventListeners.get(event).push(callback);
    }
    
    removeEventListener(event, callback) {
        if (this.eventListeners.has(event)) {
            const listeners = this.eventListeners.get(event);
            const index = listeners.indexOf(callback);
            if (index > -1) {
                listeners.splice(index, 1);
            }
        }
    }
    
    triggerEvent(event, data) {
        if (this.eventListeners.has(event)) {
            this.eventListeners.get(event).forEach(callback => {
                try {
                    callback(data);
                } catch (error) {
                    console.error('Error in event listener:', error);
                }
            });
        }
    }
    
    // Utility functions
    formatCurrency(amount) {
        return new Intl.NumberFormat('fa-IR', {
            style: 'currency',
            currency: 'USD',
            minimumFractionDigits: 2
        }).format(amount);
    }
    
    formatDate(dateString) {
        return new Intl.DateTimeFormat('fa-IR', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        }).format(new Date(dateString));
    }
    
    showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `alert alert-${type} notification fade-in`;
        notification.textContent = message;
        
        // Add to container
        let container = document.querySelector('.notifications-container');
        if (!container) {
            container = document.createElement('div');
            container.className = 'notifications-container';
            document.body.appendChild(container);
        }
        
        container.appendChild(notification);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            notification.classList.add('fade-out');
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 300);
        }, 5000);
    }
    
    showLoading(message = 'در حال بارگذاری...') {
        let loader = document.querySelector('.loading-overlay');
        if (!loader) {
            loader = document.createElement('div');
            loader.className = 'loading-overlay';
            loader.innerHTML = `
                <div class="loading-content">
                    <div class="spinner"></div>
                    <div class="loading-message">${message}</div>
                </div>
            `;
            document.body.appendChild(loader);
        } else {
            loader.querySelector('.loading-message').textContent = message;
        }
        
        loader.style.display = 'flex';
    }
    
    hideLoading() {
        const loader = document.querySelector('.loading-overlay');
        if (loader) {
            loader.style.display = 'none';
        }
    }
    
    updatePagination(paginationData) {
        const container = document.querySelector('.pagination-container');
        if (!container) return;
        
        let html = '<nav><ul class="pagination">';
        
        // Previous button
        if (paginationData.current_page > 1) {
            html += `<li class="page-item">
                <a class="page-link" href="#" data-page="${paginationData.current_page - 1}">قبلی</a>
            </li>`;
        }
        
        // Page numbers
        for (let i = 1; i <= paginationData.total_pages; i++) {
            const isActive = i === paginationData.current_page;
            html += `<li class="page-item ${isActive ? 'active' : ''}">
                <a class="page-link" href="#" data-page="${i}">${i}</a>
            </li>`;
        }
        
        // Next button
        if (paginationData.current_page < paginationData.total_pages) {
            html += `<li class="page-item">
                <a class="page-link" href="#" data-page="${paginationData.current_page + 1}">بعدی</a>
            </li>`;
        }
        
        html += '</ul></nav>';
        container.innerHTML = html;
        
        // Add click handlers
        container.querySelectorAll('.page-link').forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                const page = parseInt(e.target.dataset.page);
                this.loadBetHistory(page);
            });
        });
    }
}

// Initialize betting manager when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.bettingManager = new BettingManager();
});

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = BettingManager;
}