<?php
// api/bets.php - Bets API
// API شرط بندی

require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

$method = $_SERVER['REQUEST_METHOD'];
$endpoint = $_GET['endpoint'] ?? '';

try {
    // Require authentication for all bet operations
    requireAuth();
    
    switch ($endpoint) {
        case 'place':
            handlePlaceBetRequest($method);
            break;
            
        case 'history':
            handleBetHistoryRequest($method);
            break;
            
        case 'active':
            handleActiveBetsRequest($method);
            break;
            
        case 'cancel':
            handleCancelBetRequest($method);
            break;
            
        case 'cashout':
            handleCashoutRequest($method);
            break;
            
        case 'statistics':
            handleBetStatisticsRequest($method);
            break;
            
        default:
            throw new Exception('Invalid endpoint');
    }
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

function handlePlaceBetRequest($method) {
    global $currentUser;
    
    if ($method !== 'POST') {
        throw new Exception('Method not allowed');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    $gameType = $input['game_type'] ?? '';
    $betAmount = floatval($input['bet_amount'] ?? 0);
    $betData = $input['bet_data'] ?? [];
    
    if (!$gameType) {
        throw new Exception('Game type is required');
    }
    
    if ($betAmount <= 0) {
        throw new Exception('Invalid bet amount');
    }
    
    // Check betting limits
    $minBet = getSetting('min_bet_amount', 1.00);
    $maxBet = getSetting('max_bet_amount', 10000.00);
    
    if ($betAmount < $minBet) {
        throw new Exception("Minimum bet amount is $" . number_format($minBet, 2));
    }
    
    if ($betAmount > $maxBet) {
        throw new Exception("Maximum bet amount is $" . number_format($maxBet, 2));
    }
    
    // Check if betting is enabled
    if (!getSetting('betting_enabled', true)) {
        throw new Exception('Betting is currently disabled');
    }
    
    // Check user balance
    $user = fetchRow("SELECT balance FROM users WHERE id = ?", [$currentUser['id']]);
    
    if ($user['balance'] < $betAmount) {
        throw new Exception('Insufficient balance');
    }
    
    // Handle different game types
    switch ($gameType) {
        case 'crash':
            $betId = placeCrashBet($currentUser['id'], $betAmount, $betData);
            break;
            
        case 'sports':
            $betId = placeSportsBet($currentUser['id'], $betAmount, $betData);
            break;
            
        case 'casino':
            $betId = placeCasinoBet($currentUser['id'], $betAmount, $betData);
            break;
            
        default:
            throw new Exception('Unsupported game type');
    }
    
    // Deduct bet amount from user balance
    execute("UPDATE users SET balance = balance - ? WHERE id = ?", [$betAmount, $currentUser['id']]);
    
    // Create transaction record
    execute(
        "INSERT INTO transactions (user_id, type, amount, status, description, created_at) 
         VALUES (?, 'bet', ?, 'completed', ?, NOW())",
        [$currentUser['id'], $betAmount, "Bet placed on $gameType"]
    );
    
    echo json_encode([
        'success' => true,
        'data' => [
            'bet_id' => $betId,
            'game_type' => $gameType,
            'bet_amount' => $betAmount,
            'status' => 'active',
            'placed_at' => date('Y-m-d H:i:s')
        ]
    ]);
}

function handleBetHistoryRequest($method) {
    global $currentUser;
    
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    $page = intval($_GET['page'] ?? 1);
    $limit = intval($_GET['limit'] ?? 20);
    $gameType = $_GET['game_type'] ?? 'all';
    $status = $_GET['status'] ?? 'all';
    $dateFrom = $_GET['date_from'] ?? '';
    $dateTo = $_GET['date_to'] ?? '';
    
    $offset = ($page - 1) * $limit;
    
    // Build query conditions
    $conditions = ["user_id = ?"];
    $params = [$currentUser['id']];
    
    if ($gameType !== 'all') {
        $conditions[] = "game_type = ?";
        $params[] = $gameType;
    }
    
    if ($status !== 'all') {
        $conditions[] = "status = ?";
        $params[] = $status;
    }
    
    if ($dateFrom) {
        $conditions[] = "DATE(created_at) >= ?";
        $params[] = $dateFrom;
    }
    
    if ($dateTo) {
        $conditions[] = "DATE(created_at) <= ?";
        $params[] = $dateTo;
    }
    
    $whereClause = implode(' AND ', $conditions);
    
    // Get total count
    $totalCount = fetchRow(
        "SELECT COUNT(*) as count FROM user_bets WHERE $whereClause",
        $params
    )['count'];
    
    // Get bets
    $bets = fetchAll(
        "SELECT * FROM user_bets 
         WHERE $whereClause 
         ORDER BY created_at DESC 
         LIMIT $limit OFFSET $offset",
        $params
    );
    
    // Calculate summary statistics
    $summary = fetchRow(
        "SELECT 
            COUNT(*) as total_bets,
            COALESCE(SUM(bet_amount), 0) as total_wagered,
            COALESCE(SUM(CASE WHEN status = 'won' THEN payout_amount ELSE 0 END), 0) as total_winnings,
            COALESCE(SUM(bet_amount) - SUM(CASE WHEN status = 'won' THEN payout_amount ELSE 0 END), 0) as net_loss
         FROM user_bets 
         WHERE $whereClause",
        $params
    );
    
    echo json_encode([
        'success' => true,
        'data' => [
            'bets' => $bets,
            'summary' => $summary,
            'pagination' => [
                'current_page' => $page,
                'total_pages' => ceil($totalCount / $limit),
                'total_count' => $totalCount,
                'per_page' => $limit
            ]
        ]
    ]);
}

function handleActiveBetsRequest($method) {
    global $currentUser;
    
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    $gameType = $_GET['game_type'] ?? 'all';
    
    // Build query conditions
    $conditions = ["user_id = ?", "status IN ('active', 'pending')"];
    $params = [$currentUser['id']];
    
    if ($gameType !== 'all') {
        $conditions[] = "game_type = ?";
        $params[] = $gameType;
    }
    
    $whereClause = implode(' AND ', $conditions);
    
    $activeBets = fetchAll(
        "SELECT * FROM user_bets 
         WHERE $whereClause 
         ORDER BY created_at DESC",
        $params
    );
    
    // Add real-time data for active bets
    foreach ($activeBets as &$bet) {
        if ($bet['game_type'] === 'crash' && $bet['status'] === 'active') {
            // Get current crash multiplier
            $bet['current_multiplier'] = getCurrentCrashMultiplier();
            $bet['potential_payout'] = $bet['bet_amount'] * $bet['current_multiplier'];
        } elseif ($bet['game_type'] === 'sports' && $bet['status'] === 'active') {
            // Get current match status
            $bet['match_status'] = getCurrentMatchStatus($bet['match_id'] ?? 0);
        }
    }
    
    echo json_encode([
        'success' => true,
        'data' => [
            'active_bets' => $activeBets,
            'total_active' => count($activeBets),
            'total_stake' => array_sum(array_column($activeBets, 'bet_amount'))
        ]
    ]);
}

function handleCancelBetRequest($method) {
    global $currentUser;
    
    if ($method !== 'POST') {
        throw new Exception('Method not allowed');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    $betId = intval($input['bet_id'] ?? 0);
    
    if (!$betId) {
        throw new Exception('Bet ID is required');
    }
    
    // Get bet details
    $bet = fetchRow(
        "SELECT * FROM user_bets WHERE id = ? AND user_id = ?",
        [$betId, $currentUser['id']]
    );
    
    if (!$bet) {
        throw new Exception('Bet not found');
    }
    
    if ($bet['status'] !== 'pending') {
        throw new Exception('Only pending bets can be cancelled');
    }
    
    // Check if bet is still cancellable (e.g., sports bet before match starts)
    if (!isBetCancellable($bet)) {
        throw new Exception('This bet can no longer be cancelled');
    }
    
    // Cancel the bet
    execute(
        "UPDATE user_bets SET status = 'cancelled', updated_at = NOW() WHERE id = ?",
        [$betId]
    );
    
    // Refund the bet amount
    execute("UPDATE users SET balance = balance + ? WHERE id = ?", [$bet['bet_amount'], $currentUser['id']]);
    
    // Create refund transaction
    execute(
        "INSERT INTO transactions (user_id, type, amount, status, description, created_at) 
         VALUES (?, 'refund', ?, 'completed', ?, NOW())",
        [$currentUser['id'], $bet['bet_amount'], "Bet cancellation refund for bet #$betId"]
    );
    
    echo json_encode([
        'success' => true,
        'data' => [
            'bet_id' => $betId,
            'refund_amount' => $bet['bet_amount'],
            'message' => 'Bet cancelled successfully'
        ]
    ]);
}

function handleCashoutRequest($method) {
    global $currentUser;
    
    if ($method !== 'POST') {
        throw new Exception('Method not allowed');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    $betId = intval($input['bet_id'] ?? 0);
    $cashoutMultiplier = floatval($input['multiplier'] ?? 0);
    
    if (!$betId) {
        throw new Exception('Bet ID is required');
    }
    
    // Get bet details
    $bet = fetchRow(
        "SELECT * FROM user_bets WHERE id = ? AND user_id = ?",
        [$betId, $currentUser['id']]
    );
    
    if (!$bet) {
        throw new Exception('Bet not found');
    }
    
    if ($bet['status'] !== 'active') {
        throw new Exception('Only active bets can be cashed out');
    }
    
    if ($bet['game_type'] !== 'crash') {
        throw new Exception('Cashout is only available for crash games');
    }
    
    // Validate cashout multiplier
    $currentMultiplier = getCurrentCrashMultiplier();
    
    if ($cashoutMultiplier > $currentMultiplier) {
        throw new Exception('Invalid cashout multiplier');
    }
    
    // Calculate payout
    $payoutAmount = $bet['bet_amount'] * $cashoutMultiplier;
    $maxWin = getSetting('max_win_amount', 100000.00);
    
    if ($payoutAmount > $maxWin) {
        $payoutAmount = $maxWin;
    }
    
    // Process cashout
    execute(
        "UPDATE user_bets SET 
            status = 'won', 
            payout_amount = ?, 
            cashout_multiplier = ?, 
            updated_at = NOW() 
         WHERE id = ?",
        [$payoutAmount, $cashoutMultiplier, $betId]
    );
    
    // Add winnings to user balance
    execute("UPDATE users SET balance = balance + ? WHERE id = ?", [$payoutAmount, $currentUser['id']]);
    
    // Create winning transaction
    execute(
        "INSERT INTO transactions (user_id, type, amount, status, description, created_at) 
         VALUES (?, 'win', ?, 'completed', ?, NOW())",
        [$currentUser['id'], $payoutAmount, "Crash game cashout at {$cashoutMultiplier}x"]
    );
    
    echo json_encode([
        'success' => true,
        'data' => [
            'bet_id' => $betId,
            'cashout_multiplier' => $cashoutMultiplier,
            'payout_amount' => $payoutAmount,
            'profit' => $payoutAmount - $bet['bet_amount'],
            'message' => 'Cashout successful'
        ]
    ]);
}

function handleBetStatisticsRequest($method) {
    global $currentUser;
    
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    $period = $_GET['period'] ?? '30days';
    
    // Calculate date range
    switch ($period) {
        case 'today':
            $dateFrom = date('Y-m-d');
            $dateTo = date('Y-m-d');
            break;
        case '7days':
            $dateFrom = date('Y-m-d', strtotime('-7 days'));
            $dateTo = date('Y-m-d');
            break;
        case '30days':
            $dateFrom = date('Y-m-d', strtotime('-30 days'));
            $dateTo = date('Y-m-d');
            break;
        case 'all':
            $dateFrom = '2020-01-01';
            $dateTo = date('Y-m-d');
            break;
        default:
            $dateFrom = date('Y-m-d', strtotime('-30 days'));
            $dateTo = date('Y-m-d');
    }
    
    // Overall statistics
    $overallStats = fetchRow(
        "SELECT 
            COUNT(*) as total_bets,
            COALESCE(SUM(bet_amount), 0) as total_wagered,
            COALESCE(SUM(CASE WHEN status = 'won' THEN payout_amount ELSE 0 END), 0) as total_winnings,
            COALESCE(AVG(bet_amount), 0) as avg_bet_amount,
            COUNT(CASE WHEN status = 'won' THEN 1 END) as wins,
            COUNT(CASE WHEN status = 'lost' THEN 1 END) as losses
         FROM user_bets 
         WHERE user_id = ? AND DATE(created_at) BETWEEN ? AND ?",
        [$currentUser['id'], $dateFrom, $dateTo]
    );
    
    // Calculate additional metrics
    $overallStats['net_profit'] = $overallStats['total_winnings'] - $overallStats['total_wagered'];
    $overallStats['win_rate'] = $overallStats['total_bets'] > 0 ? 
        ($overallStats['wins'] / $overallStats['total_bets']) * 100 : 0;
    $overallStats['rtp'] = $overallStats['total_wagered'] > 0 ? 
        ($overallStats['total_winnings'] / $overallStats['total_wagered']) * 100 : 0;
    
    // Game type breakdown
    $gameTypeStats = fetchAll(
        "SELECT 
            game_type,
            COUNT(*) as bet_count,
            COALESCE(SUM(bet_amount), 0) as total_wagered,
            COALESCE(SUM(CASE WHEN status = 'won' THEN payout_amount ELSE 0 END), 0) as total_winnings,
            COUNT(CASE WHEN status = 'won' THEN 1 END) as wins
         FROM user_bets 
         WHERE user_id = ? AND DATE(created_at) BETWEEN ? AND ?
         GROUP BY game_type
         ORDER BY total_wagered DESC",
        [$currentUser['id'], $dateFrom, $dateTo]
    );
    
    // Daily activity
    $dailyActivity = fetchAll(
        "SELECT 
            DATE(created_at) as date,
            COUNT(*) as bet_count,
            COALESCE(SUM(bet_amount), 0) as total_wagered,
            COALESCE(SUM(CASE WHEN status = 'won' THEN payout_amount ELSE 0 END), 0) as total_winnings
         FROM user_bets 
         WHERE user_id = ? AND DATE(created_at) BETWEEN ? AND ?
         GROUP BY DATE(created_at)
         ORDER BY date ASC",
        [$currentUser['id'], $dateFrom, $dateTo]
    );
    
    echo json_encode([
        'success' => true,
        'data' => [
            'period' => $period,
            'date_range' => [
                'from' => $dateFrom,
                'to' => $dateTo
            ],
            'overall' => $overallStats,
            'by_game_type' => $gameTypeStats,
            'daily_activity' => $dailyActivity
        ]
    ]);
}

// Helper functions
function placeCrashBet($userId, $betAmount, $betData) {
    $autoCashout = $betData['auto_cashout'] ?? null;
    
    return execute(
        "INSERT INTO user_bets (user_id, game_type, bet_amount, auto_cashout, status, created_at) 
         VALUES (?, 'crash', ?, ?, 'active', NOW())",
        [$userId, $betAmount, $autoCashout]
    );
}

function placeSportsBet($userId, $betAmount, $betData) {
    $matchId = $betData['match_id'] ?? 0;
    $betType = $betData['bet_type'] ?? '';
    $odds = $betData['odds'] ?? 0;
    
    if (!$matchId || !$betType || !$odds) {
        throw new Exception('Invalid sports bet data');
    }
    
    $potentialPayout = $betAmount * $odds;
    
    return execute(
        "INSERT INTO user_bets (user_id, game_type, bet_amount, match_id, bet_type, odds, potential_payout, status, created_at) 
         VALUES (?, 'sports', ?, ?, ?, ?, ?, 'pending', NOW())",
        [$userId, $betAmount, $matchId, $betType, $odds, $potentialPayout]
    );
}

function placeCasinoBet($userId, $betAmount, $betData) {
    $gameId = $betData['game_id'] ?? '';
    $betType = $betData['bet_type'] ?? '';
    
    if (!$gameId || !$betType) {
        throw new Exception('Invalid casino bet data');
    }
    
    return execute(
        "INSERT INTO user_bets (user_id, game_type, bet_amount, game_id, bet_type, status, created_at) 
         VALUES (?, 'casino', ?, ?, ?, 'pending', NOW())",
        [$userId, $betAmount, $gameId, $betType]
    );
}

function getCurrentCrashMultiplier() {
    // Mock function - in production, this would get the real-time multiplier
    return 1.5 + (rand(0, 500) / 100);
}

function getCurrentMatchStatus($matchId) {
    // Mock function - in production, this would get real match data
    return [
        'status' => 'live',
        'minute' => 45,
        'score' => '1-0'
    ];
}

function isBetCancellable($bet) {
    if ($bet['game_type'] === 'sports') {
        // Sports bets can be cancelled up to 5 minutes before match start
        // This is a simplified check
        return true;
    }
    
    return false;
}

function getSetting($key, $default = null) {
    $setting = fetchRow("SELECT setting_value FROM site_settings WHERE setting_key = ?", [$key]);
    return $setting ? $setting['setting_value'] : $default;
}
?>
