<?php
// api/support.php - Support API
// API پشتیبانی

require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

$method = $_SERVER['REQUEST_METHOD'];
$endpoint = $_GET['endpoint'] ?? '';

try {
    switch ($endpoint) {
        case 'tickets':
            handleTicketsRequest($method);
            break;
            
        case 'create':
            handleCreateTicketRequest($method);
            break;
            
        case 'reply':
            handleReplyRequest($method);
            break;
            
        case 'close':
            handleCloseTicketRequest($method);
            break;
            
        case 'categories':
            handleCategoriesRequest($method);
            break;
            
        case 'faq':
            handleFaqRequest($method);
            break;
            
        case 'chat':
            handleChatRequest($method);
            break;
            
        default:
            throw new Exception('Invalid endpoint');
    }
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

function handleTicketsRequest($method) {
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    // Check if user is authenticated
    $isAuthenticated = isset($_SESSION['user_id']);
    $userId = $isAuthenticated ? $_SESSION['user_id'] : null;
    
    if (!$isAuthenticated) {
        throw new Exception('Authentication required');
    }
    
    $page = intval($_GET['page'] ?? 1);
    $limit = intval($_GET['limit'] ?? 10);
    $status = $_GET['status'] ?? 'all';
    $category = $_GET['category'] ?? 'all';
    
    $offset = ($page - 1) * $limit;
    
    // Build query conditions
    $conditions = ["user_id = ?"];
    $params = [$userId];
    
    if ($status !== 'all') {
        $conditions[] = "status = ?";
        $params[] = $status;
    }
    
    if ($category !== 'all') {
        $conditions[] = "category = ?";
        $params[] = $category;
    }
    
    $whereClause = implode(' AND ', $conditions);
    
    // Get total count
    $totalCount = fetchRow(
        "SELECT COUNT(*) as count FROM support_tickets WHERE $whereClause",
        $params
    )['count'];
    
    // Get tickets
    $tickets = fetchAll(
        "SELECT st.*, u.username, 
                (SELECT COUNT(*) FROM support_messages sm WHERE sm.ticket_id = st.id) as message_count,
                (SELECT created_at FROM support_messages sm WHERE sm.ticket_id = st.id ORDER BY created_at DESC LIMIT 1) as last_message_at
         FROM support_tickets st
         LEFT JOIN users u ON st.user_id = u.id
         WHERE $whereClause 
         ORDER BY st.updated_at DESC 
         LIMIT $limit OFFSET $offset",
        $params
    );
    
    echo json_encode([
        'success' => true,
        'data' => [
            'tickets' => $tickets,
            'pagination' => [
                'current_page' => $page,
                'total_pages' => ceil($totalCount / $limit),
                'total_count' => $totalCount,
                'per_page' => $limit
            ]
        ]
    ]);
}

function handleCreateTicketRequest($method) {
    if ($method !== 'POST') {
        throw new Exception('Method not allowed');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    $subject = trim($input['subject'] ?? '');
    $message = trim($input['message'] ?? '');
    $category = $input['category'] ?? 'general';
    $priority = $input['priority'] ?? 'medium';
    $email = trim($input['email'] ?? '');
    $name = trim($input['name'] ?? '');
    
    if (!$subject || !$message) {
        throw new Exception('Subject and message are required');
    }
    
    if (strlen($subject) < 5) {
        throw new Exception('Subject must be at least 5 characters long');
    }
    
    if (strlen($message) < 10) {
        throw new Exception('Message must be at least 10 characters long');
    }
    
    // Check if user is authenticated
    $isAuthenticated = isset($_SESSION['user_id']);
    $userId = $isAuthenticated ? $_SESSION['user_id'] : null;
    
    // For guest users, email is required
    if (!$isAuthenticated && !$email) {
        throw new Exception('Email is required for guest tickets');
    }
    
    if (!$isAuthenticated && !$name) {
        throw new Exception('Name is required for guest tickets');
    }
    
    // Generate ticket number
    $ticketNumber = 'TKT-' . date('Ymd') . '-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
    
    // Create ticket
    $ticketId = execute(
        "INSERT INTO support_tickets (ticket_number, user_id, subject, category, priority, status, guest_email, guest_name, created_at, updated_at) 
         VALUES (?, ?, ?, ?, ?, 'open', ?, ?, NOW(), NOW())",
        [$ticketNumber, $userId, $subject, $category, $priority, $email, $name]
    );
    
    // Add initial message
    execute(
        "INSERT INTO support_messages (ticket_id, user_id, message, is_staff, created_at) 
         VALUES (?, ?, ?, 0, NOW())",
        [$ticketId, $userId, $message]
    );
    
    // Send notification email (mock implementation)
    if ($email) {
        sendTicketNotification($email, $ticketNumber, $subject);
    }
    
    echo json_encode([
        'success' => true,
        'data' => [
            'ticket_id' => $ticketId,
            'ticket_number' => $ticketNumber,
            'subject' => $subject,
            'status' => 'open',
            'created_at' => date('Y-m-d H:i:s'),
            'message' => 'Support ticket created successfully'
        ]
    ]);
}

function handleReplyRequest($method) {
    if ($method !== 'POST') {
        throw new Exception('Method not allowed');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    $ticketId = intval($input['ticket_id'] ?? 0);
    $message = trim($input['message'] ?? '');
    
    if (!$ticketId) {
        throw new Exception('Ticket ID is required');
    }
    
    if (!$message) {
        throw new Exception('Message is required');
    }
    
    if (strlen($message) < 5) {
        throw new Exception('Message must be at least 5 characters long');
    }
    
    // Check if user is authenticated
    $isAuthenticated = isset($_SESSION['user_id']);
    $userId = $isAuthenticated ? $_SESSION['user_id'] : null;
    
    if (!$isAuthenticated) {
        throw new Exception('Authentication required');
    }
    
    // Verify ticket ownership
    $ticket = fetchRow(
        "SELECT * FROM support_tickets WHERE id = ? AND user_id = ?",
        [$ticketId, $userId]
    );
    
    if (!$ticket) {
        throw new Exception('Ticket not found or access denied');
    }
    
    if ($ticket['status'] === 'closed') {
        throw new Exception('Cannot reply to closed ticket');
    }
    
    // Add reply message
    $messageId = execute(
        "INSERT INTO support_messages (ticket_id, user_id, message, is_staff, created_at) 
         VALUES (?, ?, ?, 0, NOW())",
        [$ticketId, $userId, $message]
    );
    
    // Update ticket status and timestamp
    execute(
        "UPDATE support_tickets SET status = 'awaiting_staff', updated_at = NOW() WHERE id = ?",
        [$ticketId]
    );
    
    echo json_encode([
        'success' => true,
        'data' => [
            'message_id' => $messageId,
            'ticket_id' => $ticketId,
            'message' => $message,
            'created_at' => date('Y-m-d H:i:s'),
            'status' => 'Reply added successfully'
        ]
    ]);
}

function handleCloseTicketRequest($method) {
    if ($method !== 'POST') {
        throw new Exception('Method not allowed');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    $ticketId = intval($input['ticket_id'] ?? 0);
    $reason = trim($input['reason'] ?? '');
    
    if (!$ticketId) {
        throw new Exception('Ticket ID is required');
    }
    
    // Check if user is authenticated
    $isAuthenticated = isset($_SESSION['user_id']);
    $userId = $isAuthenticated ? $_SESSION['user_id'] : null;
    
    if (!$isAuthenticated) {
        throw new Exception('Authentication required');
    }
    
    // Verify ticket ownership
    $ticket = fetchRow(
        "SELECT * FROM support_tickets WHERE id = ? AND user_id = ?",
        [$ticketId, $userId]
    );
    
    if (!$ticket) {
        throw new Exception('Ticket not found or access denied');
    }
    
    if ($ticket['status'] === 'closed') {
        throw new Exception('Ticket is already closed');
    }
    
    // Close ticket
    execute(
        "UPDATE support_tickets SET status = 'closed', closed_at = NOW(), updated_at = NOW() WHERE id = ?",
        [$ticketId]
    );
    
    // Add closure message if reason provided
    if ($reason) {
        execute(
            "INSERT INTO support_messages (ticket_id, user_id, message, is_staff, created_at) 
             VALUES (?, ?, ?, 0, NOW())",
            [$ticketId, $userId, "Ticket closed by user. Reason: $reason"]
        );
    }
    
    echo json_encode([
        'success' => true,
        'data' => [
            'ticket_id' => $ticketId,
            'status' => 'closed',
            'closed_at' => date('Y-m-d H:i:s'),
            'message' => 'Ticket closed successfully'
        ]
    ]);
}

function handleCategoriesRequest($method) {
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    // Support categories
    $categories = [
        [
            'id' => 'general',
            'name' => 'General Inquiry',
            'description' => 'General questions and inquiries',
            'icon' => 'fas fa-question-circle'
        ],
        [
            'id' => 'account',
            'name' => 'Account Issues',
            'description' => 'Login, registration, and account problems',
            'icon' => 'fas fa-user-circle'
        ],
        [
            'id' => 'payment',
            'name' => 'Payment & Withdrawal',
            'description' => 'Deposit and withdrawal related issues',
            'icon' => 'fas fa-credit-card'
        ],
        [
            'id' => 'betting',
            'name' => 'Betting Issues',
            'description' => 'Problems with placing bets or game results',
            'icon' => 'fas fa-dice'
        ],
        [
            'id' => 'technical',
            'name' => 'Technical Support',
            'description' => 'Website bugs and technical problems',
            'icon' => 'fas fa-cog'
        ],
        [
            'id' => 'complaint',
            'name' => 'Complaint',
            'description' => 'File a complaint about our services',
            'icon' => 'fas fa-exclamation-triangle'
        ]
    ];
    
    echo json_encode([
        'success' => true,
        'data' => $categories
    ]);
}

function handleFaqRequest($method) {
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    $category = $_GET['category'] ?? 'all';
    
    // Mock FAQ data
    $faqs = [
        [
            'id' => 1,
            'category' => 'general',
            'question' => 'How do I create an account?',
            'answer' => 'To create an account, click on the "Register" button in the top right corner of the website and fill out the required information.',
            'order' => 1
        ],
        [
            'id' => 2,
            'category' => 'general',
            'question' => 'Is Crashify legal and licensed?',
            'answer' => 'Yes, Crashify operates under proper licensing and follows all applicable regulations.',
            'order' => 2
        ],
        [
            'id' => 3,
            'category' => 'payment',
            'question' => 'What payment methods do you accept?',
            'answer' => 'We accept credit cards, bank transfers, and various cryptocurrencies including Bitcoin and Ethereum.',
            'order' => 1
        ],
        [
            'id' => 4,
            'category' => 'payment',
            'question' => 'How long do withdrawals take?',
            'answer' => 'Withdrawal processing times vary by method: Crypto (10-60 minutes), Bank transfer (1-3 business days).',
            'order' => 2
        ],
        [
            'id' => 5,
            'category' => 'betting',
            'question' => 'What is the minimum bet amount?',
            'answer' => 'The minimum bet amount is $1.00 for most games.',
            'order' => 1
        ],
        [
            'id' => 6,
            'category' => 'betting',
            'question' => 'How does the crash game work?',
            'answer' => 'In the crash game, you place a bet and watch the multiplier increase. Cash out before it crashes to win!',
            'order' => 2
        ],
        [
            'id' => 7,
            'category' => 'account',
            'question' => 'How do I verify my account?',
            'answer' => 'Account verification requires uploading a government-issued ID and proof of address.',
            'order' => 1
        ],
        [
            'id' => 8,
            'category' => 'account',
            'question' => 'I forgot my password, what should I do?',
            'answer' => 'Click on "Forgot Password" on the login page and follow the instructions sent to your email.',
            'order' => 2
        ]
    ];
    
    // Filter by category if specified
    if ($category !== 'all') {
        $faqs = array_filter($faqs, function($faq) use ($category) {
            return $faq['category'] === $category;
        });
        $faqs = array_values($faqs);
    }
    
    // Group by category
    $groupedFaqs = [];
    foreach ($faqs as $faq) {
        $groupedFaqs[$faq['category']][] = $faq;
    }
    
    echo json_encode([
        'success' => true,
        'data' => [
            'faqs' => $faqs,
            'grouped' => $groupedFaqs,
            'total' => count($faqs)
        ]
    ]);
}

function handleChatRequest($method) {
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    // Live chat availability
    $chatStatus = [
        'available' => true,
        'online_agents' => 3,
        'average_wait_time' => '2 minutes',
        'hours' => '24/7',
        'languages' => ['English', 'Persian']
    ];
    
    echo json_encode([
        'success' => true,
        'data' => $chatStatus
    ]);
}

function sendTicketNotification($email, $ticketNumber, $subject) {
    // Mock email sending - in production, integrate with email service
    // This would send an email notification about the new ticket
    return true;
}
?>