// مدیریت سیستم چندزبانه
class LanguageManager {
    constructor() {
        this.currentLanguage = localStorage.getItem('language') || 'en';
        this.translations = {};
        this.init();
    }
    
    // مقداردهی اولیه
    init() {
        this.loadTranslations(this.currentLanguage);
        this.applyTranslations();
        this.setupLanguageSelector();
    }
    
    // بارگذاری ترجمه‌ها
    loadTranslations(language) {
        // در واقعیت این اطلاعات از سرور بارگذاری می‌شوند
        // اینجا برای نمونه از یک شیء ثابت استفاده می‌کنیم
        const sampleTranslations = {
            'en': {
                'welcome': 'Welcome',
                'login': 'Login',
                'register': 'Register',
                'balance': 'Balance',
                'bet_amount': 'Bet Amount',
                'place_bet': 'Place Bet',
                'cashout': 'Cashout'
            },
            'fa': {
                'welcome': 'خوش آمدید',
                'login': 'ورود',
                'register': 'ثبت نام',
                'balance': 'موجودی',
                'bet_amount': 'مبلغ شرط',
                'place_bet': 'ثبت شرط',
                'cashout': 'نقد کردن'
            },
            'ar': {
                'welcome': 'أهلا بك',
                'login': 'تسجيل الدخول',
                'register': 'تسجيل',
                'balance': 'الرصيد',
                'bet_amount': 'مبلغ الرهان',
                'place_bet': 'وضع الرهان',
                'cashout': 'السحب'
            },
            'tr': {
                'welcome': 'Hoş geldiniz',
                'login': 'Giriş yap',
                'register': 'Kayıt ol',
                'balance': 'Bakiye',
                'bet_amount': 'Bahis miktarı',
                'place_bet': 'Bahis oyna',
                'cashout': 'Nakit çıkış'
            },
            'de': {
                'welcome': 'Willkommen',
                'login': 'Anmelden',
                'register': 'Registrieren',
                'balance': 'Guthaben',
                'bet_amount': 'Wetteinsatz',
                'place_bet': 'Wette platzieren',
                'cashout': 'Auszahlen'
            },
            'fr': {
                'welcome': 'Bienvenue',
                'login': 'Connexion',
                'register': 'Inscription',
                'balance': 'Solde',
                'bet_amount': 'Montant du pari',
                'place_bet': 'Placer un pari',
                'cashout': 'Cashout'
            }
        };
        
        this.translations = sampleTranslations[language] || sampleTranslations['en'];
        this.currentLanguage = language;
        localStorage.setItem('language', language);
    }
    
    // اعمال ترجمه‌ها روی عناصر صفحه
    applyTranslations() {
        const elements = document.querySelectorAll('[data-translate]');
        
        elements.forEach(element => {
            const key = element.getAttribute('data-translate');
            if (this.translations[key]) {
                element.textContent = this.translations[key];
            }
        });
        
        // به روز رسانی انتخابگر زبان
        const languageSelector = document.getElementById('languageSelector');
        if (languageSelector) {
            languageSelector.value = this.currentLanguage;
        }
        
        // تغییر جهت صفحه برای زبان‌های RTL
        if (['fa', 'ar'].includes(this.currentLanguage)) {
            document.documentElement.dir = 'rtl';
        } else {
            document.documentElement.dir = 'ltr';
        }
    }
    
    // راه‌اندازی انتخابگر زبان
    setupLanguageSelector() {
        const languageSelector = document.getElementById('languageSelector');
        if (languageSelector) {
            languageSelector.addEventListener('change', (e) => {
                this.changeLanguage(e.target.value);
            });
        }
    }
    
    // تغییر زبان
    changeLanguage(language) {
        this.loadTranslations(language);
        this.applyTranslations();
        
        // ارسال درخواست به سرور برای تغییر زبان
        $.ajax({
            url: 'includes/change_language.php',
            type: 'POST',
            data: { language: language },
            success: function(response) {
                console.log('Language changed successfully');
            },
            error: function() {
                console.error('Error changing language');
            }
        });
    }
    
    // دریافت ترجمه یک کلید خاص
    translate(key) {
        return this.translations[key] || key;
    }
}

// نمونه‌ای از LanguageManager
const languageManager = new LanguageManager();

// تابع برای استفاده در سایر بخش‌های کد
function translate(key) {
    return languageManager.translate(key);
}

// استفاده از ترجمه در جاوااسکریپت
function showTranslatedMessage() {
    alert(translate('welcome'));
}